# Repository Guidelines

## Project Structure & Module Organization
- `app/` holds application logic (HTTP controllers, console, jobs); `config/` defines service defaults.
- `routes/web.php` handles site routes; `routes/api.php` is available for API endpoints.
- `resources/docs` stores authored documentation; `resources/views` Blade layouts/partials; `resources/js` Vite entry points/components; `resources/css` Tailwind entry.
- `public/` is the web root for built assets; `vite.config.js` controls the front-end build.
- `database/migrations` defines schema; `database/seeders` optional fixtures; `storage/` and `logs/` keep runtime/cache output.
- Tests live in `tests/Feature` (HTTP/CLI) and `tests/Unit` (pure logic).

## Build, Test, and Development Commands
- Bootstrap: `composer install && npm install`, then `cp .env.example .env` and `php artisan key:generate`.
- Database: `php artisan migrate` (SQLite at `database/database.sqlite` works for local).
- Full dev loop: `composer dev` runs PHP server, queue listener, log viewer, and Vite with HMR.
- Individual services: `php artisan serve` for backend; `npm run dev` for Vite; `npm run build` for production assets.
- Tests: `composer test` or `php artisan test`; clear cached config before env-sensitive runs with `php artisan config:clear`.

## Coding Style & Naming Conventions
- PHP 8.2+, PSR-12; 4-space indentation for PHP/Blade; 2 spaces for JS/CSS.
- Format with `./vendor/bin/pint` (or `./vendor/bin/pint --test` in CI) and keep imports ordered.
- Classes (controllers/models/jobs) use PascalCase; route names use `kebab-case`; env keys stay `UPPER_SNAKE_CASE`; view/component files use `kebab-case`.

## Testing Guidelines
- Add behavior tests in `tests/Feature` and unit-level cases in `tests/Unit`; files end with `*Test.php`.
- Use `php artisan test --filter ClassName` for focused runs; seed/factory data when touching the DB.
- Cover success, failure, and edge paths (queues, listeners, markdown parsing) for new work.

## Commit & Pull Request Guidelines
- No history yet—prefer short, imperative Conventional Commit prefixes (e.g., `feat: add doc landing page`, `chore: run pint`).
- PRs should describe scope, link issues, call out migrations/env changes, and include screenshots when docs/UI change.
- Note verification steps (commands/tests run) and keep diffs small and reviewable.

## Security & Configuration Tips
- Keep secrets in `.env`; never commit real keys—update `.env.example` when adding required vars.
- Rebuild assets after config/env changes to bust caches; prune/flush queues or logs via Artisan when debugging noisy local state.

## Docs Rebuild Plan (from Markdown to curated views)
- Map sources: list all `resources/docs/**/*.md`, noting sections (API, OG framework, Queue, etc.), and tag each file with type (concept, how-to, reference, checklist).
- Define IA skeleton: propose top-level chapters (Overview, Getting Started, Framework Internals, API, Queue, Security, Ops) with a short 1–2 sentence goal per chapter.
- Curate content per chapter: for each section, pick the relevant MDs (no direct copy), extract key points/outcomes, and mark gaps to rewrite/condense.
- Draft view specs: for every chapter, outline Blade/Livewire view structure (hero, TOC, callouts, examples) and what needs rewriting versus porting.
- Rewrite iteratively: convert curated notes into prose/snippets directly in views (or structured partials), keeping Flux components; keep a checklist of completed chapters.
- Verification loop: after each batch, review for accuracy, link consistency, and add navigation entries; mark outdated MDs for archival.

### Current Markdown Inventory (use only as reference, not rendered)
- Root: Authentication-System-Guide.md, Bootstrap-System-Documentation.md
- API/Auth: README.md, helpers-guide.md
- API/Tickets: README.md, guia-completo-fetcher-tickets.md
- Database: OGDB_forge.md
- Queue: Queue-Retry-Command.md, Queue-Scheduling-System.md
- Queue/QueueMaster: API_REFERENCE.md, ARCHITECTURE.md, CONFIGURATION.md, DEPLOY_CHECKLIST.md, DEVELOPMENT.md, EXAMPLES.md, MONITORING.md, README.md, ROADMAP.md, TROUBLESHOOTING.md, USAGE_GUIDE.md
- Security: Authentication.md
- Memory optimization: memory-optimized-trait-usage.md, saftpt-generators-implementation.md
- Middleware/guard system: README.md, future-implementations.md, guard-advanced-guide.md, middleware-examples.md, router-refactoring.md
- OG framework: README.md, auth-system.md, cache-system.md, cli-system.md, container-system.md, documentation-prompts.md, facades-system.md, filesystem-system.md, guard-system.md, helpers-system.md, notifications-system.md, pagination-system.md, queue-system.md, resource-system.md, service-providers-system.md, technical-appendix-patterns-use-cases.md, traits-system.md, utils-system.md, validation-system.md, web-layer-system.md
