<?php

namespace App\Livewire;

use Livewire\Component;
use Livewire\Attributes\Computed;

class SearchDocs extends Component
{
    public string $query = '';
    public bool $showResults = false;
    public int $selectedIndex = -1;

    protected array $searchIndex = [];

    // Mapping from search-index paths to valid route slugs
    protected array $pathMapping = [
        'og-framework/bootstrap-lifecycle' => 'bootstrap',
        'og-framework/container-system' => 'container',
        'og-framework/service-providers' => 'service-providers',
        'og-framework/authentication-system' => 'auth',
        'og-framework/facades' => 'facades',
        'og-framework/cache-system' => 'cache',
        'og-framework/database-system' => 'database',
        'og-framework/filesystem' => 'filesystem',
        'og-framework/routing-system' => 'routing',
        'og-framework/helpers' => 'helpers',
        'og-framework/resources' => 'resources',
        'og-framework/notifications' => 'notifications',
        'og-framework/validation' => 'validation',
        'og-framework/web-layer-system' => 'web-layer',
        'og-framework/queue-system' => 'queue',
    ];

    public function mount(): void
    {
        $this->loadSearchIndex();
    }

    protected function loadSearchIndex(): void
    {
        $indexPath = public_path('search-index.json');
        
        if (file_exists($indexPath)) {
            $this->searchIndex = json_decode(file_get_contents($indexPath), true) ?? [];
        }
    }

    protected function convertUrl(string $originalUrl): ?string
    {
        // Try to extract the path and map to correct route
        foreach ($this->pathMapping as $searchPath => $routeSlug) {
            if (str_contains($originalUrl, $searchPath)) {
                return route('docs.show', $routeSlug);
            }
        }
        
        // For index/overview pages
        if (str_contains($originalUrl, 'overview') || $originalUrl === '/docs' || $originalUrl === '/docs/') {
            return route('docs.index');
        }
        
        return null;
    }

    #[Computed]
    public function results(): array
    {
        if (strlen($this->query) < 2) {
            return [];
        }

        $this->loadSearchIndex();
        $query = mb_strtolower($this->query);
        $results = [];
        $maxResults = 8;
        $seenUrls = [];

        foreach ($this->searchIndex as $item) {
            $title = mb_strtolower($item['title'] ?? '');
            $content = mb_strtolower($item['content'] ?? '');
            $description = mb_strtolower($item['description'] ?? '');
            $excerpt = mb_strtolower($item['excerpt'] ?? '');

            // Score based on where the match is found
            $score = 0;
            if (str_contains($title, $query)) {
                $score += 100;
            }
            if (str_contains($description, $query)) {
                $score += 50;
            }
            if (str_contains($excerpt, $query)) {
                $score += 30;
            }
            if (str_contains($content, $query)) {
                $score += 10;
            }

            if ($score > 0) {
                $originalUrl = $item['url'] ?? '#';
                $convertedUrl = $this->convertUrl($originalUrl);
                
                // Skip if URL couldn't be converted or already seen
                if (!$convertedUrl || isset($seenUrls[$convertedUrl])) {
                    continue;
                }
                
                $seenUrls[$convertedUrl] = true;

                $results[] = [
                    'title' => $item['title'] ?? '',
                    'url' => $convertedUrl,
                    'description' => $this->extractSnippet($item['excerpt'] ?? $item['content'] ?? '', $query),
                    'section' => $item['section'] ?? null,
                    'score' => $score,
                ];
            }
        }

        // Sort by score descending
        usort($results, fn($a, $b) => $b['score'] <=> $a['score']);

        return array_slice($results, 0, $maxResults);
    }

    protected function extractSnippet(string $content, string $query, int $contextLength = 80): string
    {
        $position = mb_stripos($content, $query);
        
        if ($position === false) {
            return mb_substr($content, 0, $contextLength) . '...';
        }

        $start = max(0, $position - 40);
        $snippet = mb_substr($content, $start, $contextLength * 2);

        if ($start > 0) {
            $snippet = '...' . $snippet;
        }
        if (mb_strlen($content) > $start + ($contextLength * 2)) {
            $snippet .= '...';
        }

        return $snippet;
    }

    public function updatedQuery(): void
    {
        $this->showResults = strlen($this->query) >= 2;
        $this->selectedIndex = -1;
    }

    public function selectNext(): void
    {
        $count = count($this->results);
        if ($count > 0) {
            $this->selectedIndex = min($this->selectedIndex + 1, $count - 1);
        }
    }

    public function selectPrevious(): void
    {
        $this->selectedIndex = max($this->selectedIndex - 1, -1);
    }

    public function goToSelected(): void
    {
        $results = $this->results;
        if ($this->selectedIndex >= 0 && isset($results[$this->selectedIndex])) {
            $this->redirect($results[$this->selectedIndex]['url']);
        }
    }

    public function hideResults(): void
    {
        $this->showResults = false;
    }

    public function render()
    {
        return view('livewire.search-docs');
    }
}
