# 🎯 Guia Completo: Sistema Fetcher de Tickets para Desenvolvedores Júnior

> **Para quem é este guia?** Você acabou de chegar na empresa e precisa entender/trabalhar com o sistema que busca emails e transforma em tickets. Este guia vai te ensinar TUDO do zero!

## 🤔 O que é o Fetcher de Tickets?

**Em palavras simples:** É um sistema que fica "vigiando" caixas de email (como Gmail, Outlook) e quando chega um email novo, ele automaticamente vira um ticket no nosso sistema.

**Exemplo prático:**
1. Cliente envia email para `suporte@empresa.com`
2. O Fetcher vê que chegou email novo
3. Automaticamente cria um ticket no sistema
4. A equipe de suporte vê o ticket e responde

**Por que isso é útil?** Sem isso, alguém teria que ficar copiando emails manualmente para o sistema de tickets. Com o Fetcher, tudo é automático!

## 🗂️ Onde estão os arquivos importantes?

```
📁 Projeto
├── 📄 api/_controllers/Tickets.php          ← Endpoint que ativa o fetcher
├── 📄 _files/ticketsfetch.class.php         ← Classe principal do fetcher
├── 📄 _files/tickets.class.php              ← Sistema de tickets
└── 🗃️ Base de dados: tabela `tik_email`     ← Configurações de email
```

## 🏗️ Como funciona? (Fluxo simples)

```
📧 Email chega → 🔍 Fetcher detecta → 📝 Cria ticket → ✅ Pronto!
```

**Fluxo detalhado:**
1. **Timer/Cron** chama o endpoint `/api/tickets/fetcher`
2. **Sistema carrega** todas as contas de email configuradas
3. **Para cada conta**, conecta ao servidor (Gmail/Outlook/etc)
4. **Busca emails novos** não lidos
5. **Para cada email novo**:
   - Analisa se é spam/bounce → Se for, ignora
   - Verifica se é resposta a ticket existente → Se for, adiciona ao ticket
   - Se não, cria novo ticket
   - Processa anexos (se houver)
   - Move email para pasta "processados" (opcional)

## 🛠️ Configuração: Como adicionar uma nova conta de email?

### Passo 1: Inserir na base de dados

```sql
-- Exemplo: Adicionar suporte@empresa.com
INSERT INTO tik_email (
    email,           -- Email que vamos monitorizar
    name,            -- Nome descritivo
    in_user,         -- Username para login
    in_host,         -- Servidor IMAP
    in_port,         -- Porta
    in_proto,        -- imap ou pop3
    in_ssl,          -- tls, ssl ou none
    in_inbox,        -- Pasta a monitorizar (geralmente INBOX)
    in_delete,       -- Apagar emails após processar? 0=Não, 1=Sim
    active           -- Conta ativa? 1=Sim, 0=Não
) VALUES (
    'suporte@empresa.com',
    'Suporte da Empresa',
    'suporte@empresa.com',
    'imap.empresa.com',
    993,
    'imap',
    'tls',
    'INBOX',
    0,
    1
);
```

### Passo 2: Testar a configuração

```bash
# Testa apenas esta conta (substitua 123 pelo ID inserido)
curl "https://seudominio.com/api/tickets/fetcher/123"
```

### Passo 3: Verificar logs

```bash
# Ver se funcionou (logs ficam em...)
tail -f /caminho/dos/logs/tickets.log
```

## 🔧 Como usar o sistema? (Para desenvolvedores)

### Opção 1: Via API (mais comum)

```bash
# Processa TODAS as contas configuradas
GET /api/tickets/fetcher

# Processa apenas a conta com ID 123
GET /api/tickets/fetcher/123
```

### Opção 2: Via PHP direto

```php
// Processa todas as contas
TicketsFetch::run();

// Processa apenas a conta ID 123
TicketsFetch::run(123);
```

### Opção 3: Via linha de comando (para crons)

```bash
# Chama via curl (mais comum)
curl -X GET "https://seudominio.com/api/tickets/fetcher"

# Ou via PHP CLI (se configurado)
php -f /caminho/fetcher_script.php
```

## 📊 Respostas da API: O que significam?

```json
// ✅ Sucesso - Tudo correu bem
{
    "code": 1000,
    "message": "Success"
}

// ❌ Erro do servidor - Algo falhou (conexão, parsing, etc)
{
    "code": 500,
    "message": "Unable to fetch tickets at the moment. Please try again later."
}

// 🚫 Sem permissão - User não tem acesso a tickets
{
    "code": 2011,
    "message": "Access denied"
}
```

## 🐛 Debugging: Como saber se está a funcionar?

### 1. Verificar logs

```bash
# Ver logs em tempo real
tail -f /var/log/tickets.log

# Procurar por linhas como:
# [Fetcher] Starting email fetch for ID: 123
# [Fetcher] Found 5 new messages
# [Fetcher] Created ticket #1234 from email
```

### 2. Verificar base de dados

```sql
-- Ver contas configuradas
SELECT id, email, name, in_errors, active FROM tik_email;

-- Ver tickets criados recentemente
SELECT id, subject, date_reg, source FROM tik_tickets 
WHERE source = 'email' 
ORDER BY date_reg DESC 
LIMIT 10;
```

### 3. Testar manualmente

```bash
# Testa uma conta específica
curl -v "https://seudominio.com/api/tickets/fetcher/123"
```

## ❗ Problemas comuns e soluções

### Problema: "Connection timeout" ou "Can't connect"

**O que significa:** Não consegue conectar ao servidor de email

**Possíveis causas:**
- Host/porta errados
- Firewall a bloquear
- Credenciais erradas

**Como resolver:**
```bash
# 1. Testa conexão manual
telnet imap.gmail.com 993

# 2. Verifica configurações na base de dados
SELECT in_host, in_port, in_proto, in_ssl FROM tik_email WHERE id = 123;

# 3. Verifica se a password está correta
```

### Problema: "Authentication failed"

**O que significa:** Password/username errados

**Como resolver:**
1. **Gmail:** Precisa de "App Password", não a password normal
2. **Outlook:** Pode precisar de OAuth, não password
3. **Outros:** Verificar se 2FA está ativo

### Problema: "No new messages" mas há emails novos

**Possíveis causas:**
- Emails já foram marcados como lidos
- Data de sincronização muito antiga
- Pasta errada (não é INBOX)

**Como resolver:**
```sql
-- Reset data de sincronização
UPDATE tik_email SET in_start_sync_date = NULL WHERE id = 123;

-- Verifica pasta configurada
SELECT in_inbox FROM tik_email WHERE id = 123;
```

### Problema: Emails duplicados virando tickets

**Causa:** Message-ID não está a ser capturado/verificado

**Como resolver:**
```sql
-- Ver se há duplicados
SELECT message_id, COUNT(*) FROM tik_mail_headers 
GROUP BY message_id 
HAVING COUNT(*) > 1;
```

## ⚙️ Configurações importantes no sistema

### Limitar quantos emails processar por vez

```php
// No ficheiro de configuração ou via admin
$a->setParam("TIK_FETCH_COUNT", 20); // Máximo 20 emails por execução
```

### Não arquivar emails (deixar na inbox)

```php
$a->setParam("TIK_FETCH_NO_ARCHIVE", 'T'); // T = não arquivar, F = arquivar
```

### Timeout de conexões IMAP

```php
if (function_exists('imap_timeout')) {
    imap_timeout(1, 30); // 30 segundos timeout
}
```

## 🧪 Como testar alterações?

### 1. Ambiente de teste

```bash
# Cria conta de teste
INSERT INTO tik_email (...) VALUES (...);

# Testa só esta conta
curl "https://test.seudominio.com/api/tickets/fetcher/999"
```

### 2. Logs detalhados

```php
// Ativa modo debug
define('DEBUG', true);

// Agora os logs ficam mais detalhados
tail -f /var/log/tickets.log
```

### 3. Testes unitários (se existirem)

```bash
# Roda testes específicos do fetcher
./vendor/bin/pest --filter TicketsControllerTest
```

## 🚀 Implementar uma nova funcionalidade

### Exemplo: Adicionar filtro de assunto

**1. Modificar base de dados:**
```sql
ALTER TABLE tik_email ADD COLUMN subject_filter VARCHAR(255) DEFAULT NULL;
```

**2. Modificar lógica de processamento:**
```php
// Em ticketsfetch.class.php, no método fetchMessage()
protected function fetchMessage($mail_i): bool {
    // ... código existente ...
    
    $subject = $m_analyze['Subject'];
    
    // NOVA FUNCIONALIDADE: Filtro de assunto
    if (!empty($this->ht['subject_filter'])) {
        if (strpos(strtolower($subject), strtolower($this->ht['subject_filter'])) === false) {
            Funcs::log("tickets", "[Fetcher] Subject filtered out: $subject");
            return true; // Ignora este email
        }
    }
    
    // ... resto do código ...
}
```

**3. Testar:**
```sql
-- Adiciona filtro à conta
UPDATE tik_email SET subject_filter = 'URGENT' WHERE id = 123;

-- Agora só emails com "URGENT" no assunto serão processados
```

## 📝 Checklist para novos desenvolvedores

### Antes de mexer no código:

- [ ] Li esta documentação toda
- [ ] Entendo o fluxo básico (email → ticket)
- [ ] Sei onde estão os arquivos principais
- [ ] Consigo executar manualmente: `curl /api/tickets/fetcher`
- [ ] Sei ver os logs: `tail -f /var/log/tickets.log`
- [ ] Conheço a estrutura da tabela `tik_email`

### Antes de fazer alterações:

- [ ] Fiz backup da base de dados
- [ ] Testei em ambiente de desenvolvimento
- [ ] Criei conta de email de teste
- [ ] Documentei as mudanças

### Antes de fazer deploy:

- [ ] Testei com emails reais
- [ ] Verifiquei que não quebra funcionalidades existentes
- [ ] Atualizei esta documentação (se necessário)
- [ ] Informei a equipa das mudanças

## 🔍 Estrutura de ficheiros detalhada

### ticketsfetch.class.php (o "cérebro")

```php
class TicketsFetch {
    // MÉTODOS PRINCIPAIS que vais usar:
    
    public static function run($id = null)     // Ponto de entrada principal
    public function fetchEmails()             // Busca emails de uma conta
    protected function fetchMessage($mail)    // Processa um email específico
    
    // MÉTODOS AUXILIARES:
    
    public function getMailboxes()           // Lista pastas do email
    protected function checkMailbox()       // Verifica se pasta existe
    protected function getArchiveFolder()   // Pasta para emails processados
    
    // CONFIGURAÇÕES:
    
    public function getHost()               // imap.gmail.com
    public function getPort()               // 993
    public function getProtocol()           // imap/pop3
    public function getUsername()           // email@domain.com
}
```

### Tickets.php (API Controller)

```php
// O endpoint que chamas via URL
class Controller extends RestController {
    protected function select() {
        switch ($this->request->getRequestURL(1)) {
            case 'fetcher':  // ← ESTA É A PARTE QUE INTERESSA!
                $tmp = $this->request->getRequestURL(2); // ID da conta (opcional)
                
                try {
                    if (!empty($tmp) && is_numeric($tmp)) {
                        TicketsFetch::run($tmp); // Conta específica
                    } else {
                        TicketsFetch::run();     // Todas as contas
                    }
                    RestUtils::sendResponse(1000); // Sucesso
                } catch (Exception $exception) {
                    RestUtils::sendResponse(500, [
                        'message' => 'Unable to fetch tickets at the moment. Please try again later.'
                    ]);
                }
                break;
        }
    }
}
```

## 🎨 Fluxograma visual do processo

```
📞 CRON/Timer (cada X minutos)
         ↓
🌐 Chama: GET /api/tickets/fetcher
         ↓
🔐 Verifica permissões (user tem acesso a 'tickets'?)
         ↓ SIM
📋 Carrega contas ativas da base de dados (tik_email)
         ↓
🔄 Para cada conta:
   ├─ 📡 Conecta ao servidor (Gmail/Outlook/etc)
   ├─ 📥 Busca emails não lidos
   ├─ 🔄 Para cada email:
   │   ├─ 🔍 É spam/bounce? → SIM → ❌ Ignora
   │   ├─ 🔍 É resposta? → SIM → ➕ Adiciona ao ticket existente
   │   ├─ 🔍 É email novo? → SIM → 🆕 Cria novo ticket
   │   ├─ 📎 Processa anexos
   │   └─ 📁 Move para pasta "processados"
   └─ ✅ Próxima conta
         ↓
✅ Resposta: {"code": 1000} (sucesso) ou {"code": 500} (erro)
```

## 💡 Dicas de produtividade

### 1. Aliases úteis no terminal

```bash
# Adiciona ao teu .bashrc/.zshrc
alias fetch-test="curl https://seudominio.com/api/tickets/fetcher"
alias fetch-logs="tail -f /var/log/tickets.log"
alias fetch-db="mysql -u user -p database -e 'SELECT * FROM tik_email;'"
```

### 2. Snippets de código VSCode

```json
{
  "Fetch Test": {
    "prefix": "fetchtest",
    "body": [
      "// Testa fetcher para conta específica",
      "TicketsFetch::run($1);"
    ]
  }
}
```

### 3. Queries SQL úteis

```sql
-- Ver estatísticas de fetching
SELECT 
    email,
    name,
    in_errors as errors,
    active,
    CASE WHEN in_errors > 0 THEN '❌' ELSE '✅' END as status
FROM tik_email;

-- Ver últimos tickets criados via email
SELECT 
    t.id,
    t.subject,
    t.date_reg,
    e.email as from_account
FROM tik_tickets t
JOIN tik_email e ON t.email_id = e.id
WHERE t.source = 'email'
ORDER BY t.date_reg DESC
LIMIT 10;
```

---

## 🎯 Resumo para começar HOJE

**Se só tiveres 5 minutos:**

1. **Testa se funciona:** `curl https://seudominio.com/api/tickets/fetcher`
2. **Vê o resultado nos logs:** `tail -f /var/log/tickets.log`
3. **Se der erro, verifica:** Base de dados `tik_email` tem contas ativas?

**Se tiveres 30 minutos:**

1. Faz tudo acima
2. Cria uma conta de teste na `tik_email`
3. Testa: `curl https://seudominio.com/api/tickets/fetcher/ID_DA_CONTA`
4. Verifica se criou tickets na `tik_tickets`

**Se tiveres 2 horas:**

1. Faz tudo acima
2. Lê o código de `ticketsfetch.class.php`
3. Entende como funciona o parsing MIME
4. Testa adicionar filtros personalizados

**Agora já sabes o suficiente para começar a trabalhar no sistema de Fetcher de Tickets! 🚀**