# Queue Retry Command Documentation

## Table of Contents
- [Overview](#overview)
- [What is a Failed Job?](#what-is-a-failed-job)
- [Command Syntax](#command-syntax)
- [Parameters and Options](#parameters-and-options)
- [Usage Examples](#usage-examples)
- [Understanding the Output](#understanding-the-output)
- [Common Use Cases](#common-use-cases)
- [Best Practices](#best-practices)
- [Troubleshooting](#troubleshooting)
- [Technical Details](#technical-details)

## Overview

The `queue:retry` command allows you to **manually retry failed jobs** from the Dead Letter Queue (DLQ). When jobs fail after exhausting their maximum retry attempts during normal processing, they are moved to the `failed_jobs` table for manual intervention.

### Key Concepts
- **Failed Jobs**: Jobs that have exceeded their maximum retry attempts
- **Dead Letter Queue (DLQ)**: Storage for failed jobs awaiting manual retry
- **Manual Retry**: Developer intervention to reprocess failed jobs
- **Fresh Attempts**: Retried jobs get renewed attempt counters

## What is a Failed Job?

A job becomes "failed" when:
1. It throws an exception during execution
2. It has already used up all its `max_attempts` (usually 3 by default)
3. The queue system moves it to the `failed_jobs` table
4. It requires manual intervention to retry

### Example Job Lifecycle
```
Job Created → Queued → Processing → Failed (1/3) → Retry → Failed (2/3) → Retry → Failed (3/3) → Dead Letter Queue
                                                                                                      ↓
                                                                                            Manual Retry Command
```

## Command Syntax

```bash
php og queue:retry [ids] [--queue=QUEUE] [--failed-within=MINUTES]
```

### Basic Usage
```bash
# Retry all failed jobs from the last 24 hours (default)
php og queue:retry

# Retry a specific job by ID
php og queue:retry 123

# Retry multiple jobs
php og queue:retry 123,456,789
```

## Parameters and Options

### `ids` (optional)
- **Type**: String or comma-separated list
- **Description**: Specific job ID(s) to retry
- **Default**: If not provided, retries all jobs within time filter

```bash
# Single job
php og queue:retry 123

# Multiple jobs (comma-separated)
php og queue:retry 123,456,789
```

### `--queue` (optional)
- **Type**: String
- **Description**: Filter jobs by specific queue name
- **Default**: All queues

```bash
# Retry failed jobs only from 'email' queue
php og queue:retry --queue=email

# Retry jobs from 'reports' queue within last 2 hours
php og queue:retry --queue=reports --failed-within=120
```

### `--failed-within` (optional)
- **Type**: Integer (minutes)
- **Description**: Only retry jobs that failed within the last N minutes
- **Default**: 1440 (24 hours)

```bash
# Retry jobs that failed in the last hour
php og queue:retry --failed-within=60

# Retry jobs that failed in the last week
php og queue:retry --failed-within=10080
```

## Usage Examples

### Example 1: Retry All Recent Failed Jobs
```bash
# Retry all jobs that failed in the last 24 hours
php og queue:retry
```

**Output:**
```
 3/3 [============================] 100%

Retry completed: 3 success(es), 0 failure(s)
```

### Example 2: Retry Specific Job
```bash
# You noticed job ID 456 failed due to a temporary database issue
php og queue:retry 456
```

**Output:**
```
 1/1 [============================] 100%

Retry completed: 1 success(es), 0 failure(s)
```

### Example 3: Retry Multiple Specific Jobs
```bash
# Retry jobs 123, 456, and 789
php og queue:retry 123,456,789
```

**Output:**
```
 3/3 [============================] 100%

Retry completed: 2 success(es), 1 failure(s)
Could not retry job 789
```

### Example 4: Retry by Queue
```bash
# Only retry failed email jobs from the last 2 hours
php og queue:retry --queue=email --failed-within=120
```

**Output:**
```
 5/5 [============================] 100%

Retry completed: 5 success(es), 0 failure(s)
```

### Example 5: Retry Very Recent Failures
```bash
# Only retry jobs that failed in the last 30 minutes
php og queue:retry --failed-within=30
```

**Output:**
```
No jobs found for retry
```

## Understanding the Output

### Progress Bar
```
 3/3 [============================] 100%
```
- Shows real-time progress of retry operations
- Format: `current/total [progress_bar] percentage`

### Summary Line
```
Retry completed: 2 success(es), 1 failure(s)
```
- **Success count**: Jobs successfully requeued
- **Failure count**: Jobs that couldn't be retried

### Error Messages
```
Could not retry job 456
Error retrying job 789: Connection timeout
```
- Individual job failures with specific error details
- Helps identify persistent issues

### Warning Messages
```
No jobs found for retry
```
- Indicates no jobs matched the filter criteria
- Common when time filters are too restrictive

## Common Use Cases

### 1. After System Maintenance
After database maintenance or service updates:
```bash
# Retry all jobs that failed during the maintenance window
php og queue:retry --failed-within=120
```

### 2. Specific Service Recovery
When a specific external service was down:
```bash
# Retry all failed payment processing jobs
php og queue:retry --queue=payments
```

### 3. Individual Job Investigation
When investigating a specific job failure:
```bash
# Retry a specific job you've been debugging
php og queue:retry 12345
```

### 4. Bulk Recovery
After fixing a widespread issue:
```bash
# Retry all failed jobs from the last 8 hours
php og queue:retry --failed-within=480
```

### 5. Queue-Specific Recovery
When one service was problematic:
```bash
# Retry only email jobs that failed recently
php og queue:retry --queue=email --failed-within=60
```

## Best Practices

### 1. Investigate Before Retry
```bash
# First, check what failed and why
php og queue:stats
php og queue:failed

# Then retry after fixing the root cause
php og queue:retry --failed-within=60
```

### 2. Use Time Filters Wisely
```bash
# Good: Retry recent failures after fixing an issue
php og queue:retry --failed-within=120

# Avoid: Retrying very old failures without investigation
# php og queue:retry --failed-within=43200  # 30 days - probably not wise
```

### 3. Monitor Results
Always check the output to ensure retries are successful:
```bash
php og queue:retry 123
# Look for "1 success(es), 0 failure(s)"
```

### 4. Start Small
When dealing with many failed jobs:
```bash
# Start with a small time window
php og queue:retry --failed-within=30

# Then expand if successful
php og queue:retry --failed-within=60
```

### 5. Queue-Specific Recovery
Isolate retries by queue when possible:
```bash
# Retry email jobs separately from report jobs
php og queue:retry --queue=email
php og queue:retry --queue=reports
```

## Troubleshooting

### Problem: "No jobs found for retry"
**Possible Causes:**
- Time filter too restrictive
- All jobs already retried
- Jobs in different queue

**Solutions:**
```bash
# Check available failed jobs
php og queue:failed

# Expand time window
php og queue:retry --failed-within=1440

# Check specific queue
php og queue:retry --queue=your-queue-name
```

### Problem: "Could not retry job X"
**Possible Causes:**
- Job record corrupted
- Database connection issues
- Job already being processed

**Solutions:**
```bash
# Try retrying just that job
php og queue:retry X

# Check job details
php og queue:failed --id=X

# Consider removing the job if it's permanently broken
php og queue:forget X
```

### Problem: "Error retrying job X: [error message]"
**Possible Causes:**
- Serialization issues
- Missing job class
- Broken payload data

**Solutions:**
1. Fix the underlying issue first
2. Then retry the job
3. If unfixable, consider forgetting the job

### Problem: Jobs retry but fail again immediately
**Possible Causes:**
- Root cause not fixed
- Job depends on external service still down
- Job has invalid data

**Solutions:**
1. Investigate job failure logs
2. Fix the root cause
3. Consider modifying job data if needed
4. Wait for external services to recover

## Technical Details

### Database Impact
The retry command:
1. Reads from `failed_jobs` table
2. Updates `retry_count` and `last_retry_at` fields
3. Re-queues jobs using their `original_payload`
4. Does **not** delete jobs from `failed_jobs` (they remain for history)

### Retry Logic
```php
// Simplified retry process:
1. Find job by ID in failed_jobs table
2. Extract original_payload 
3. Push original_payload back to the original queue
4. Increment retry_count and update last_retry_at
5. Job gets fresh attempt_number when reprocessed
```

### Memory and Performance
- Command processes jobs in batches
- Shows progress bar for user feedback
- Handles errors gracefully without stopping
- Memory usage scales with number of jobs being retried

### Security Considerations
- Jobs retain original user/tenant context
- No privilege escalation occurs
- Failed jobs maintain their original permissions
- Audit trail preserved in failed_jobs table

### Integration with Queue System
- Retried jobs enter normal queue processing flow
- They get fresh `max_attempts` counter
- Original failure history preserved in failed_jobs table
- New failures will create new failed_jobs entries

## Related Commands

```bash
# View failed jobs
php og queue:failed

# Get queue statistics
php og queue:stats

# Remove failed jobs permanently
php og queue:forget 123

# Clear old failed jobs
php og queue:flush
```

## Summary

The `queue:retry` command is a powerful tool for manually recovering from job failures. It's designed to be safe, informative, and flexible, allowing developers to recover from various failure scenarios while maintaining audit trails and system integrity.

Remember: **Always investigate the cause of failures before retrying** to avoid repeating the same errors.