# QueueMaster - Configuração Completa

## Arquivo de Configuração

**Localização**: `Modules/Common/Config/queue-master.php`

Este arquivo contém toda a configuração do QueueMaster, organizada em seções lógicas.

## Estrutura Geral

```php
return [
    'enabled'      => true|false,
    'use'          => 'default',
    'redis'        => [...],
    'environments' => [...],
    'defaults'     => [...],
    'autoscaling'  => [...],
    'integration'  => [...],
    'process'      => [...],
    'logging'      => [...],
];
```

## Seções Detalhadas

### 1. Enabled (Ativar/Desativar)

```php
'enabled' => envVar('QUEUE_MASTER_ENABLED', true),
```

**Descrição**: Liga ou desliga completamente o QueueMaster.

**Tipo**: `boolean`

**Padrão**: `true`

**Variável de Ambiente**: `QUEUE_MASTER_ENABLED`

**Quando desabilitar**:
- Durante manutenção
- Para usar apenas workers manuais
- Em ambientes de teste específicos

### 2. Redis Configuration

#### 2.1 Conexão Base

```php
'use' => envVar('QUEUE_MASTER_REDIS', 'default'),
```

**Descrição**: Nome da conexão Redis a ser usada (referência a `database.connections.redis`).

**Tipo**: `string`

**Padrão**: `'default'`

**Opções**: Qualquer conexão definida em `database.php`

#### 2.2 Parâmetros de Conexão

```php
'redis' => [
    'timeout'      => envVar('QUEUE_MASTER_REDIS_TIMEOUT', 5),
    'read_timeout' => envVar('QUEUE_MASTER_REDIS_READ_TIMEOUT', 3),
    'persistent'   => envVar('QUEUE_MASTER_REDIS_PERSISTENT', true),
],
```

| Parâmetro | Tipo | Padrão | Descrição |
|-----------|------|--------|-----------|
| `timeout` | `int` | `5` | Timeout de conexão (segundos) |
| `read_timeout` | `int` | `3` | Timeout de leitura (segundos) |
| `persistent` | `bool` | `true` | Usar conexão persistente |

**Variáveis de Ambiente**:
- `QUEUE_MASTER_REDIS_TIMEOUT`
- `QUEUE_MASTER_REDIS_READ_TIMEOUT`
- `QUEUE_MASTER_REDIS_PERSISTENT`

**Recomendações**:
- **Production**: `persistent = true` (melhor performance)
- **Development**: Qualquer valor funciona
- **Timeout**: Aumente se Redis está em outra rede

### 3. Environments (Ambientes)

Define configuração de supervisors por ambiente (production, development, etc).

#### Estrutura Básica

```php
'environments' => [
    'production' => [
        'queue-name' => [
            // Configuração do supervisor para essa fila
        ],
    ],
    'development' => [
        // ...
    ],
],
```

#### Exemplo Completo (Production)

```php
'environments' => [
    'production' => [
        'default' => [
            'connection'    => 'rabbitmq',
            'queue'         => 'default',
            'balance'       => 'auto',
            'maxProcesses'  => envVar('DEFAULT_WORKERS', 5),
            'minProcesses'  => 1,
            'maxTime'       => 3600,
            'workerMaxTime' => 300,
            'maxJobs'       => 0,
            'memory'        => 512,
            'timeout'       => 60,
            'tries'         => 3,
            'sleep'         => 3,
            'backoff'       => 5,
            'vhost'         => envVar('RABBITMQ_VHOST', ('/' . envVar('DB_SERVER'))) ?? '/',
        ],
    ],
],
```

### 4. Parâmetros de Supervisor (Por Fila)

Cada supervisor (queue) pode ter os seguintes parâmetros:

#### 4.1 Conexão e Fila

| Parâmetro | Tipo | Descrição | Exemplo |
|-----------|------|-----------|---------|
| `connection` | `string` | Nome da conexão (RabbitMQ, Redis, etc) | `'rabbitmq'` |
| `queue` | `string` | Nome da fila a ser monitorada | `'default'`, `'reports'` |
| `vhost` | `string` | VHost do RabbitMQ para isolamento | `'/tenant1'`, `'/'` |

**Nota sobre vhost**:
- Formato: `'/tenant_name'` ou `'/'` (default)
- Usado para isolamento multi-tenant
- Pode ser sobrescrito via `--vhost` na linha de comando

#### 4.2 Autoscaling

| Parâmetro | Tipo | Valores | Descrição |
|-----------|------|---------|-----------|
| `balance` | `string` | `'auto'`, `'simple'`, `'off'` | Estratégia de autoscaling |
| `maxProcesses` | `int` | `1-100` | Número máximo de workers |
| `minProcesses` | `int` | `1-maxProcesses` | Número mínimo de workers |

**Estratégias de Balance**:

- **`off`**: Sem autoscaling, sempre `maxProcesses` workers
  ```php
  'balance' => 'off',
  'maxProcesses' => 3,  // Sempre 3 workers
  ```

- **`simple`**: Scaling básico baseado em tamanho da fila
  ```php
  'balance' => 'simple',
  'minProcesses' => 1,
  'maxProcesses' => 10,
  ```

- **`auto`**: Scaling avançado (considera carga, latência, pressão)
  ```php
  'balance' => 'auto',
  'minProcesses' => 2,
  'maxProcesses' => 20,
  ```

**Variáveis de Ambiente**:
- `DEFAULT_WORKERS`: Para fila 'default'
- `REPORTS_WORKERS`: Para fila 'reports'
- `SAFT_WORKERS`: Para fila 'saft-export'

#### 4.3 Limites de Worker

| Parâmetro | Tipo | Unidade | Descrição | Recomendação |
|-----------|------|---------|-----------|--------------|
| `maxTime` | `int` | segundos | Tempo máximo de vida do SUPERVISOR | 3600 (1h) |
| `workerMaxTime` | `int` | segundos | Tempo máximo de vida do WORKER | 300 (5min) |
| `maxJobs` | `int` | jobs | Máximo de jobs antes de reciclar (`0` = ilimitado) | 0 ou 50-100 |
| `memory` | `int` | MB | Limite de memória por worker | 256-1024 |

**IMPORTANTE**:
- `maxTime` vs `workerMaxTime`:
  - `maxTime`: Supervisor (não implementado/usado)
  - `workerMaxTime`: Worker individual (este que funciona!)
- Limites verificados **ENTRE jobs** (nunca mata job em execução)
- Se `maxJobs = 0`, workers reciclam apenas por tempo/memória

**Exemplos por Tipo de Job**:

```php
// Jobs rápidos (API calls, emails)
'workerMaxTime' => 300,   // 5 min
'maxJobs'       => 100,   // Recicla a cada 100 jobs
'memory'        => 256,   // 256MB

// Jobs médios (relatórios simples)
'workerMaxTime' => 600,   // 10 min
'maxJobs'       => 50,
'memory'        => 512,   // 512MB

// Jobs pesados (SAFT, relatórios complexos)
'workerMaxTime' => 3600,  // 1 hora
'maxJobs'       => 10,
'memory'        => 1024,  // 1GB
```

#### 4.4 Job Execution Settings

| Parâmetro | Tipo | Unidade | Descrição | Recomendação |
|-----------|------|---------|-----------|--------------|
| `timeout` | `int` | segundos | Timeout máximo por JOB individual | 60-600 |
| `tries` | `int` | tentativas | Número de tentativas em caso de falha | 1-5 |
| `sleep` | `int` | segundos | Tempo de sleep quando não há jobs | 3 |
| `backoff` | `int` | segundos | Delay entre retries | 5-60 |

**Timeout por Job**:
- Mata job se exceder tempo (via `SIGALRM`)
- Deve ser >= tempo esperado do job mais lento
- Se jobs variam muito, considere filas separadas

**Tries (Tentativas)**:
```php
'tries' => 3,   // Tenta até 3x antes de mover para DLQ
```

**Sleep (Idle)**:
```php
'sleep' => 3,   // Aguarda 3s quando não há jobs
```
- Menor valor = mais responsivo (mas mais CPU)
- Maior valor = menos CPU (mas delay ao receber jobs)

**Backoff (Retry Delay)**:
```php
'backoff' => 5,   // Aguarda 5s antes de retry
```
- Evita retry storms
- Dá tempo para condições transitórias resolverem

### 5. Defaults (Valores Padrão)

```php
'defaults' => [
    'connection'    => 'rabbitmq',
    'balance'       => 'auto',
    'maxProcesses'  => 3,
    'minProcesses'  => 1,
    'maxTime'       => 3600,
    'workerMaxTime' => 300,
    'maxJobs'       => 0,
    'memory'        => 512,
    'timeout'       => 60,
    'tries'         => 3,
    'sleep'         => 3,
    'backoff'       => 5,
],
```

**Descrição**: Valores aplicados a supervisors que não especificam esses parâmetros.

**Uso**: Define baseline para todas as filas. Pode ser sobrescrito por configuração específica.

### 6. Autoscaling Configuration

```php
'autoscaling' => [
    'enabled'              => envVar('QUEUE_MASTER_AUTOSCALING', true),
    'strategy'             => 'time',
    'cooldown'             => 3,
    'maxShift'             => 1,
    'scale_up_threshold'   => 3.0,
    'scale_down_threshold' => 1.0,
],
```

| Parâmetro | Tipo | Descrição | Recomendação |
|-----------|------|-----------|--------------|
| `enabled` | `bool` | Habilita autoscaling global | `true` em produção |
| `strategy` | `string` | Estratégia (`'time'`, `'size'`) | `'time'` |
| `cooldown` | `int` | Segundos entre ajustes | 3-10 |
| `maxShift` | `int` | Máx workers add/remove por vez | 1-2 |
| `scale_up_threshold` | `float` | Jobs/worker para escalar UP | 3.0 |
| `scale_down_threshold` | `float` | Jobs/worker para escalar DOWN | 1.0 |

**Estratégias**:

- **`time`**: Baseada em tempo de processamento
  - Escala UP se jobs demoram muito
  - Escala DOWN se workers ficam ociosos

- **`size`**: Baseada em tamanho da fila
  - Escala UP se fila cresce
  - Escala DOWN se fila vazia

**Thresholds**:

```php
// Se pressão > 3.0 jobs/worker → adiciona workers
'scale_up_threshold' => 3.0,

// Se pressão < 1.0 jobs/worker → remove workers
'scale_down_threshold' => 1.0,
```

**Cooldown**:
- Previne "flapping" (oscilação rápida)
- Aguarda N segundos após ajuste antes do próximo

**MaxShift**:
- Limita mudanças bruscas
- `maxShift = 1`: Adiciona/remove 1 worker por vez
- Evita overshooting

**Variável de Ambiente**: `QUEUE_MASTER_AUTOSCALING`

### 7. Integration Settings

```php
'integration' => [
    'queue_master' => [
        'enabled'            => true,
        'register_workers'   => true,
        'heartbeat_interval' => 30,
        'track_jobs'         => true,
        'track_shutdowns'    => true,
    ],
],
```

| Parâmetro | Tipo | Descrição |
|-----------|------|-----------|
| `enabled` | `bool` | Integração com Redis registry |
| `register_workers` | `bool` | Registrar workers no Redis |
| `heartbeat_interval` | `int` | Intervalo de heartbeat (segundos) |
| `track_jobs` | `bool` | Rastrear jobs ativos |
| `track_shutdowns` | `bool` | Registrar histórico de shutdowns |

**Quando desabilitar**:
- Testes unitários
- Ambiente isolado sem Redis
- Debug de problemas de integração

### 8. Process Management

```php
'process' => [
    'restart_cooldown'          => 1,
    'max_restart_attempts'      => 3,
    'graceful_shutdown_timeout' => 60,
],
```

| Parâmetro | Tipo | Descrição |
|-----------|------|-----------|
| `restart_cooldown` | `int` | Segundos antes de restart (evita restart loops) |
| `max_restart_attempts` | `int` | Tentativas antes de marcar worker como dead |
| `graceful_shutdown_timeout` | `int` | Tempo para graceful shutdown antes de force kill |

**Graceful Shutdown Timeout**:
```php
'graceful_shutdown_timeout' => 60,  // Aguarda 60s job terminar
```

- Workers recebem sinal de término
- Aguardam job atual finalizar
- Após timeout, força término (SIGKILL)

### 9. Logging

```php
'logging' => [
    'level'     => envVar('QUEUE_MASTER_LOG_LEVEL', 'info'),
    'to_syslog' => envVar('QUEUE_MASTER_SYSLOG', true),
],
```

| Parâmetro | Tipo | Valores | Descrição |
|-----------|------|---------|-----------|
| `level` | `string` | `'debug'`, `'info'`, `'warning'`, `'error'` | Nível de log |
| `to_syslog` | `bool` | `true`, `false` | Enviar logs para syslog |

**Níveis de Log**:

- **`debug`**: Tudo (verbose, muito output)
- **`info`**: Eventos importantes (recomendado development)
- **`warning`**: Apenas problemas (recomendado production)
- **`error`**: Apenas erros críticos

**Variáveis de Ambiente**:
- `QUEUE_MASTER_LOG_LEVEL`
- `QUEUE_MASTER_SYSLOG`

## Configurações por Ambiente

### Development (Exemplo Completo)

```php
'environments' => [
    'development' => [
        'default' => [
            'connection'    => 'rabbitmq',
            'queue'         => 'default',
            'balance'       => 'off',              // Sem autoscaling
            'maxProcesses'  => 2,                  // Poucos workers
            'minProcesses'  => 1,
            'maxTime'       => 1800,               // 30 min
            'workerMaxTime' => 300,                // 5 min
            'maxJobs'       => 0,                  // Ilimitado
            'memory'        => 256,                // 256MB
            'timeout'       => 60,                 // 1 min por job
            'tries'         => 3,
            'sleep'         => 3,
            'backoff'       => 5,
            'vhost'         => envVar('RABBITMQ_VHOST', ('/' . envVar('DB_SERVER'))) ?? '/',
        ],
    ],
],
```

**Características Development**:
- Menos workers (economia de recursos)
- Balance OFF (comportamento previsível)
- Memória menor
- Ideal para laptop/desktop

### Production (Exemplo Completo)

```php
'environments' => [
    'production' => [
        // Fila padrão (jobs gerais)
        'default' => [
            'connection'    => 'rabbitmq',
            'queue'         => 'default',
            'balance'       => 'auto',             // Autoscaling ativo
            'maxProcesses'  => envVar('DEFAULT_WORKERS', 5),
            'minProcesses'  => 1,
            'maxTime'       => 3600,               // 1 hora
            'workerMaxTime' => 300,                // 5 min
            'maxJobs'       => 0,
            'memory'        => 512,                // 512MB
            'timeout'       => 60,
            'tries'         => 3,
            'sleep'         => 3,
            'backoff'       => 5,
            'vhost'         => envVar('RABBITMQ_VHOST', ('/' . envVar('DB_SERVER'))) ?? '/',
        ],

        // Fila de relatórios (jobs pesados)
        'reports' => [
            'connection'    => 'rabbitmq',
            'queue'         => 'reports',
            'balance'       => 'auto',
            'maxProcesses'  => envVar('REPORTS_WORKERS', 5),
            'minProcesses'  => 1,
            'maxTime'       => 3600,
            'workerMaxTime' => 1800,               // 30 min (reports demoram)
            'maxJobs'       => 50,                 // Restart após 50 jobs
            'memory'        => 1024,               // 1GB
            'timeout'       => 600,                // 10 min por job
            'tries'         => 1,                  // Não retry (jobs pesados)
            'sleep'         => 5,
            'backoff'       => 60,
            'vhost'         => envVar('RABBITMQ_VHOST', ('/' . envVar('DB_SERVER'))) ?? '/',
        ],

        // Fila SAFT (jobs muito pesados)
        'saft-export' => [
            'connection'    => 'rabbitmq',
            'queue'         => 'saft-export',
            'balance'       => 'auto',
            'maxProcesses'  => envVar('SAFT_WORKERS', 3),
            'minProcesses'  => 1,
            'maxTime'       => 3600,
            'workerMaxTime' => 3600,               // 1 hora (SAFT pode demorar)
            'maxJobs'       => 0,
            'memory'        => 256,
            'timeout'       => 3600,
            'tries'         => 3,
            'sleep'         => 3,
            'backoff'       => 10,
            'vhost'         => envVar('RABBITMQ_VHOST', ('/' . envVar('DB_SERVER'))) ?? '/',
        ],
    ],
],
```

**Características Production**:
- Mais workers
- Autoscaling ativo
- Filas especializadas por tipo de job
- Limites ajustados por workload

## Configuração de VHosts

### VHost Padrão (Single-Tenant)

```php
'vhost' => '/',
```

### VHost Multi-Tenant

```php
'vhost' => envVar('RABBITMQ_VHOST', ('/' . envVar('DB_SERVER'))) ?? '/',
```

**Formato**: `'/tenant_name'` ou `'/'`

**Exemplos**:
```bash
# Tenant 1
RABBITMQ_VHOST=/tenant1
DB_SERVER=tenant1

# Tenant 2
RABBITMQ_VHOST=/tenant2
DB_SERVER=tenant2
```

### Sobrescrever VHost via CLI

```bash
# Ignora config e usa vhost específico
php og queue-master --vhost=/custom_tenant
```

## Variáveis de Ambiente Disponíveis

| Variável | Padrão | Descrição |
|----------|--------|-----------|
| `QUEUE_MASTER_ENABLED` | `true` | Habilita QueueMaster |
| `QUEUE_MASTER_REDIS` | `'default'` | Conexão Redis |
| `QUEUE_MASTER_REDIS_TIMEOUT` | `5` | Timeout conexão Redis (s) |
| `QUEUE_MASTER_REDIS_READ_TIMEOUT` | `3` | Timeout leitura Redis (s) |
| `QUEUE_MASTER_REDIS_PERSISTENT` | `true` | Conexão persistente Redis |
| `QUEUE_MASTER_AUTOSCALING` | `true` | Habilita autoscaling |
| `QUEUE_MASTER_LOG_LEVEL` | `'info'` | Nível de log |
| `QUEUE_MASTER_SYSLOG` | `true` | Log para syslog |
| `QUEUE_MASTER_ENV` | - | Ambiente (production/development) |
| `DEFAULT_WORKERS` | `5` | Workers fila default |
| `REPORTS_WORKERS` | `5` | Workers fila reports |
| `SAFT_WORKERS` | `3` | Workers fila saft-export |
| `RABBITMQ_VHOST` | `'/'.DB_SERVER` | VHost RabbitMQ |

## Boas Práticas de Configuração

### 1. Sizing de Workers

```php
// Fórmula básica:
maxProcesses = (CPU_cores × 2) / número_de_filas

// Exemplo: 4 cores, 2 filas
// default: 4 workers
// reports: 4 workers
```

### 2. Limites de Memória

```php
// Total memory = maxProcesses × memory × número_de_filas + overhead
// Overhead = ~500MB (Master + Supervisors + Redis)

// Exemplo: 2 filas, 5 workers, 512MB cada
// Total = (5 × 512MB × 2) + 500MB = 5.6GB
```

### 3. Timeout vs WorkerMaxTime

```php
'timeout'       => 60,    // Tempo POR JOB
'workerMaxTime' => 300,   // Tempo POR WORKER

// Relação: workerMaxTime >= timeout × avg_jobs_processed
// Se timeout=60s e processa ~5 jobs, workerMaxTime=300s OK
```

### 4. Tries e Backoff

```php
// Jobs idempotentes (podem ser retentados)
'tries'   => 3,
'backoff' => 5,

// Jobs não-idempotentes (não podem ser retentados)
'tries'   => 1,
'backoff' => 0,
```

### 5. Sleep e Responsividade

```php
// Alta responsividade (jobs frequentes)
'sleep' => 1,  // Verifica fila a cada 1s

// Baixa carga (jobs esporádicos)
'sleep' => 5,  // Verifica fila a cada 5s
```

## Troubleshooting de Configuração

### Workers não escalam

**Problema**: `balance = 'auto'` mas workers não aumentam

**Verificar**:
```php
'autoscaling' => [
    'enabled' => true,  // Deve estar true
],
'balance' => 'auto',    // Deve estar 'auto'
'maxProcesses' => 10,   // Deve ter headroom
```

### Workers morrem rápido demais

**Problema**: Workers reciclam muito rápido

**Verificar**:
```php
'workerMaxTime' => 300,  // Aumentar se necessário
'memory'        => 512,  // Aumentar se jobs consomem muita RAM
'maxJobs'       => 0,    // 0 = ilimitado
```

### Jobs falham por timeout

**Problema**: Jobs morrem com `timeout exceeded`

**Verificar**:
```php
'timeout' => 60,  // Aumentar para jobs lentos
```

## Próximos Documentos

- **[USAGE_GUIDE.md](USAGE_GUIDE.md)** - Como usar essas configurações na prática
- **[EXAMPLES.md](EXAMPLES.md)** - Exemplos completos de configuração
- **[MONITORING.md](MONITORING.md)** - Como monitorar impacto das configurações