# QueueMaster - Monitoramento

## Índice

1. [Verificação de Saúde](#verificação-de-saúde)
2. [Métricas Importantes](#métricas-importantes)
3. [Dashboard Web](#dashboard-web)
4. [Alertas Recomendados](#alertas-recomendados)
5. [Comandos de Diagnóstico](#comandos-de-diagnóstico)
6. [Shutdown Analysis](#shutdown-analysis)
7. [Workers QueueMaster vs Manual](#workers-queuemaster-vs-manual)
8. [Integração com Ferramentas](#integração-com-ferramentas)
9. [Autoscale Tab](#autoscale-tab)

---

## Verificação de Saúde

### Check Básico

```bash
# 1. Master está rodando?
ps aux | grep "queue-master" | grep -v "queue:work" | grep -v grep

# Esperado: 1 linha (1 processo Master)
```

**Output esperado**:
```
user  12340  0.1  0.5  php og queue-master
```

**Problemas**:
- **0 linhas**: Master não está rodando
- **2+ linhas**: Múltiplos masters (PROBLEMA!)

### Check de Workers

```bash
# Quantos workers ativos?
ps aux | grep "queue:work" | grep -v grep | wc -l

# Esperado: Depende da configuração (ex: 10 workers)
```

### Check do Listener de Comandos

```bash
# Processo listener ativo?
ps aux | grep 'queue-master:listen-commands' | grep -v grep

# Há backlog na fila de comandos?
redis-cli LLEN qm:command_queue:$(hostname)
```

**Interpretando**:
- Deve haver **um processo** `queue-master:listen-commands` por host. Zero linhas indica que autoscaling/comandos não serão aplicados.
- `LLEN` precisa permanecer próximo de `0`. Valores crescentes (>5 em produção) sinalizam que o listener não está consumindo ou não consegue executar os comandos (faltando sudo/systemctl).

**Alertar se**:
- Processo ausente por mais de 1 minuto.
- `LLEN` > 10 durante mais de 2 polls consecutivos.

### Check de Memória

```bash
# Uso total de memória
ps aux | grep -E "queue-master|queue:work" | grep -v grep | \
  awk '{sum+=$6} END {print sum/1024 " MB"}'

# Esperado: < 50% da RAM do servidor
```

### Check via API

```bash
# Overview rápido
curl -s https://og-tenant1.test:8003/api/queue-master/overview | jq '{
  total_workers,
  processing_workers,
  idle_workers,
  avg_queue_pressure,
  rabbitmq_healthy
}'
```

**Output esperado**:
```json
{
  "total_workers": 10,
  "processing_workers": 3,
  "idle_workers": 7,
  "avg_queue_pressure": 1.5,
  "rabbitmq_healthy": true
}
```

### Métricas instantâneas (CLI)

```bash
# Stats atuais (workers/filas/jobs) - passa --json para exportar
php og queue-master:stats --hours=24

# Apenas JSON (útil para anexar em RCA)
php og queue-master:stats --json > /tmp/qm-stats.json
```

### Captura histórica (Snapshots)

```bash
# Guarda snapshot no Redis (fica disponível na aba Monitoring)
php og queue-master:snapshot --store --label="deploy-$(date +%Y%m%d-%H%M)"

# Só gera arquivo local
php og queue-master:snapshot --json --output "_docs/snapshots/queue-master-$(date +%s).json"
```

> Use os filtros `--hours`, `--queue` e `--tenant` para capturar janelas específicas. Os snapshots guardados pelo `--store` podem ser visualizados/baixados na aba Monitoring do dashboard.

---

## Métricas Importantes

### 1. Worker Metrics

#### Total de Workers

**Métrica**: Número total de workers ativos

**Como obter**:
```bash
# Via CLI
ps aux | grep "queue:work" | grep -v grep | wc -l

# Via API
curl -s https://og-tenant1.test:8003/api/queue-master/overview | jq '.data.total_workers'
```

**Valores esperados**:
- Development: 2-5 workers
- Production: 10-50 workers (depende da config)

**Alerta se**:
- `total_workers = 0` (nenhum worker ativo)
- `total_workers < minProcesses` (configuração)
- `total_workers > maxProcesses × 1.2` (possível bug)

#### Taxa de Utilização

**Métrica**: Percentual de workers processando jobs

**Fórmula**: `(processing_workers / total_workers) × 100`

**Como obter**:
```bash
curl -s https://og-tenant1.test:8003/api/queue-master/overview | jq '
  (.data.processing_workers / .data.total_workers) * 100
'
```

**Interpretação**:
- **0-20%**: Sistema ocioso (pode reduzir workers)
- **20-60%**: Utilização normal
- **60-90%**: Alta carga (considere mais workers)
- **90-100%**: Saturação (URGENTE - adicione workers)

#### Uptime Médio de Workers

**Métrica**: Tempo médio de vida dos workers

**Como obter**:
```bash
curl -s https://og-tenant1.test:8003/api/queue-master/workers | jq '
  [.data.workers[].uptime] | add / length
'
```

**Valores esperados**:
- Com `workerMaxTime=300s`: ~150-280s (recicla próximo ao limite)
- Com `maxJobs=50`: Varia conforme velocidade de processamento

**Alerta se**:
- `avg_uptime < 60s` (workers morrem muito rápido)
- `avg_uptime > workerMaxTime × 1.2` (workers não estão reciclando)

#### Uso de Memória

**Métrica**: Memória usada por worker

**Como obter**:
```bash
curl -s https://og-tenant1.test:8003/api/queue-master/workers | jq '
  .data.workers[] | {pid, memory_usage_mb, memory_limit_mb: (.memory_limit / 1024 / 1024)}
'
```

**Interpretação**:
- **< 50% do limite**: Saudável
- **50-80% do limite**: Atenção (pode reciclar em breve)
- **> 80% do limite**: Crítico (reciclará logo)

**Alerta se**:
- Média > 80% do limite (considere aumentar limite)
- Tendência crescente (possível vazamento de memória)

### 2. Queue Metrics

#### Tamanho da Fila

**Métrica**: Número de mensagens pendentes

**Como obter**:
```bash
curl -s https://og-tenant1.test:8003/api/queue-master/queues | jq '
  .data.queues[] | {name, messages, consumers}
'
```

**Interpretação**:
- **0-10**: Fila vazia/normal
- **10-100**: Carga moderada
- **100-1000**: Alta carga
- **> 1000**: Backlog significativo

**Alerta se**:
- `messages > 1000` e crescendo
- Tendência de crescimento por > 30 minutos

#### Queue Pressure

**Métrica**: Razão messages / consumers

**Fórmula**: `pressure = messages / consumers`

**Como obter**:
```bash
curl -s https://og-tenant1.test:8003/api/queue-master/queues | jq '
  .data.queues[] | {name, pressure, pressure_status}
'
```

**Interpretação**:
- **< 2.0**: `low` - Sistema ocioso
- **2.0-5.0**: `medium` - Carga normal
- **5.0-10.0**: `high` - Considere mais workers
- **> 10.0**: `critical` - URGENTE - adicione workers

**Alerta se**:
- `pressure > 10.0`
- `pressure_status = "critical"`

#### Message Rate

**Métrica**: Taxa de mensagens/segundo

**Como obter**:
```bash
curl -s https://og-tenant1.test:8003/api/queue-master/queues | jq '
  .data.queues[] | {name, message_rate}
'
```

**Interpretação**:
- Depende do tipo de job e volume esperado
- Compare com baseline normal

**Alerta se**:
- Taxa > 2× baseline (pico inesperado)
- Taxa = 0 quando deveria haver atividade

### 3. Job Metrics

#### Jobs Processados

**Métrica**: Total de jobs completados

**Como obter**:
```bash
curl -s https://og-tenant1.test:8003/api/queue-master/overview | jq '
  .data.total_jobs_processed
'
```

**Use para**:
- Calcular throughput (jobs/hora)
- Comparar com períodos anteriores

#### Taxa de Falha

**Métrica**: Percentual de jobs falhados

**Fórmula**: `(failed_jobs / total_jobs) × 100`

**Como obter**:
```bash
curl -s https://og-tenant1.test:8003/api/queue-master/jobs/overview?hours=24 | jq '
  (.data.failed_jobs / .data.total_jobs) * 100
'
```

**Interpretação**:
- **< 1%**: Normal (alguns erros esperados)
- **1-5%**: Atenção (investigar causas)
- **> 5%**: Crítico (problema sistêmico)

**Alerta se**:
- Taxa > 5%
- Aumento súbito (> 2× baseline)

#### Duração Média de Jobs

**Métrica**: Tempo médio de execução

**Como obter**:
```bash
curl -s https://og-tenant1.test:8003/api/queue-master/jobs/overview | jq '
  .data.avg_duration_seconds
'
```

**Interpretação**:
- Depende do tipo de job
- Estabeleça baseline por queue

**Alerta se**:
- Duração > 2× baseline (jobs ficando lentos)
- Tendência crescente (possível degradação)

### 4. System Metrics

#### CPU Usage

```bash
# CPU total dos processos QueueMaster
ps aux | grep -E "queue-master|queue:work" | grep -v grep | \
  awk '{sum+=$3} END {print sum "%"}'
```

**Interpretação**:
- **< 50%**: Normal
- **50-80%**: Alta utilização
- **> 80%**: Saturação de CPU

#### Heartbeat Status

**Métrica**: Tempo desde último heartbeat

**Como obter**:
```bash
curl -s https://og-tenant1.test:8003/api/queue-master/workers | jq '
  .data.workers[] | {pid, last_heartbeat_ago}
'
```

**Alerta se**:
- `last_heartbeat_ago > 60s` (worker possivelmente travado)
- Verificar stale workers

---

## Dashboard Web

### Acessar Dashboard

```
URL: https://og-tenant1.test:8003/queue-master/dashboard
```

### Seções do Dashboard

#### 1. Overview

Mostra visão geral do sistema:
- Total de workers (ativos, processando, ociosos)
- Filas (contagem, mensagens, pressão)
- Jobs (totais, taxa de falha)
- Saúde do RabbitMQ

**Refresh**: A cada 5 segundos

#### 2. Workers Tab

Lista todos os workers com:
- PID, hostname, queue, vhost
- Status (processing, idle, active)
- Memória usada vs limite
- Jobs processados
- Uptime
- Job atual (se processando)

**Filtros disponíveis**:
- Por status
- Por queue
- Por hostname

**Ações**:
- Detalhes do worker (clique na linha)

#### 3. Queues Tab

Lista filas RabbitMQ:
- Nome da fila
- Mensagens pendentes
- Consumers
- Pressure (mensagens/consumer)
- Status (low, medium, high, critical)

**Ordenação**: Por pressure (descendente)

#### 4. Jobs Tab

##### Active Jobs
Jobs em execução no momento:
- Job class
- Worker (hostname:pid)
- Queue, vhost, tenant
- Tempo de execução
- Tentativa atual

##### Recent Jobs
Jobs completados recentemente:
- Job class
- Tempo de execução
- Status (completed)
- Resultado

##### Failed Jobs
Jobs falhados:
- Job class
- Exception class e message
- Stack trace snippet
- Tentativas

#### 5. Shutdown Analysis Tab

Análise detalhada de shutdowns:
- Histórico de shutdowns
- Estatísticas por razão
- Shutdowns anômalos
- Gráficos de tendência

**Ver seção [Shutdown Analysis](#shutdown-analysis) abaixo**

#### 6. Tenant Switches Tab

Rastreamento de mudanças de tenant:
- Histórico de switches
- Estatísticas (taxa de sucesso)
- Switches problemáticos

**Útil para**: Debug de problemas multi-tenant

#### 7. Autoscale Tab

Visualização dedicada ao AutoScaler:

- **Supervisor / Queue / Vhost**: identifica exatamente qual supervisor controla qual fila/vhost.
- **Processes**: número de workers atualmente em execução (contagem real dos pids registrados).
- **Target**: quantidade que o AutoScaler determinou na última decisão (depois do cooldown).
- **Cooldown**: badge mostra se o supervisor está elegível para nova decisão (`Available`) ou quantos segundos faltam.
- **Last reason**: string completa com motive (`scale_up backlog=…`, `scale_down …`, `no_change …`).
- **Updated at**: timestamp do último `persist()` do supervisor (deve mudar a cada ~1 s).

**Fonte dos dados**: `GET /api/v2/queue-master/autoscale/status`

```bash
curl -s https://og-tenant1.test:8003/api/v2/queue-master/autoscale/status | jq '.data[] | {
  supervisor,
  queue,
  current: .current_processes,
  target: .target_processes,
  cooldown: .cooldown
}'
```

**Interpretação rápida**:
- `Processes < Target`: comando de scale ainda está em processamento.
- `Processes > Target`: o AutoScaler pediu redução; aguarde o supervisor encerrar workers.
- `Cooldown.active = true`: novo comando só poderá ser enviado após o tempo restante.
- `reason` começando com `scale_up` ou `scale_down` indica ações recentes; `no_change` mostra equilíbrio.

Use essa aba para validar se o autoscaling está reagindo à carga e para cruzar com logs de `QueueMaster AutoScaler scheduled scale`.

---

## Alertas Recomendados

### Alertas Críticos (P1)

#### 1. Master Não Está Rodando

```bash
# Check
MASTER_COUNT=$(ps aux | grep "queue-master" | grep -v "queue:work" | grep -v grep | wc -l)
if [ "$MASTER_COUNT" -eq 0 ]; then
    echo "CRITICAL: QueueMaster not running"
fi
```

**Ação**: Iniciar QueueMaster imediatamente

#### 2. Nenhum Worker Ativo

```bash
# Check
WORKER_COUNT=$(ps aux | grep "queue:work" | grep -v grep | wc -l)
if [ "$WORKER_COUNT" -eq 0 ]; then
    echo "CRITICAL: No workers active"
fi
```

**Ação**: Verificar logs, reiniciar Master

#### 3. Queue Pressure Crítica

```bash
# Check via API
CRITICAL_QUEUES=$(curl -s https://og-tenant1.test:8003/api/queue-master/queues | \
  jq '[.data.queues[] | select(.pressure_status == "critical")] | length')

if [ "$CRITICAL_QUEUES" -gt 0 ]; then
    echo "CRITICAL: $CRITICAL_QUEUES queues with critical pressure"
fi
```

**Ação**: Aumentar workers, verificar se jobs estão travando

#### 4. Taxa de Falha Alta

```bash
# Check
FAILURE_RATE=$(curl -s https://og-tenant1.test:8003/api/queue-master/jobs/overview | \
  jq '.data.retry_rate')

if (( $(echo "$FAILURE_RATE > 5" | bc -l) )); then
    echo "CRITICAL: High job failure rate: $FAILURE_RATE%"
fi
```

**Ação**: Investigar exceções nos logs

### Alertas de Atenção (P2)

#### 1. Workers Stale

```bash
# Check
STALE_COUNT=$(curl -s https://og-tenant1.test:8003/api/queue-master/workers/stale | \
  jq '.data.total')

if [ "$STALE_COUNT" -gt 0 ]; then
    echo "WARNING: $STALE_COUNT stale workers"
fi
```

**Ação**: Investigar workers travados, considerar restart

#### 2. Memória Alta

```bash
# Check
AVG_MEMORY=$(curl -s https://og-tenant1.test:8003/api/queue-master/workers | \
  jq '[.data.workers[] | (.memory_usage / .memory_limit)] | add / length * 100')

if (( $(echo "$AVG_MEMORY > 80" | bc -l) )); then
    echo "WARNING: Average memory usage: $AVG_MEMORY%"
fi
```

**Ação**: Aumentar limite de memória ou investigar vazamentos

#### 3. Shutdowns Anômalos

```bash
# Check
ANOMALOUS=$(curl -s https://og-tenant1.test:8003/api/queue-master/shutdowns/anomalous | \
  jq '.data.total')

if [ "$ANOMALOUS" -gt 10 ]; then
    echo "WARNING: $ANOMALOUS anomalous shutdowns"
fi
```

**Ação**: Analisar shutdown reasons, corrigir problemas

### Alertas Informativos (P3)

#### 1. Baixa Utilização

```bash
# Check
UTILIZATION=$(curl -s https://og-tenant1.test:8003/api/queue-master/overview | \
  jq '(.data.processing_workers / .data.total_workers) * 100')

if (( $(echo "$UTILIZATION < 10" | bc -l) )); then
    echo "INFO: Low worker utilization: $UTILIZATION%"
fi
```

**Ação**: Considere reduzir workers (economia de recursos)

#### 2. Autoscaling em Ação

Monitore scaling events nos logs:

```bash
tail -f docker/logs/worker/error.log | grep -i "scaling\|scale"
```

---

## Comandos de Diagnóstico

### Diagnóstico Geral

#### Script de Health Check

```bash
#!/bin/bash

echo "=== QueueMaster Health Check ==="
echo ""

# 1. Master
MASTER_PID=$(ps aux | grep "queue-master" | grep -v "queue:work" | grep -v grep | awk '{print $2}')
if [ -z "$MASTER_PID" ]; then
    echo "[ERROR] Master not running"
    exit 1
else
    echo "[OK] Master running (PID $MASTER_PID)"
fi

# 2. Workers
WORKER_COUNT=$(ps aux | grep "queue:work" | grep -v grep | wc -l)
echo "[OK] $WORKER_COUNT workers active"

# 3. Memory
TOTAL_MEM=$(ps aux | grep -E "queue-master|queue:work" | grep -v grep | awk '{sum+=$6} END {print sum/1024}')
echo "[OK] Total memory: ${TOTAL_MEM} MB"

# 4. CPU
TOTAL_CPU=$(ps aux | grep -E "queue-master|queue:work" | grep -v grep | awk '{sum+=$3} END {print sum}')
echo "[OK] Total CPU: ${TOTAL_CPU}%"

# 5. Queue pressure
QUEUES=$(curl -s https://og-tenant1.test:8003/api/queue-master/queues)
echo ""
echo "Queue Pressure:"
echo "$QUEUES" | jq -r '.data.queues[] | "\(.name): \(.pressure) (\(.pressure_status))"'

# 6. Stale workers
STALE=$(curl -s https://og-tenant1.test:8003/api/queue-master/workers/stale | jq '.data.total')
if [ "$STALE" -gt 0 ]; then
    echo ""
    echo "[WARNING] $STALE stale workers detected"
else
    echo ""
    echo "[OK] No stale workers"
fi

echo ""
echo "=== Health Check Complete ==="
```

### Diagnóstico de Performance

```bash
#!/bin/bash

echo "=== QueueMaster Performance Report ==="
echo ""

# Jobs overview
JOBS=$(curl -s https://og-tenant1.test:8003/api/queue-master/jobs/overview)

echo "Jobs (Last 24h):"
echo "$JOBS" | jq '{
  total: .data.total_jobs,
  completed: .data.completed_jobs,
  failed: .data.failed_jobs,
  failure_rate: .data.retry_rate,
  avg_duration: .data.avg_duration_seconds
}'

echo ""
echo "Top Workers by Jobs Processed:"
curl -s https://og-tenant1.test:8003/api/queue-master/workers | \
  jq -r '.data.workers | sort_by(.jobs_processed) | reverse | .[0:5] |
    .[] | "\(.pid)\t\(.jobs_processed) jobs\t\(.memory_usage_mb)MB\t\(.uptime_human)"' | \
  column -t

echo ""
echo "Shutdown Reasons (Last 7 days):"
curl -s https://og-tenant1.test:8003/api/queue-master/shutdowns/stats?days=7 | \
  jq -r '.data.by_reason | to_entries[] | "\(.key): \(.value)"'
```

### Monitoramento Contínuo

```bash
# Watch workers count
watch -n 2 'ps aux | grep "queue:work" | grep -v grep | wc -l'

# Watch queue pressure
watch -n 5 'curl -s https://og-tenant1.test:8003/api/queue-master/queues | \
  jq -r ".data.queues[] | \"\(.name): \(.messages) msgs, \(.pressure) pressure\""'

# Watch memory usage
watch -n 5 'ps aux | grep -E "queue-master|queue:work" | grep -v grep | \
  awk "{sum+=\$6} END {print sum/1024 \" MB\"}"'
```

---

## Shutdown Analysis

### Interpretar Shutdown Reasons

#### Normal (Esperado)

| Reason | Descrição | Ação |
|--------|-----------|------|
| `max_time_reached` | Worker atingiu `workerMaxTime` | Nenhuma (comportamento normal) |
| `max_jobs_reached` | Worker processou `maxJobs` | Nenhuma (reciclagem normal) |
| `signal_sigterm` | Master enviou sinal de término | Nenhuma (shutdown gracioso) |
| `signal_sigint` | Ctrl+C ou kill manual | Nenhuma (shutdown manual) |

#### Atenção

| Reason | Descrição | Ação |
|--------|-----------|------|
| `memory_limit` | Worker excedeu limite de memória | Verificar se jobs consomem muita RAM, aumentar limite |

#### Crítico

| Reason | Descrição | Ação |
|--------|-----------|------|
| `fatal_error` | Erro fatal no código | URGENTE - Investigar stack trace, corrigir bug |
| `unknown` | Razão desconhecida | Investigar logs, verificar sistema |

### Análise de Padrões

#### Pattern 1: Shutdowns Frequentes por `memory_limit`

**Sintoma**:
```bash
curl -s https://og-tenant1.test:8003/api/queue-master/shutdowns/stats | \
  jq '.data.by_reason.memory_limit'
# Output: 850 (muito alto)
```

**Diagnóstico**:
- Jobs consomem muita memória
- Limite muito baixo
- Possível vazamento de memória

**Ação**:
1. Aumentar limite: `'memory' => 1024`
2. Otimizar jobs (liberar objetos grandes)
3. Reduzir `maxJobs` para forçar reciclagem mais cedo

#### Pattern 2: Workers com Uptime Muito Baixo

**Sintoma**:
```bash
curl -s https://og-tenant1.test:8003/api/queue-master/shutdowns/stats | \
  jq '.data.avg_uptime_seconds'
# Output: 15 (muito baixo!)
```

**Diagnóstico**:
- Workers morrem rapidamente
- Possível erro de configuração
- Jobs falhando imediatamente

**Ação**:
1. Ver shutdowns anômalos
2. Verificar logs de erro
3. Testar job manualmente

#### Pattern 3: Muitos `fatal_error`

**Sintoma**:
```bash
curl -s https://og-tenant1.test:8003/api/queue-master/shutdowns/stats | \
  jq '.data.by_reason.fatal_error'
# Output: 120 (crítico!)
```

**Diagnóstico**:
- Bug no código dos jobs
- Problema de dependências
- Corrupção de dados

**Ação**:
1. Ver failed jobs para identificar job problemático
2. Analisar stack traces
3. Reproduzir localmente e corrigir

### Usar Dashboard Shutdown Analysis

**Acesse**: `https://og-tenant1.test:8003/queue-master/dashboard` → Shutdown Analysis

**Visualizações**:

1. **Histórico**: Lista cronológica de shutdowns
2. **By Reason**: Gráfico de pizza com distribuição
3. **Anomalous**: Lista de shutdowns suspeitos
4. **Timeline**: Gráfico de linha ao longo do tempo

**Filtros**:
- Por hostname
- Por queue
- Por período

---

## Workers QueueMaster vs Manual

### Identificar Tipo de Worker

```bash
curl -s https://og-tenant1.test:8003/api/queue-master/workers | \
  jq '.data.workers[] | {pid, manager, managed_by}'
```

**Output**:
```json
{
  "pid": 12345,
  "manager": "queue-master",
  "managed_by": "QueueMaster"
}

{
  "pid": 12350,
  "manager": null,
  "managed_by": "Manual"
}
```

### Diferenças no Monitoramento

| Aspecto | QueueMaster | Manual |
|---------|-------------|--------|
| **Registro no Redis** | Sim | Depende (se usar integration) |
| **Heartbeat** | Automático (30s) | Sim (se configurado) |
| **Shutdown tracking** | Sim (registra razão) | Não |
| **Supervisor** | Sim (gerenciado) | Não |
| **Auto-restart** | Sim | Não |
| **Dashboard** | Visível | Visível se registrado |

### Monitorar Workers Manuais

```bash
# Via processo
ps aux | grep "queue:work" | grep -v "queue-master"

# Se registrados no Redis
curl -s https://og-tenant1.test:8003/api/queue-master/workers | \
  jq '.data.workers[] | select(.manager == null)'
```

**IMPORTANTE**: Workers manuais não aparecem em shutdown history.

---

## Integração com Ferramentas

### Prometheus/Grafana

**Endpoint de Métricas**:

Crie um exportador customizado:

```php
// Endpoint: /api/queue-master/metrics
Route::get('/metrics', function () {
    $overview = app(DashboardControllerAPI::class)->overview()->getData(true);
    $queues = app(DashboardControllerAPI::class)->queues()->getData(true);

    $metrics = [];

    // Workers
    $metrics[] = "queuemaster_workers_total {$overview['data']['total_workers']}";
    $metrics[] = "queuemaster_workers_processing {$overview['data']['processing_workers']}";
    $metrics[] = "queuemaster_workers_idle {$overview['data']['idle_workers']}";

    // Queues
    foreach ($queues['data']['queues'] as $queue) {
        $name = $queue['name'];
        $metrics[] = "queuemaster_queue_messages{queue=\"$name\"} {$queue['messages']}";
        $metrics[] = "queuemaster_queue_pressure{queue=\"$name\"} {$queue['pressure']}";
    }

    return response(implode("\n", $metrics))->header('Content-Type', 'text/plain');
});
```

**Grafana Dashboard**: Importar métricas via Prometheus.

### Nagios/Icinga

**Check Script**:

```bash
#!/bin/bash
# /usr/local/nagios/libexec/check_queuemaster.sh

STATE_OK=0
STATE_WARNING=1
STATE_CRITICAL=2
STATE_UNKNOWN=3

# Check Master
MASTER_COUNT=$(ps aux | grep "queue-master" | grep -v "queue:work" | grep -v grep | wc -l)
if [ "$MASTER_COUNT" -eq 0 ]; then
    echo "CRITICAL: Master not running"
    exit $STATE_CRITICAL
fi

# Check Queue Pressure
CRITICAL_QUEUES=$(curl -s https://og-tenant1.test:8003/api/queue-master/queues | \
  jq '[.data.queues[] | select(.pressure_status == "critical")] | length')

if [ "$CRITICAL_QUEUES" -gt 0 ]; then
    echo "CRITICAL: $CRITICAL_QUEUES queues with critical pressure"
    exit $STATE_CRITICAL
fi

echo "OK: QueueMaster healthy"
exit $STATE_OK
```

### Sentry/Error Tracking

QueueMaster já integra exceptions via:

```php
// Worker.php
app(QueueMasterIntegration::class)->recordException($exception);
```

**Ver exceptions**:
- Dashboard Sentry: Buscar tag `queuemaster`
- Filtrar por `exception_class`

### Logs Centralizados (ELK Stack)

**Formato de Log**:

```json
{
  "timestamp": "2024-01-01T00:00:00+00:00",
  "level": "info",
  "message": "Worker stopped",
  "context": {
    "pid": 12345,
    "shutdown_reason": "max_time_reached",
    "uptime_seconds": 300,
    "jobs_processed": 60,
    "queue": "default"
  }
}
```

**Kibana Queries**:

```
# Workers mortos por fatal_error
context.shutdown_reason: "fatal_error"

# Workers com uptime < 60s
context.uptime_seconds: [0 TO 60]

# Fila específica
context.queue: "reports"
```

---

## Próximos Documentos

- **[TROUBLESHOOTING.md](TROUBLESHOOTING.md)** - Resolução de problemas específicos
- **[DEVELOPMENT.md](DEVELOPMENT.md)** - Como desenvolver e extender QueueMaster
- **[API_REFERENCE.md](API_REFERENCE.md)** - Referência completa da API
