# Autenticação (Web, AJAX e API) — Documentação Técnica (Alto Nível)

Este documento descreve, em alto nível, os requisitos funcionais e não funcionais da autenticação no OfficeGest, com base nos ficheiros index.php (web), AjaxServer.php (AJAX) e api/index.php (API). O objetivo é orientar como funciona o login, manutenção de sessão, proteção das ações AJAX e o acesso à API (v1 e v2), bem como esclarecer como as migrações/upgrade correm no contexto do DSA.


## Visão Geral

- Web (index.php):
  - Redireciona utilizadores não autenticados para /login (exceto rotas específicas como dsa, cb, painéis e assets);
  - Processa credenciais (utilizador/senha) ou entrada por “panel” (PIN), valida IP/horário, limita sessões simultâneas e cria sessão de utilizador;
  - Integra 2FA via SMS (opcional), atualiza last seen e gere forçar alteração de password.

- AJAX (AjaxServer.php):
  - Por omissão, todas as ações exigem sessão válida (validateSession), exceto lista de ações públicas (login, login_panel, smsAuthLogin, etc.);
  - Fornece endpoints JSON de autenticação/heartbeat e despacha ações para scripts em _ajax/.

- API (api/index.php):
  - Versão v1: Autenticação HTTP Basic com utilizador + API Hash (não aceita password normal); aplica CORS e throttling em Redis (mensal e por IP/60s);
  - Versão v2: Bypass no ficheiro de entrada; a autenticação é delegada às rotas (Guard/Router), permitindo outros esquemas (ex.: bearer) definidos a nível de módulos.

- DSA e Migrações/Upgrade:
  - Quando a versão da aplicação (ficheiro/licença) diverge da versão gravada na BD, a API executa OG_UpgradeProcess (migrando/atualizando) de forma automática (api/index.php);
  - Em “/dsa”, existem ações guiadas upgradeprocess/upgrade (_views/dsa.php) que correm o OG_UpgradeProcess com UI informativa (útil em manutenção/upgrade assistido);
  - A página DSA também expõe listsessions (gestão de sessões ativas) com controlo de permissões.


## Requisitos Funcionais

1) Autenticação Web (login por formulário)
- Entrada: POST user, password; modo “panel” aceita PIN (checkUserPin).
- Validações:
  - Existência de credenciais; utilizador ativo (IsActive());
  - Limite de sessões ativas (qtd_slots vs a->maxusers());
  - Regras de IP (validateIP) e janela temporal (validateTimeable);
  - Forçar alteração de password por primeira entrada, expiração (Passwordlifetime) ou flag (force_password_change).
- Sessão e Estado:
  - Cria sessão via Session->create($u);
  - Guarda $_SESSION['user'], $_SESSION['panel'] (se aplicável) e marca $_SESSION['just_login'];
  - Atualiza web_lastseen_date_c e web_lastseen_ip_c;
  - Suporta 2FA por SMS (parâmetro LoginSMSAuth = 'S'): envia código, valida com smsAuthLogin e fixa cookie de confiança por duração configurável.
- Resposta/Fluxo:
  - index.php redireciona para / após sucesso; AjaxServer->login responde JSON padronizado (success/erro com codes). Tentativas falhadas aplicam sleep progressivo.

2) Proteção de Ações AJAX
- validateSession() é obrigatório, exceto para ações públicas listadas em $ignValidate;
- Critérios de sessão válida:
  - Flag de admin em $_SESSION['isadmin'] compatível com md5(session_id . REMOTE_ADDR) OU
  - $_SESSION['user'] presente e verificada por Session->verify($_SESSION['user']);
- Em sucesso, o objeto User é carregado (loadUser) se ainda não estiver em memória;
- Respostas JSON uniformes: respond() e directrespond().

3) Autenticação API v1 (HTTP Basic + API Hash)
- Headers e CORS configurados (origem, métodos, headers, credenciais);
- Throttling (Redis):
  - Mensal por instalação (api:monthly_tokens com TTL para o último dia do mês), retorna 429 se esgotado;
  - Por IP/60s (api:tokens:<ip>), retorna 429 quando excede; respostas informam retry_after;
- Processo de autenticação:
  - Exige Authorization Basic: username = web_login_c; password = API Hash do utilizador (40 chars);
  - Se password enviada for a password real (e não o hash), recusa e devolve 401 com payload contendo o hash recomendado, licença e user_id;
  - Carrega utilizador e mantém contexto para a execução de controladores/rotas;
- Gate de módulo: se instalação não tiver módulo API ativo (salvo utilitários/GDPR/host específico), retorna 503;
- Registos de acesso (log) e debug de POST, conforme flags.

4) Autenticação API v2 (Guard/Router)
- Se handler inicia por v2/, o ficheiro de entrada faz bypass à autenticação básica e delega para o router/guards;
- A definição do esquema (ex.: Bearer token/JWT/API key por header X-ACCESS_TOKEN) é feita em camadas de rota/guard (Og\Modules\Common\Guard\GuardFacade), possibilitando políticas por endpoint, scopes e perfis.

5) DSA — Ações e Upgrade/Migrações
- Navegação para /dsa:<token> ativa ações:
  - listsessions: UI de gestão de sessões (terminar sessão própria ou geral consoante permissões);
  - upgradeprocess: UI para correr OG_UpgradeProcess (FILES_DIR/upgrade/upgrade.php) — executa migrações/alterações necessárias para alinhar BD e código;
  - upgrade (com sessão especial og_upgrade): continua/update assistido (FILES_DIR/upgrade/update.php);
- API também invoca OG_UpgradeProcess automaticamente quando og_version != og_db_version, garantindo que migrações correm antes do processamento do pedido.


## Requisitos Não Funcionais

- Segurança
  - Web: sessão atrelada a verificação de IP (opcional por regra de negócio) e janela temporal; admin usa checksum md5(session_id + REMOTE_ADDR);
  - API v1: nunca enviar a password normal — usar API Hash; CORS controlado; throttling para mitigação de abuso; logs de acesso;
  - AJAX: todas as ações (fora da whitelist) exigem sessão válida; respostas não expõem detalhes sensíveis.

- Idempotência e Robustez
  - Controlo de tentativas no login AJAX com atraso progressivo (sleep);
  - Deteção de divergência de versão app/BD dispara upgrade automático (API) ou assistido (DSA), evitando estados inconsistentes;
  - Heartbeat permite checagem simples de sessão.

- Compatibilidade
  - API v2 delega auth a Guards, permitindo múltiplos esquemas de token sem alterar o front‑controller;
  - CORS flexível (origem do request) para integrações externas, mantendo credenciais quando necessário.

- Observabilidade
  - Registos por canal (ex.: "api") para acessos, POST em DEBUG, e erros de autenticação (utilizador inválido/hash inválido);
  - DSA upgradeprocess com UI informativa (mensagens incrementais via e_updconsole).

- Desempenho
  - Throttling Redis minimiza abuso de endpoints;
  - Verificações de sessão e carregamento de utilizador são realizados on‑demand;
  - Upgrade executado apenas quando necessário (mismatch de versões) ou sob ação DSA.


## Fluxos Resumidos

- Login Web (form):
  1) Utilizador submete user/password (ou PIN por panel);
  2) Sistema verifica limites/ativação/IP/time‑window e 2FA, cria sessão e define variáveis de sessão;
  3) Redireciona para / (ou painel) e atualiza last seen.

- Login AJAX:
  1) Chamar AjaxServer->login/login_panel via POST;
  2) Resposta JSON com success e eventuais flags (force_change_pass) ou códigos de erro (-1 sem credenciais, -2 inativo, -3 inválido, -4 limite sessões, -5 IP inválido, -6 janela de horário).

- Acesso API v1:
  1) Enviar Authorization: Basic base64(user:APIHASH);
  2) Em caso de password normal, 401 com hash recomendado; com hash inválido, 401;
  3) Passando autenticação, aplica throttling e despacha para rota/controlador.

- Acesso API v2:
  1) Pedido v2/… é tratado pelo router;
  2) Guards definidos nas rotas executam autenticação/autorização (ex.: Bearer, escopos);
  3) Em caso de falha, resposta conforme política da rota/guard.

- Migrações/Upgrade (DSA/API):
  - Automático na API quando versões divergem;
  - Manual/assistido via /dsa:upgradeprocess (UI) e /dsa:upgrade;
  - Em ambos os casos, executa OG_UpgradeProcess (migrando BD, ficheiros, etc.).


## Boas Práticas de Uso

- API v1: gerar e usar sempre o API Hash, nunca a password; armazenar o hash de forma segura no integrador;
- Ativar/ajustar throttling (limites no Redis) conforme o volume esperado;
- Para integrações modernas, preferir API v2 com Guards e tokens de curta duração (Bearer/JWT) e scopes específicos;
- Usar 2FA por SMS para utilizadores sensíveis; utilizar cookies de confiança com duração alinhada à política de segurança;
- Em upgrades, preferir o fluxo DSA para transparência ao operador ou garantir janelas controladas quando ocorrer via API.


## Referências de Código

- Web: index.php — blocos de login, redireções e carregamento de views; integração com Router e DSA;
- AJAX: AjaxServer.php — validateSession(), login(), login_panel(), smsAuthLogin();
- API: api/index.php — CORS, throttling Redis, Basic + API Hash, bypass v2, verificação de módulos, OG_UpgradeProcess;
- DSA: _views/dsa.php — listsessions, upgradeprocess, upgrade, consola de upgrade.


---

## Guia Rápido (para iniciantes)

Para quem é: Devs que querem autenticar-se via Web/AJAX/API rapidamente.
O que você vai aprender: fazer login, chamar a API v1 com hash, entender o fluxo da API v2 e rodar o DSA.

1) Login AJAX (exemplo de request e resposta)

Request (form-urlencoded ou JSON):
```
POST /AjaxServer.php?action=login
user=jose&password=segredo
```
Resposta de sucesso:
```json
{"success":true,"params":{}}
```
Resposta de erro (exemplo: credenciais inválidas):
```json
{"success":false,"code":-3}
```
Códigos comuns: -1 faltam credenciais, -2 inativo, -3 inválido, -4 limite sessões, -5 IP inválido, -6 fora do horário.

2) Login com 2FA por SMS (fluxo)
- Faça o POST de login; se exigir 2FA, virá code=3 e um smsid.
- Envie o pin recebido por SMS:
```
POST /AjaxServer.php?action=smsAuthLogin
id=<smsid>&pin=<código>
```
- Resposta: success=true e a sessão será criada.

3) API v1 com curl (HTTP Basic usando API Hash)

- Gere/obtenha o API Hash do utilizador (40 chars). Nunca use a password normal.
- Faça a chamada:
```
curl -u usuario@empresa:APIHASH \
  -H "Content-Type: application/json" \
  "https://seu.dominio/api/?handler=entities/list"
```
Erros comuns:
- 401 se usar password normal (a resposta pode sugerir o hash correto e trazer user_id/licença).
- 429 se ultrapassar limites (mensal/IP). Veja retry_after na resposta.

4) API v2 (delegada às rotas/guards)

- Requests para handler iniciando com v2/ fazem bypass do Basic. A autenticação acontece nas rotas (guards).
- Exemplo (pseudocode):
```
GET /api/?handler=v2/users
Authorization: Bearer <token>
```
A validação do token, escopos e políticas depende do guard configurado na rota.

5) DSA — Executar Upgrade de forma assistida

- Acesse: /dsa:upgradeprocess (base64 tokenizado pelo sistema) para UI do upgrade.
- A API também executa upgrade automaticamente quando a versão da app diverge da versão em BD.


## Fluxos (diagramas simplificados)

Login AJAX:
```
Cliente -> AjaxServer(login) -> validações (IP/horário/slot/2FA) -> cria sessão -> JSON {success}
```

API v1:
```
Cliente -> api/index.php -> CORS + throttling -> HTTP Basic (user/APIHASH) -> carrega user -> Router/Controller
```

API v2:
```
Cliente -> api/index.php (bypass) -> Router -> Guards (Bearer/JWT/API Key) -> Controller
```


## Checklist de Integração Rápida

- [ ] Web/AJAX: confirmar que a app redireciona não logados para /login e que AjaxServer valida sessão.
- [ ] API v1: usar sempre API Hash (não password); configurar CORS; observar limites no Redis (mensal e por IP/60s).
- [ ] API v2: definir esquema de auth nas rotas/guards (tokens de curta duração e escopos específicos).
- [ ] DSA: conhecer /dsa:upgradeprocess e garantir que upgrades não interrompam operações críticas.


## Troubleshooting

- 401 na API v1 usando password normal: gere o API Hash e replique com -u user:APIHASH.
- 429 na API: aumente limites no Redis (se apropriado) ou reduza taxa de chamadas; cheque retry_after.
- Sessão inválida no AJAX: verifique $_SESSION e a verificação md5(session_id + REMOTE_ADDR) para admin.
- 2FA não chega: revisar provedores de SMS e validade do pin (date_expire no ActValidate).


## Glossário Rápido

- API Hash: token de 40 chars por utilizador, usado em Basic Auth no lugar da password.
- Guard: camada que valida o acesso nas rotas (API v2), podendo exigir Bearer, escopos, etc.
- Throttling: limitação de uso (Redis) por mês e por IP/60s para proteger a API.
