# Sistema de Traits - Framework OfficeGest

## Visão Geral do Sistema de Traits

O sistema de traits do framework OfficeGest implementa o padrão **Mixin** através do sistema nativo de traits do PHP, permitindo a composição de funcionalidades reutilizáveis entre classes sem recorrer à herança múltipla. Este sistema é fundamental para a arquitectura modular do framework, proporcionando funcionalidades transversais que podem ser aplicadas a diferentes tipos de classes.

O sistema está organizado em 8 traits especializados, cada um responsável por uma área específica de funcionalidade, seguindo o princípio da **Single Responsibility** e permitindo composições flexíveis através do padrão **Composition over Inheritance**.

## Arquitectura e Padrões de Design

### Mixin Pattern
Os traits implementam o padrão Mixin, permitindo que classes recebam funcionalidades específicas sem estabelecer relações hierárquicas. Cada trait adiciona um conjunto coeso de métodos e propriedades relacionados a um domínio específico.

### Composition Pattern
O sistema privilegia a composição sobre a herança, onde classes podem combinar múltiplos traits para obter as funcionalidades necessárias, mantendo flexibilidade e evitando os problemas clássicos da herança múltipla.

### Single Responsibility
Cada trait tem uma responsabilidade bem definida:
- **Filterable**: Normalização de filtros de entrada
- **MemoryManagementTrait**: Gestão de memória e optimização
- **Macroable**: Extensão dinâmica de funcionalidades
- **ErrorHandlingTrait**: Manipulação robusta de erros
- **Dumpable**: Debugging e diagnóstico
- **EnumCaseToArray**: Conversão de enums para arrays
- **MemoryOptimizedTrait**: Processamento optimizado de grandes datasets
- **ResolvesDumpSource**: Resolução de origem em debugging

### Dependency Injection Integration
Os traits integram-se nativamente com o sistema de dependency injection do framework, utilizando helper functions como `app()` para resolver dependências e aceder ao container quando necessário.

## Estrutura e Componentes

### Traits de Base (Core Traits)

#### Macroable
**Localização**: `Modules/Common/Traits/Macroable.php`

Implementa o padrão **Macro** permitindo a extensão dinâmica de classes em runtime. Este trait é fundamental para a arquitectura extensível do framework.

**Funcionalidades Principais**:
- **Dynamic Method Registration**: `macro(string $name, callable|object $macro)` - regista novos métodos dinamicamente
- **Method Mixing**: `mixin(object $mixin, bool $replace = true)` - adiciona todos os métodos públicos/protegidos de um objecto
- **Method Resolution**: `__call()` e `__callStatic()` - resolução automática de macros em contextos de instância e estáticos
- **Method Verification**: `hasMacro(string $name)` - verifica existência de macros

**Padrões de Integração**:
```php
use Og\Modules\Common\Traits\Macroable;

class Request extends SymfonyRequest {
    use Macroable;

    // Permite extensões dinâmicas:
    // Request::macro('customMethod', function() { ... });
}
```

**Casos de Uso**:
- Extensão de funcionalidades em classes base sem modificar o código fonte
- Plugin systems e módulos dinâmicos
- Adaptação de APIs legacy

### Traits Comportamentais (Behavioral Traits)

#### Filterable
**Localização**: `Modules/Common/Traits/Filterable.php`

Trait especializado para normalização de parâmetros de filtro em requests. Implementa o padrão **Transformer** para padronizar formatos de entrada.

**Funcionalidades Principais**:
- **Filter Normalization**: `prepareForValidation()` - normaliza filtros de entrada para formato padrão
- **Validation Rules**: `getFilterRules()` - fornece regras de validação para estruturas de filtro
- **Chain Compatibility**: Integração com sistemas de validação parent através de `method_exists` checks

**Formato de Normalização**:
```php
// Input simples é transformado:
['filter' => ['status' => 'active']]
// Para formato estruturado:
['filter' => ['status' => ['value' => 'active', 'operator' => 'equal']]]
```

**Padrões de Uso**:
Utilizado primariamente em **FilterRequest classes** para APIs:
```php
class AmsHandlingServiceFilterRequest {
    use Filterable;
    // Herda capacidades de normalização de filtros
}
```

#### Dumpable
**Localização**: `Modules/Common/Traits/Dumpable.php`

Sistema avançado de debugging que adiciona capacidades de dump com rastreamento de origem. Implementa o padrão **Observer** para debugging contexts.

**Funcionalidades Principais**:
- **Smart Dumping**: `dump(?string $message, array $properties)` - dump com contexto opcional
- **Selective Dumping**: `dumpOnly(string ...$properties)` - dump de propriedades específicas
- **Debug Tracing**: `getDebugInfo()` - resolve automaticamente ficheiro e linha de origem
- **Chained Operations**: Todos os métodos retornam `$this` para method chaining

**Sistema de Rastreamento**:
O trait utiliza `debug_backtrace()` com algoritmo inteligente para determinar o contexto real da chamada, ignorando os métodos internos do trait.

**Métodos Avançados**:
- `dd()` / `ddOnly()` - dump and die variants
- `dumpWith()` / `ddWith()` - dump com mensagem personalizada
- Formatação automática de contexto debug com ficheiro:linha

### Traits de Base de Dados (Database Traits)

#### MemoryOptimizedTrait
**Localização**: `Modules/Common/Traits/MemoryOptimizedTrait.php`

Sistema sofisticado de processamento optimizado para grandes datasets. Implementa **Generator Pattern** e **Streaming Pattern** para processamento eficiente de memória.

**Funcionalidades Principais**:
- **Chunked Processing**: `processRecordsInChunks()` - processa registos em chunks usando generators
- **Custom Query Processing**: `processCustomQueryInChunks()` - suporte para SQL customizado
- **Array Processing**: `processArrayInChunks()` - processamento de arrays grandes
- **Stream Generation**: `streamCSVFromGenerator()` - geração de CSV streaming
- **File Writing**: `writeDataToFile()` - escrita eficiente de grandes datasets

**Padrões de Database Connection**:
O trait implementa **Strategy Pattern** para resolução de conexões:
1. `$this->database` (BaseClass pattern)
2. `$this->db` (Controller pattern)
3. `app('database')` (Container pattern)
4. `global $db` (Legacy pattern)

**Optimizações de Memória**:
- Garbage collection automático a cada 4 chunks
- Cálculo dinâmico de chunk size baseado na memória disponível
- Logging de progresso para datasets grandes (>10k records)
- Streaming de dados para ficheiros sem carregar tudo em memória

**Exemplos de Uso**:
```php
// Processamento de grandes reports
foreach ($this->processRecordsInChunks('orders', ['*'], ['status' => 'active']) as $order) {
    // Processa uma order de cada vez sem impacto na memória
}
```

#### ResolvesDumpSource
**Localização**: `Modules/Common/Traits/ResolvesDumpSource.php`

Sistema especializado para resolução de origem em contextos de debugging, especialmente integrado com sistemas de dump externos como Symfony VarDumper.

**Funcionalidades Principais**:
- **Trace Resolution**: `resolveDumpSource()` - resolve origem real de dumps através de call stacks complexos
- **Editor Integration**: `resolveSourceHref()` - gera links para editores (VSCode, PHPStorm, etc.)
- **Path Normalization**: Conversão automática para paths relativos baseado em `rootPath()`

**Sistema de Ajuste de Trace**:
```php
protected static array $adjustableTraces = [
    'symfony/var-dumper/Resources/functions/dump.php' => 1,
    // Permite ajustar níveis de trace para diferentes ferramentas
];
```

### Traits de Validação e Conversão

#### EnumCaseToArray
**Localização**: `Modules/Common/Traits/EnumCaseToArray.php`

Trait especializado para enums que fornece funcionalidades de conversão e manipulação. Implementa **Converter Pattern** para transformações de dados.

**Funcionalidades Principais**:
- **SQL Case Generation**: `toSqlCase()` - gera CASE SQL statements para enums
- **Array Conversions**: `toArray()`, `valuesToArray()` - conversões para arrays simples
- **Label Management**: `labels()` - extracto de labels de enums
- **Select2 Integration**: `toArraySelect2()` - formato específico para Select2 widget

**Exemplo de SQL Generation**:
```php
// Gera: CASE status WHEN 'A' THEN LOWER('active') WHEN 'I' THEN LOWER('inactive') ELSE '' END = ?
CustomerType::toSqlCase('status', '= ?');
```

**Padrão de Uso em Enums**:
```php
enum CustomerType: string {
    use EnumCaseToArray;

    case COMPANY = 'E';
    case PRIVATE_INDIVIDUAL = 'P';

    public function label(): string {
        return match($this) {
            self::COMPANY => 'Empresa',
            self::PRIVATE_INDIVIDUAL => 'Particular',
        };
    }
}
```

### Traits de Performance e Gestão de Memória

#### MemoryManagementTrait
**Localização**: `Modules/Common/Traits/MemoryManagementTrait.php`

Sistema completo de gestão de memória com monitoring e optimização automática. Implementa **Monitor Pattern** e **Strategy Pattern** para gestão de recursos.

**Funcionalidades Principais**:
- **Memory Configuration**: `configureMemoryLimits()` - configuração dinâmica de limites
- **Threshold Monitoring**: `checkMemoryThreshold()` - monitoring automático com thresholds warning/critical
- **Batch Processing**: `processInBatches()` - processamento em lotes com optimização
- **Memory Statistics**: `getMemoryStats()` - estatísticas detalhadas de uso
- **Auto Optimization**: `optimizeMemoryUsage()` - garbage collection automático

**Sistema de Thresholds**:
```php
private array $memoryThresholds = [
    'warning' => 0.75,   // 75% - trigger optimization
    'critical' => 0.90,  // 90% - trigger critical handling
];
```

**Integração com Logging**:
O trait integra-se com múltiplos sistemas de logging:
- `$this->e_updconsole()` (Console output)
- `loggerBatch()` (Structured logging)

#### ErrorHandlingTrait
**Localização**: `Modules/Common/Traits/ErrorHandlingTrait.php`

Sistema robusto de handling de erros especialmente concebido para jobs e processamentos críticos. Implementa **Error Handler Pattern** com categorização inteligente.

**Funcionalidades Principais**:
- **Lenient Error Handling**: `withLenientErrorHandling()` - execução com supressão selectiva de erros
- **Error Categorization**: Classificação automática entre erros fatais, críticos e ignoráveis
- **Smart Notifications**: Sistema inteligente de notificações baseado em padrões de erro
- **Integration Systems**: Integração com Mattermost, logging estruturado e formatters

**Padrões de Error Detection**:
```php
private const array FATAL_PATTERNS = [
    'undefined function',
    'class .*not found',
    'maximum execution time',
    'guzzlehttp\\\\exception\\\\clientexception',
    'connection.*refused',
    // 46 padrões definidos para detecção automática
];
```

**Sistema de Notificações**:
- **Context Formatters**: `ContextFormatter`, `JobContextFormatter`, `StackTraceFormatter`
- **Mattermost Integration**: Notificações automáticas para erros críticos
- **Rich Context**: Informações de host, worker_id, memory usage, load average

**Padrão de Uso em Jobs**:
```php
class EmailJob extends Job {
    use ErrorHandlingTrait;

    public function handle(): void {
        $this->withLenientErrorHandling(function () {
            // Código que pode gerar erros não-críticos
            // Erros críticos ainda causam falha do job
        });
    }
}
```

## Funcionalidades Avançadas

### Trait Composition

O sistema de traits do OfficeGest foi concebido para composição harmoniosa, evitando conflitos através de design cuidadoso:

**Princípios de Design**:
- **Non-overlapping Methods**: Cada trait tem métodos únicos, evitando conflicts
- **Consistent Naming**: Padrões consistentes de nomenclatura (`get*`, `process*`, `configure*`)
- **Chainable Methods**: Muitos métodos retornam `$this` para method chaining
- **Optional Dependencies**: Uso de `method_exists()` e `function_exists()` para dependências opcionais

**Padrões de Composição Comum**:
```php
class AdvancedProcessor {
    use MemoryOptimizedTrait;        // Para processamento de grandes dados
    use MemoryManagementTrait;       // Para monitoring de memória
    use ErrorHandlingTrait;          // Para handling robusto de erros
    use Dumpable;                    // Para debugging durante desenvolvimento
}
```

### Cross-Trait Dependencies

Alguns traits implementam padrões de dependência soft através de verificações dinâmicas:

**MemoryManagementTrait + Classes**:
```php
// Verifica se a classe tem método de limpeza
if (method_exists($this, 'clearTemporaryData')) {
    $this->clearTemporaryData();
}
```

**Filterable + Validation Systems**:
```php
// Integração com sistemas de validação parent
if (method_exists(get_parent_class($this), 'prepareForValidation')) {
    parent::prepareForValidation();
}
```

### Dynamic Behavior

#### Macroable Runtime Extension
O trait Macroable permite modificação de comportamento em runtime:

```php
// Extensão dinâmica de Request class
Request::macro('isApiRequest', function() {
    return $this->wantsJson() || $this->expectsJson();
});

// Mixin de funcionalidades completas
Request::mixin(new ApiRequestMixin());
```

## Integração com Framework

### Class Integration

**Controllers Pattern**:
```php
class BaseController {
    use Macroable;           // Para extensibilidade
    use Dumpable;           // Para debugging

    // Classes derivadas herdam todas as funcionalidades
}
```

**Request Handling Pattern**:
```php
class FilterRequest {
    use Filterable;         // Normalização automática de filtros

    // Validação e normalização transparente
}
```

**Job Processing Pattern**:
```php
class BaseJob {
    use ErrorHandlingTrait;      // Handling robusto de erros
    use MemoryManagementTrait;   // Gestão de memória
    use MemoryOptimizedTrait;    // Processamento eficiente
}
```

### Dependency Injection

Os traits integram-se naturalmente com o container DI:

**Container Resolution**:
```php
// MemoryOptimizedTrait usa múltiplas estratégias de resolução
if (function_exists('app')) {
    $dbFromContainer = app('database');
}
```

**Service Integration**:

```php
// ErrorHandlingTrait resolve serviços automaticamente
$formattedText = traits-system.mdapp(ContextFormatter::class)->format($context)
              . app(JobContextFormatter::class)->format($jobContext);
```

### Event System

**ErrorHandlingTrait** integra-se com o sistema de eventos através de notificações:
```php
Notification::using(MattermostChannel::class)->debugging()->send([
    'text' => "CRITICAL JOB ERROR: $message",
    'subject' => 'Erro Crítico no Processamento de Jobs',
]);
```

## Performance e Optimização

### Trait Loading

**Autoloading Optimization**:
- Traits seguem PSR-4 sob namespace `Og\Modules\Common\Traits`
- Loading lazy através do autoloader do Composer
- Sem overhead até primeira utilização

**Memory Impact**:
- Traits não aumentam footprint de memória base
- Métodos são compilados apenas quando utilizados
- Properties são adicionadas apenas às classes que usam os traits

### Memory Management

**MemoryOptimizedTrait Strategies**:
```php
// Cálculo dinâmico de chunk size
protected function getOptimalChunkSize(int $estimatedRecordSizeBytes = 1024): int {
    $availableMemory = $this->getAvailableMemory();
    $safeMemoryLimit = $availableMemory * 0.1; // 10% da memória disponível
    return max(100, min(5000, $safeMemoryLimit / $estimatedRecordSizeBytes));
}
```

**Garbage Collection Strategies**:
```php
// Optimização automática em intervalos regulares
if ($offset % ($chunkSize * 4) === 0) {
    gc_collect_cycles();
}
```

### Caching

**Static Property Caching**:
```php
// Macroable mantém cache estático de macros
protected static array $macros = [];
```

**Memory Statistics Caching**:
```php
// MemoryManagementTrait oferece statistics em tempo real
return [
    'current_mb' => round(memory_get_usage(true) / 1024 / 1024, 2),
    'peak_mb' => round(memory_get_peak_usage(true) / 1024 / 1024, 2),
];
```

## Extensibilidade

### Custom Traits

**Guidelines para Novos Traits**:
1. **Single Responsibility**: Uma responsabilidade clara e bem definida
2. **Consistent Naming**: Seguir padrões de nomenclatura existentes
3. **Non-conflicting Methods**: Evitar overlapping com traits existentes
4. **Documentation**: Documentar propósito e integração
5. **Testing**: Testes unitários para funcionalidades core

**Template para Novos Traits**:
```php
<?php
namespace Og\Modules\Common\Traits;

trait CustomTrait {
    /**
     * Primary functionality method
     */
    protected function primaryMethod(): mixed {
        // Implementation
    }

    /**
     * Helper methods with consistent naming
     */
    private function getCustomData(): array {
        // Support functionality
    }
}
```

### Trait Inheritance

**Composition Patterns**:
```php
// Trait especializado que estende funcionalidade base
trait AdvancedMemoryTrait {
    use MemoryManagementTrait;

    // Adiciona funcionalidades específicas mantendo base
    protected function advancedMemoryAnalysis(): array {
        $baseStats = $this->getMemoryStats();
        // Extend with advanced metrics
        return array_merge($baseStats, $advancedMetrics);
    }
}
```

### Plugin System

**Macroable como Plugin Foundation**:
```php
// Sistema de plugins baseado em Macroable
class PluginSystem {
    public static function registerPlugin(string $name, object $plugin): void {
        foreach (get_class_methods($plugin) as $method) {
            static::macro($name . ucfirst($method), [$plugin, $method]);
        }
    }
}
```

## Exemplos Práticos

### Uso Básico

**Debugging com Dumpable**:
```php
class CustomerService {
    use Dumpable;

    public function processCustomer($customer) {
        $this->dumpWith('Processing customer')->dumpOnly('id', 'name');
        // Debug output mostra contexto e propriedades específicas
    }
}
```

**Filtros com Filterable**:
```php
class CustomerFilterRequest {
    use Filterable;

    // Automaticamente normaliza:
    // ['filter' => ['status' => 'active']]
    // Para:
    // ['filter' => ['status' => ['value' => 'active', 'operator' => 'equal']]]
}
```

### Casos Avançados

**Processamento de Grandes Datasets**:
```php
class ReportGenerator {
    use MemoryOptimizedTrait;
    use MemoryManagementTrait;

    public function generateLargeReport() {
        $this->configureMemoryLimits('2G');

        $stats = $this->writeDataToFile(
            $this->processRecordsInChunks('orders', ['*'], [], 1000),
            '/tmp/large_report.csv',
            'csv'
        );

        $this->logMemoryUsage('Report completed');
    }
}
```

**Job com Error Handling Robusto**:
```php
class CriticalDataJob extends Job {
    use ErrorHandlingTrait;
    use MemoryManagementTrait;

    public function handle(): void {
        $this->withLenientErrorHandling(function () {
            $this->processInBatches($this->getData(), function($item) {
                // Erros não-críticos são loggedmas não param o job
                $this->processItem($item);
            }, 100, 'Critical Data Processing');
        });
    }
}
```

### Integration Patterns

**Multi-Trait Composition**:
```php
class AdvancedApiController extends BaseController {
    use Macroable;              // Extensibilidade dinâmica
    use Dumpable;               // Debugging capabilities
    use MemoryManagementTrait;  // Resource monitoring

    public function handleLargeDataRequest() {
        $this->configureMemoryLimits('1G');

        $this->dump('Processing large request');

        // Macros podem ser adicionados dinamicamente
        if (static::hasMacro('customProcessing')) {
            return $this->customProcessing();
        }

        return $this->standardProcessing();
    }
}
```

**Enum with Full Capabilities**:
```php
enum OrderStatus: string {
    use EnumCaseToArray;
    use Dumpable;  // Para debugging durante desenvolvimento

    case PENDING = 'P';
    case SHIPPED = 'S';
    case DELIVERED = 'D';

    public function label(): string {
        return match($this) {
            self::PENDING => 'Pendente',
            self::SHIPPED => 'Enviado',
            self::DELIVERED => 'Entregue',
        };
    }

    // Métodos do EnumCaseToArray ficam disponíveis:
    // OrderStatus::toArray(), OrderStatus::labels(), etc.
}
```

## Conclusão

O sistema de traits do framework OfficeGest representa uma implementação sofisticada do padrão Mixin, fornecendo funcionalidades transversais reutilizáveis que são essenciais para a arquitectura modular do sistema. Com 8 traits especializados, cada um com responsabilidades bem definidas, o sistema oferece:

- **Composição Flexível**: Permite combinações de funcionalidades sem acoplamento rígido
- **Performance Optimizada**: Traits especializados para gestão de memória e processamento eficiente
- **Debugging Avançado**: Sistema completo de debugging e error handling
- **Extensibilidade**: Capacidades dinâmicas de extensão através do Macroable trait
- **Integração Harmoniosa**: Designed para trabalhar em conjunto sem conflitos

Este sistema é fundamental para manter a escalabilidade e manutenibilidade do OfficeGest, permitindo que funcionalidades complexas sejam reutilizadas de forma consistente através de toda a aplicação.
