# Sistema de Utilities - Framework OfficeGest

## Visão Geral do Sistema de Utilities

O sistema de utilities do framework OfficeGest implementa uma arquitectura completa de classes auxiliares especializadas que fornecem funcionalidades transversais essenciais para o funcionamento do sistema. Este conjunto de 20+ classes utilitárias está organizado hierarquicamente em 8 categorias especializadas, seguindo padrões de design modernos e oferecendo desde ferramentas de performance até extensões de frameworks externos.

O sistema está estruturado para suportar tanto funcionalidades de desenvolvimento (debugging, profiling, testing) quanto operações de produção (configuração, URL handling, processamento de dados), mantendo uma separação clara entre responsabilidades e permitindo uso modular das funcionalidades.

## Arquitectura e Padrões de Design

### Utility Pattern
As classes implementam o padrão **Utility** através de métodos estáticos e instâncias singleton, fornecendo funcionalidades auxiliares sem necessidade de instanciação complexa. Cada classe tem uma responsabilidade específica e bem definida.

### Facade Pattern
O sistema implementa extensivamente o padrão **Facade** através da classe base `Facade`, que simplifica o acesso a serviços complexos do framework e integra-se nativamente com o container de dependency injection.

### Factory Pattern
Muitas utilities utilizam o padrão **Factory** para criação de objectos, especialmente visível no sistema de Faker e na construção de URLs.

### Chain of Responsibility Pattern
A classe `ProcessChain` implementa explicitamente o padrão **Chain of Responsibility**, permitindo processamento sequencial de dados através de múltiplas etapas.

### Strategy Pattern
O sistema de configuração e URL handling utilizam **Strategy Pattern** para diferentes contextos e ambientes.

### Dependency Injection Integration
Integração nativa com o container DI do framework através de `Container::getInstance()` e resolução automática de dependências.

## Estrutura e Componentes

### Utilities de Performance (Performance Utilities)

#### Benchmark
**Localização**: `Modules/Common/Utils/Benchmark.php`

Sistema avançado de profiling de performance especificamente concebido para aplicações legacy e CodeIgniter 3. Implementa **Observer Pattern** para monitoring contínuo.

**Funcionalidades Principais**:
- **Precision Timing**: `measure()`, `value()` - medição de execução com precisão de microssegundos
- **Query Integration**: `trackQuery()`, `timeQuery()` - integração com CodeIgniter DB layer
- **Memory Monitoring**: `getMemoryStats()` - tracking completo de uso de memória
- **Batch Processing**: `processInBatches()` - processamento optimizado de grandes datasets
- **Automated Reporting**: `getReport()`, `display()` - relatórios detalhados de performance

**Configuração Avançada**:
```php
Benchmark::init([
    'enabled' => true,
    'log_threshold' => 1.0,      // Log operações > 1ms
    'query_threshold' => 10.0,   // Log queries > 10ms
    'track_memory' => true,
    'track_queries' => true,
]);
```

**Padrões de Uso**:
```php
// Medição simples
$result = Benchmark::value(function() {
    return expensiveOperation();
});

// Tracking automático de queries CodeIgniter
$data = Benchmark::timeQuery($db, function($db) {
    return $db->get('users')->result();
});

// Comparação de performance
$comparison = Benchmark::compare([
    $method1, $method2, $method3
], 10); // 10 iterations each
```

**Integração com Legacy Systems**:
O Benchmark integra-se nativamente com CodeIgniter 3 através de `trackCIQuery()` e suporte para `$db->last_query()`.

#### SqlProfiler
**Localização**: `Modules/Common/Utils/SqlProfiler.php`

Profiler SQL especializado para aplicações que utilizam OGDB_mysql_driver. Implementa **Strategy Pattern** para diferentes tipos de análise.

**Funcionalidades Principais**:
- **Query Timing**: `startQuery()`, `endQuery()` - timing preciso de queries individuais
- **Memory Tracking**: Monitoring de uso de memória por query
- **Slow Query Detection**: Detecção automática e logging de queries lentas
- **EXPLAIN Integration**: `explainQuery()` - análise automática de queries SELECT
- **Backtrace Support**: Rastreamento de origem de queries

**Sistema de Thresholds**:
```php
SqlProfiler::init([
    'log_threshold_ms' => 100,        // Log queries > 100ms
    'explain_queries' => true,        // Auto EXPLAIN para queries lentas
    'backtrace_enabled' => true,      // Capture call stack
    'keep_max_queries' => 100,        // Limite de queries em memória
]);
```

**Análise Avançada**:
```php
// Análise por tipo de query
$selectQueries = SqlProfiler::getQueriesByType('SELECT');
$slowestQueries = SqlProfiler::getSlowestQueries(10);

// Relatório completo
$report = SqlProfiler::getReport();
// Inclui: total time, memory usage, query types breakdown, slowest queries
```

### Framework Extensions

#### Facade
**Localização**: `Modules/Common/Utils/Facade.php`

Implementação completa do padrão **Facade** com suporte a testing através de Mockery integration.

**Funcionalidades Principais**:
- **Container Integration**: Resolução automática via `Container::getInstance()`
- **Static Proxy**: `__callStatic()` - proxy para métodos de instância
- **Test Support**: `spy()`, `mock()`, `partialMock()` - integração com Mockery
- **Instance Caching**: Cache automático de instâncias resolvidas
- **Instance Swapping**: `swap()` - substituição de instâncias para testing

**Padrão de Implementação**:
```php
class ConfigFacade extends Facade {
    protected static function getFacadeAccessor(): string {
        return 'config';
    }
}

// Uso transparente
ConfigFacade::get('app.name'); // Resolve via container
```

**Testing Integration**:
```php
// Spy pattern para testing
$spy = ConfigFacade::spy();
$spy->shouldReceive('get')->with('app.name')->andReturn('TestApp');
```

#### Carbon
**Localização**: `Modules/Common/Utils/Carbon.php`

Extensão do Carbon com debugging capabilities e localização automática.

**Funcionalidades Principais**:
- **Automatic Localization**: Detecção automática de língua via `app('language')`
- **Debug Methods**: `dump()`, `dd()` - debugging chainable
- **Extended Construction**: Suporte para todos os tipos de input do Carbon base

**Localização Automática**:
```php
$date = new Carbon('2025-09-16');
// Automaticamente definido para pt_PT baseado na detecção de língua
echo $date->diffForHumans(); // "há 2 dias" em português
```

#### Collection
**Localização**: `Modules/Common/Utils/Collection.php`

Wrapper simplificado para Laravel Collections com method delegation.

**Funcionalidades Principais**:
- **Laravel Integration**: Utiliza `collect()` internamente
- **Method Proxy**: `__call()` - delegation transparente para LaravelCollection
- **Familiar API**: Interface idêntica às Laravel Collections

### Processing Utilities

#### ProcessChain
**Localização**: `Modules/Common/Utils/ProcessChain.php`

Implementação do padrão **Chain of Responsibility** para processamento sequencial de dados.

**Funcionalidades Principais**:
- **Fluent Interface**: `step()` - adição fluente de etapas
- **Promise-like API**: `then()`, `catch()` - handling de sucesso/falha
- **Error Propagation**: Propagação automática de erros na chain
- **Functional Composition**: Composição funcional de transformações

**Padrões de Uso**:
```php
$result = (new ProcessChain())
    ->step(fn($data) => validateData($data))
    ->step(fn($data) => transformData($data))
    ->step(fn($data) => persistData($data))
    ->then(
        fn($result) => returnSuccess($result),
        $initialData
    )
    ->catch(fn($error) => handleError($error));
```

### Configuration Utilities

#### Config
**Localização**: `Modules/Common/Utils/Config/Config.php`

Sistema de configuração com suporte a cache e legacy integration.

**Funcionalidades Principais**:
- **Dot Notation**: `get('database.host')` - acesso hierárquico a configurações
- **Cache System**: `cache()`, `clearCache()` - performance optimizations
- **Legacy Support**: `loadLegacyConfig()` - integração com configurações legacy
- **Hot Reloading**: `reload()` - recarregamento em runtime

**Cache Strategy**:
```php
// Primeira carga - lê ficheiros PHP e cria cache
$config = new Config('/path/to/config', '/path/to/cache/config.php');

// Cargas subsequentes - utiliza cache serializado
$value = $config->get('app.debug', false);
```

#### ConstantHandler
**Localização**: `Modules/Common/Utils/ConstantHandler.php`

Manipulação dinâmica de constantes PHP com suporte para `define()` e `const`.

**Funcionalidades Principais**:
- **Dynamic Constant Management**: `updateOrCreateConstant()` - criação/actualização dinâmica
- **Multiple Syntax Support**: Suporta tanto `define()` quanto `const`
- **Value Formatting**: `formatValue()` - formatting automático baseado no tipo
- **Existence Checking**: `constantExists()` - verificação de existência

**Tipos Suportados**:
```php
ConstantHandler::updateOrCreateConstant('config.php', 'APP_DEBUG', true);        // boolean
ConstantHandler::updateOrCreateConstant('config.php', 'APP_NAME', 'OfficeGest'); // string
ConstantHandler::updateOrCreateConstant('config.php', 'FEATURES', ['api', 'web']); // array
```

### URL and Routing Utilities

#### URL
**Localização**: `Modules/Common/Utils/URL/URL.php`

Constructor fluente de URLs com suporte completo para componentes URL.

**Funcionalidades Principais**:
- **Fluent Interface**: Method chaining para construção de URLs
- **Component Management**: Scheme, host, port, path, query, fragment
- **Query Parameter Handling**: Manipulação avançada de query strings
- **Route Integration**: Integração com sistema de routing
- **Enum Support**: Suporte nativo para BackedEnum em parâmetros

**Builder Pattern**:
```php
$url = (new URL())
    ->scheme('https')
    ->host('api.officegest.com')
    ->path('/v1/users')
    ->query(['active' => true, 'limit' => 50])
    ->fragment('results');

echo $url; // https://api.officegest.com/v1/users?active=1&limit=50#results
```

#### RouteUrlGenerator & CurrentRouteInfo
Sistema completo de geração de URLs baseado em routes com context awareness.

### Development Tools

#### Faker/VehicleProvider
**Localização**: `Modules/Common/Utils/Faker/VehicleProvider.php`

Provider personalizado para FakerPHP com dados portugueses realistas.

**Funcionalidades Principais**:
- **Portuguese Vehicle Data**: Marcas, modelos, matrículas portuguesas
- **Realistic Data**: Dados baseados no mercado automóvel português
- **Extended Faker**: Integração nativa com FakerPHP ecosystem

**Dados Disponíveis**:
```php
// Marcas mais populares em Portugal
$brands = ['Renault', 'Peugeot', 'Volkswagen', 'Opel', 'Ford', 'Citroën'];

// Modelos por marca
$models = [
    'Renault' => ['Clio', 'Megane', 'Captur', 'Scenic'],
    'Peugeot' => ['208', '308', '3008', '5008'],
    // ... dados completos para 21 marcas
];
```

#### Rector Rules
**Localização**: `Modules/Common/Utils/Rector/`

Conjunto de 10 rules customizadas para modernização automática de código.

**Rules Implementadas**:
- **NullSafeRoundFunctionsRector**: Adiciona null coalescing a funções matemáticas
- **NullSafeStringFunctionsRector**: Proteção null-safe para funções de string
- **NullSafeArrayFunctionsRector**: Proteção para funções de array
- **CustomUtf8DecodeEncodeRector**: Modernização de encoding functions
- **FileExistsNullSafeRector**: Proteção null-safe para file operations
- **ReplaceShortEchoTagWithEchoRector**: Modernização de echo tags

**Exemplo de Transformação**:
```php
// Antes
$result = round($number);

// Depois (NullSafeRoundFunctionsRector)
$result = round($number ?? 0);
```

### Utility Traits

#### Macroable
**Localização**: `Modules/Common/Utils/Traits/Macroable.php`

Trait para extensão dinâmica de classes (idêntico ao sistema principal de Traits).

#### InteractsWithTime & EagerLoader
**Localização**: `Modules/Common/Utils/Traits/`

Traits auxiliares para interacção com tempo e loading optimizado.

### HTML and String Utilities

#### HtmlString
**Localização**: `Modules/Common/Utils/HtmlString.php`

Wrapper para strings HTML com type safety.

**Funcionalidades Principais**:
- **Type Safety**: Implementa `Stringable` interface
- **HTML Semantics**: `toHtml()` - conversão semântica
- **Empty Checking**: `isEmpty()`, `isNotEmpty()` - validação de conteúdo
- **Safe Casting**: Conversão segura para string

### Maps and Geospatial

#### Map
**Localização**: `Modules/Common/Utils/Maps/Map.php`

Utilidades básicas para manipulação de dados cartográficos.

## Funcionalidades Avançadas

### Performance Integration

**Benchmark + SqlProfiler Integration**:
```php
// Inicialização combinada para profiling completo
Benchmark::init(['track_queries' => true]);
SqlProfiler::init(['explain_queries' => true]);

// Tracking automático de performance SQL
$data = Benchmark::timeQuery($db, function($db) {
    return $db->select('*')->from('large_table')->get()->result();
});
```

### Development Workflow Integration

**Rector + Config Integration**:
```php
// Aplicação automática de rules de modernização
$configUpdated = ConstantHandler::updateOrCreateConstant(
    'rector.php',
    'RECTOR_RULES',
    ['NullSafeRoundFunctionsRector', 'NullSafeStringFunctionsRector']
);
```

### Testing Support

**Facade Testing Pattern**:
```php
// Setup de mocks para testing
class ConfigFacadeTest extends TestCase {
    public function testConfigRetrieval() {
        $mock = ConfigFacade::mock();
        $mock->shouldReceive('get')->with('app.name')->andReturn('TestApp');

        $result = ConfigFacade::get('app.name');
        $this->assertEquals('TestApp', $result);
    }
}
```

### Chain Processing

**Complex Pipeline Example**:
```php
$processed = (new ProcessChain())
    ->step(fn($data) => $this->validateInput($data))
    ->step(fn($data) => $this->enrichWithExternalData($data))
    ->step(fn($data) => $this->transform($data))
    ->step(fn($data) => $this->persist($data))
    ->then(
        fn($result) => $this->returnSuccessResponse($result),
        $requestData
    )
    ->catch(fn($error) => $this->returnErrorResponse($error));
```

## Integração com Framework

### Container Integration

**Facade Registration**:
```php
// Container binding para facades
Container::getInstance()->bind('config', function() {
    return new Config('/path/to/config', '/cache/config.php');
});

// Uso automático através de facades
ConfigFacade::get('database.host'); // Resolve automaticamente
```

### Legacy System Integration

**CodeIgniter Integration**:
```php
// Integração com CI3 database layer
class LegacyController extends CI_Controller {
    public function performanceOptimizedMethod() {
        Benchmark::init();

        $result = Benchmark::timeQuery($this->db, function($db) {
            return $db->get('users')->result();
        });

        $report = Benchmark::getReport();
        // Performance data disponível para optimização
    }
}
```

### Modern Framework Extensions

**Carbon Localization**:
```php
// Detecção automática de locale
$date = new \Og\Modules\Common\Utils\Carbon('2025-09-16');
// Automaticamente configura pt_PT baseado em app('language')->detectLanguage()
```

## Performance e Optimização

### Memory Management

**Benchmark Memory Strategies**:
```php
// Configuração para aplicações memory-intensive
Benchmark::init([
    'track_memory' => true,
    'log_threshold' => 5.0,  // Apenas operações > 5ms
    'query_threshold' => 50.0 // Apenas queries > 50ms
]);
```

### Caching Strategies

**Config System Caching**:
```php
// Cache automático para performance
$config = new Config('/config', '/cache/config.php');
// Primeira carga: lê PHP files e cria cache
// Cargas subsequentes: usa cache serializado

// Force rebuild quando necessário
$config->reload(true); // ignora cache existente
```

### SQL Query Optimization

**SqlProfiler Performance Analysis**:
```php
SqlProfiler::init([
    'explain_queries' => true,    // EXPLAIN automático
    'backtrace_enabled' => true,  // Rastreamento de origem
    'keep_max_queries' => 50,     // Limite memoria usage
]);

// Análise de performance queries
$slowQueries = SqlProfiler::getSlowestQueries(10);
foreach ($slowQueries as $query) {
    if ($query['duration_ms'] > 100) {
        // Optimização necessária
        analyzeQuery($query['sql'], $query['explain']);
    }
}
```

## Extensibilidade

### Custom Rector Rules

**Template para Nova Rule**:
```php
namespace Og\Modules\Common\Utils\Rector;

use Rector\Rector\AbstractRector;
use PhpParser\Node;

class CustomModernizationRector extends AbstractRector {
    public function getNodeTypes(): array {
        return [/* Node types to process */];
    }

    public function refactor(Node $node): ?Node {
        // Custom transformation logic
        return $transformedNode;
    }

    public function getRuleDefinition(): RuleDefinition {
        return new RuleDefinition(
            'Description of transformation',
            [new CodeSample('// Before', '// After')]
        );
    }
}
```

### Custom Faker Providers

**Extension Pattern**:
```php
namespace Og\Modules\Common\Utils\Faker;

class CompanyProvider extends Base {
    protected static array $portugueseCompanies = [
        'EDP', 'Galp', 'CTT', 'NOS', 'MEO', 'Sonae'
    ];

    public function portugueseCompany(): string {
        return static::randomElement(static::$portugueseCompanies);
    }
}
```

### Custom Facades

**Implementation Pattern**:
```php
class EmailFacade extends Facade {
    protected static function getFacadeAccessor(): string {
        return 'email.service';
    }
}

// Container registration
Container::getInstance()->bind('email.service', function() {
    return new EmailService(config('mail'));
});
```

## Exemplos Práticos

### Performance Monitoring

**Complete Application Profiling**:
```php
class PerformanceOptimizedController {
    public function processLargeDataset() {
        // Initialize profiling
        Benchmark::init(['track_queries' => true]);
        SqlProfiler::init(['explain_queries' => true]);

        Benchmark::mark('data_processing_start');

        $result = (new ProcessChain())
            ->step(fn($data) => $this->validateData($data))
            ->step(fn($data) => Benchmark::timeQuery($this->db,
                fn($db) => $this->loadRelatedData($db, $data)))
            ->step(fn($data) => $this->transformData($data))
            ->then(
                fn($result) => $this->returnResponse($result),
                $this->requestData
            )
            ->catch(fn($error) => $this->handleError($error));

        Benchmark::mark('data_processing_end');

        // Performance analysis
        $benchmarkReport = Benchmark::getReport();
        $sqlReport = SqlProfiler::getReport();

        if ($benchmarkReport['total_time_ms'] > 1000) {
            $this->alertSlowOperation($benchmarkReport, $sqlReport);
        }

        return $result;
    }
}
```

### Configuration Management

**Dynamic Configuration System**:
```php
class ConfigurationManager {
    private Config $config;

    public function __construct() {
        $this->config = new Config(
            rootPath('config'),
            cachePath('config/compiled.php')
        );
    }

    public function updateFeatureFlag(string $feature, bool $enabled): void {
        ConstantHandler::updateOrCreateConstant(
            rootPath('config/features.php'),
            strtoupper($feature) . '_ENABLED',
            $enabled
        );

        // Clear config cache para recarregar
        $this->config->clearCache();
        $this->config->reload();
    }

    public function getOptimizedConfig(): array {
        // Cache hit optimization
        if ($this->config->isCached()) {
            return $this->config->all();
        }

        // Cache miss - load and cache
        $configs = $this->config->all();
        $this->config->cache();

        return $configs;
    }
}
```

### URL Generation System

**Advanced URL Building**:
```php
class ApiUrlBuilder {
    public function buildApiEndpoint(string $version, string $resource, array $filters = []): string {
        $url = (new URL())
            ->scheme(config('api.scheme', 'https'))
            ->host(config('api.host'))
            ->path("/api/{$version}/{$resource}");

        // Add filtering parameters
        foreach ($filters as $key => $value) {
            if ($value instanceof BackedEnum) {
                $url->query([$key => $value->value]);
            } else {
                $url->query([$key => $value]);
            }
        }

        // Add pagination if needed
        if (request('page')) {
            $url->query(['page' => request('page')]);
        }

        return $url->toString();
    }
}
```

### Testing Integration

**Complete Testing Setup**:
```php
class UtilityIntegrationTest extends TestCase {
    public function testPerformanceMonitoring(): void {
        // Mock config for testing
        $configMock = ConfigFacade::mock();
        $configMock->shouldReceive('get')
               ->with('benchmark.enabled')
               ->andReturn(true);

        // Test benchmark functionality
        Benchmark::init(['enabled' => true]);

        $result = Benchmark::value(function() {
            return array_sum(range(1, 1000));
        });

        $this->assertIsArray($result);
        $this->assertEquals(500500, $result[0]); // Sum result
        $this->assertIsFloat($result[1]);        // Execution time

        // Verify performance data
        $report = Benchmark::getReport();
        $this->assertArrayHasKey('total_time_ms', $report);
        $this->assertArrayHasKey('memory', $report);
    }

    public function testProcessChainErrorHandling(): void {
        $errorMessage = '';

        $result = (new ProcessChain())
            ->step(fn($data) => $data * 2)
            ->step(fn($data) => throw new Exception('Test error'))
            ->step(fn($data) => $data * 3) // Should not execute
            ->then(
                fn($result) => $result,
                5
            )
            ->catch(fn($error) => $errorMessage = $error->getMessage());

        $this->assertEquals('Test error', $errorMessage);
    }
}
```

## Conclusão

O sistema de utilities do framework OfficeGest representa uma implementação abrangente e sofisticada de classes auxiliares que cobrem todas as necessidades de uma aplicação enterprise moderna. Com 20+ classes organizadas em 8 categorias especializadas, o sistema oferece:

- **Performance Optimization**: Tools avançados de profiling e benchmarking
- **Development Productivity**: Rector rules, Faker providers, debugging utilities
- **Framework Integration**: Facades, Carbon extensions, Collection wrappers
- **Configuration Management**: Sistema robusto de configuração com cache
- **URL Management**: Constructor fluente e route integration
- **Processing Utilities**: Chain of responsibility e pipeline processing
- **Legacy Support**: Integração seamless com sistemas CodeIgniter 3
- **Modern Patterns**: Implementação de padrões de design contemporary

Este sistema é essencial para manter a produtividade de desenvolvimento, performance de produção e capacidade de manutenção do OfficeGest, fornecendo uma base sólida de utilities que podem ser utilizadas consistentemente através de toda a aplicação. A arquitectura modular permite uso selectivo das funcionalidades, enquanto a integração com o container DI garante acesso transparente e testable a todos os serviços.