@extends('layouts.docs')

@section('title', 'Autenticação - OG Framework')

@section('body')
    <div class="relative min-h-screen bg-transparent px-4 pb-12 pt-6 sm:px-6 lg:px-8">
        <div class="floating-blur blue -left-10 -top-10 hidden lg:block"></div>
        <div class="floating-blur amber right-10 top-32 hidden lg:block"></div>

        <div class="mx-auto flex w-full max-w-7xl gap-8 bg-transparent">
            {{-- Left Sidebar: Navigation --}}
            @include('docs.partials.sidebar')

            {{-- Main Content --}}
            <div class="flex-1 min-w-0">
                <div class="relative overflow-hidden rounded-3xl border border-white/50 bg-white/85 p-8 shadow-xl shadow-primary/20 backdrop-blur dark:border-zinc-800 dark:bg-zinc-900/80">

                    {{-- Page Header --}}
                    <div class="mb-8">
                        <a href="{{ route('docs.index') }}" class="inline-flex items-center gap-1 text-sm text-primary hover:underline mb-4">
                            <flux:icon.arrow-left class="size-4" />
                            Voltar para Documentação
                        </a>
                        <p class="text-xs uppercase tracking-[0.3em] text-primary dark:text-blue-200">Segurança</p>
                        <h1 class="font-display text-4xl font-bold text-zinc-900 dark:text-white sm:text-5xl">Autenticação & Autorização</h1>
                        <p class="mt-3 text-base text-zinc-700 dark:text-zinc-200 max-w-3xl">
                            O Auth System é responsável por verificar identidade (autenticação), permissões (autorização), e manter o estado do utilizador — incluindo suporte a impersonation para debugging.
                        </p>
                    </div>

                    <div class="space-y-10">

                        {{-- Concepts --}}
                        <section id="conceitos" class="scroll-mt-28 space-y-4">
                            <div class="prose prose-zinc dark:prose-invert max-w-none">
                                <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                    <flux:icon.light-bulb class="size-6 text-amber-500" />
                                    Conceitos Chave
                                </h2>
                            </div>

                            <div class="overflow-x-auto">
                                <table class="w-full text-sm text-left">
                                    <thead class="text-xs uppercase bg-zinc-100 dark:bg-zinc-800">
                                        <tr>
                                            <th class="px-4 py-3">Conceito</th>
                                            <th class="px-4 py-3">Pergunta</th>
                                            <th class="px-4 py-3">Descrição</th>
                                        </tr>
                                    </thead>
                                    <tbody class="divide-y divide-zinc-200 dark:divide-zinc-700">
                                        <tr class="bg-white dark:bg-zinc-900">
                                            <td class="px-4 py-3 font-medium text-blue-600">Autenticação</td>
                                            <td class="px-4 py-3">"Quem é você?"</td>
                                            <td class="px-4 py-3">Verificar identidade do utilizador</td>
                                        </tr>
                                        <tr class="bg-zinc-50 dark:bg-zinc-800/50">
                                            <td class="px-4 py-3 font-medium text-green-600">Autorização</td>
                                            <td class="px-4 py-3">"O que pode fazer?"</td>
                                            <td class="px-4 py-3">Verificar permissões para ações</td>
                                        </tr>
                                        <tr class="bg-white dark:bg-zinc-900">
                                            <td class="px-4 py-3 font-medium text-purple-600">Permissão</td>
                                            <td class="px-4 py-3">—</td>
                                            <td class="px-4 py-3">Direito específico (ex: <code>invoices:create</code>)</td>
                                        </tr>
                                        <tr class="bg-zinc-50 dark:bg-zinc-800/50">
                                            <td class="px-4 py-3 font-medium text-amber-600">Policy</td>
                                            <td class="px-4 py-3">—</td>
                                            <td class="px-4 py-3">Classe que define regras de autorização</td>
                                        </tr>
                                        <tr class="bg-white dark:bg-zinc-900">
                                            <td class="px-4 py-3 font-medium text-red-600">Impersonation</td>
                                            <td class="px-4 py-3">—</td>
                                            <td class="px-4 py-3">Admin atuar como outro utilizador</td>
                                        </tr>
                                    </tbody>
                                </table>
                            </div>
                        </section>

                        {{-- Quick Start --}}
                        <section id="quick-start" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <div class="prose prose-zinc dark:prose-invert max-w-none">
                                <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                    <flux:icon.rocket-launch class="size-6 text-green-500" />
                                    Quick Start
                                </h2>
                            </div>

                            <div class="rounded-xl bg-zinc-950 p-4 text-sm overflow-x-auto">
                                <pre class="language-php"><code>// 1. OBTER o AuthManager
$auth = app('auth');

// 2. VERIFICAR se utilizador está logado
if ($auth->check()) {
    $user = $auth->user();
    echo "Olá, " . $user->getUsername();
}

// 3. VERIFICAR uma permissão
if ($auth->can('invoices:create')) {
    // Utilizador pode criar faturas
}

// 4. VERIFICAR múltiplas permissões
if ($auth->canAny(['sales:view', 'sales:edit'])) {
    // Tem pelo menos uma das permissões
}

// 5. OBTER ID do utilizador
$userId = $auth->id();</code></pre>
                            </div>

                            {{-- Permission Pattern --}}
                            <div class="feature-card">
                                <p class="font-semibold text-zinc-900 dark:text-white mb-2">Padrão de Permissões</p>
                                <p class="text-sm text-zinc-700 dark:text-zinc-300 mb-3">O formato segue <code>módulo:ação</code>:</p>
                                <div class="grid gap-2 md:grid-cols-3 text-xs font-mono">
                                    <div class="bg-zinc-100 dark:bg-zinc-800 px-2 py-1 rounded"><code>entidades:clientes</code></div>
                                    <div class="bg-zinc-100 dark:bg-zinc-800 px-2 py-1 rounded"><code>stocks:artigos</code></div>
                                    <div class="bg-zinc-100 dark:bg-zinc-800 px-2 py-1 rounded"><code>vendas:facturas</code></div>
                                    <div class="bg-zinc-100 dark:bg-zinc-800 px-2 py-1 rounded"><code>invoices:create</code></div>
                                    <div class="bg-zinc-100 dark:bg-zinc-800 px-2 py-1 rounded"><code>invoices:admin</code></div>
                                    <div class="bg-zinc-100 dark:bg-zinc-800 px-2 py-1 rounded"><code>crm:edit</code></div>
                                </div>
                            </div>
                        </section>

                        {{-- AuthManager --}}
                        <section id="auth-manager" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <div class="prose prose-zinc dark:prose-invert max-w-none">
                                <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                    <flux:icon.user-circle class="size-6 text-blue-500" />
                                    AuthManager
                                </h2>
                                <p class="text-zinc-700 dark:text-zinc-300">
                                    O <code>AuthManager</code> é o ponto central para toda a lógica de autenticação:
                                </p>
                            </div>

                            <div class="grid gap-4 md:grid-cols-2">
                                {{-- State Verification --}}
                                <div class="feature-card">
                                    <p class="font-mono text-sm font-semibold text-blue-600 dark:text-blue-400 mb-2">Verificação de Estado</p>
                                    <div class="rounded-lg bg-zinc-950 p-3 text-xs overflow-x-auto">
                                        <pre class="language-php"><code>$auth = app('auth');

// Está autenticado?
$auth->check(): bool

// Obter utilizador atual
$auth->user(): ?User

// Obter ID do utilizador
$auth->id(): ?int

// Está ativo?
$auth->isActive(): bool</code></pre>
                                    </div>
                                </div>

                                {{-- Permission Verification --}}
                                <div class="feature-card">
                                    <p class="font-mono text-sm font-semibold text-green-600 dark:text-green-400 mb-2">Verificação de Permissões</p>
                                    <div class="rounded-lg bg-zinc-950 p-3 text-xs overflow-x-auto">
                                        <pre class="language-php"><code>// Verifica UMA permissão
$auth->can('stocks:artigos'): bool

// Verifica se tem ALGUMA
$auth->canAny(['view', 'edit']): bool

// Verifica se tem TODAS
$auth->canAll(['view', 'edit']): bool</code></pre>
                                    </div>
                                </div>

                                {{-- User Management --}}
                                <div class="feature-card">
                                    <p class="font-mono text-sm font-semibold text-purple-600 dark:text-purple-400 mb-2">Gestão de Utilizador</p>
                                    <div class="rounded-lg bg-zinc-950 p-3 text-xs overflow-x-auto">
                                        <pre class="language-php"><code>// Atualiza utilizador (sync legacy)
$auth->setUser(User $user): void

// Dados como array
$auth->getUserData(): array</code></pre>
                                    </div>
                                </div>

                                {{-- Impersonation --}}
                                <div class="feature-card">
                                    <p class="font-mono text-sm font-semibold text-amber-600 dark:text-amber-400 mb-2">Impersonation</p>
                                    <div class="rounded-lg bg-zinc-950 p-3 text-xs overflow-x-auto">
                                        <pre class="language-php"><code>// Em modo impersonation?
$auth->isImpersonating(): bool

// É admin developer?
$auth->isDeveloperAdmin(): bool

// ID do admin original
$auth->impersonatorId(): ?int</code></pre>
                                    </div>
                                </div>
                            </div>
                        </section>

                        {{-- Policies --}}
                        <section id="policies" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <div class="prose prose-zinc dark:prose-invert max-w-none">
                                <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                    <flux:icon.shield-check class="size-6 text-green-500" />
                                    Policies (Autorização)
                                </h2>
                                <p class="text-zinc-700 dark:text-zinc-300">
                                    Policies encapsulam regras de autorização específicas para um recurso. Em vez de espalhar <code>if</code> statements pelo código, centralize a lógica:
                                </p>
                            </div>

                            <div class="rounded-xl bg-zinc-950 p-4 text-sm overflow-x-auto">
                                <pre class="language-php"><code>use Og\Modules\Common\Auth\Authorization;

class InvoicePolicy extends Authorization
{
    // Pode visualizar a fatura?
    public function view($invoice): bool
    {
        return $this->user->checkPriv('invoices:view');
    }

    // Pode editar a fatura?
    public function edit($invoice): bool
    {
        // Criador OU admin
        return $invoice->created_by === $this->user->getId()
            || $this->user->checkPriv('invoices:admin');
    }

    // Pode eliminar a fatura?
    public function delete($invoice): bool
    {
        return $this->user->checkPriv('invoices:admin');
    }
}</code></pre>
                            </div>

                            <div class="feature-card">
                                <p class="font-semibold text-zinc-900 dark:text-white mb-2">Usando uma Policy</p>
                                <div class="rounded-lg bg-zinc-950 p-3 text-xs overflow-x-auto">
                                    <pre class="language-php"><code>$policy = new InvoicePolicy($user, $app);

// Verificar manualmente
if ($policy->edit($invoice)) {
    // Pode editar
}

// Ou usar authorize() — lança exceção se falhar
try {
    $policy->authorize('edit', $invoice);
} catch (AuthorizationException $e) {
    return response()->error($e->getMessage(), 403);
}</code></pre>
                                </div>
                            </div>
                        </section>

                        {{-- Access Response --}}
                        <section id="access-response" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <div class="prose prose-zinc dark:prose-invert max-w-none">
                                <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                    <flux:icon.check-circle class="size-6 text-purple-500" />
                                    Access Response
                                </h2>
                                <p class="text-zinc-700 dark:text-zinc-300">
                                    A classe <code>Response</code> encapsula decisões de acesso com mensagens e códigos HTTP personalizados:
                                </p>
                            </div>

                            <div class="grid gap-4 md:grid-cols-2">
                                <div class="feature-card border-l-4 border-green-500">
                                    <p class="font-mono text-sm font-semibold text-green-600 dark:text-green-400 mb-2">Permitir</p>
                                    <div class="rounded-lg bg-zinc-950 p-3 text-xs overflow-x-auto">
                                        <pre class="language-php"><code>Response::allow();
Response::allow('OK', 'ACCESS_OK');</code></pre>
                                    </div>
                                </div>
                                <div class="feature-card border-l-4 border-red-500">
                                    <p class="font-mono text-sm font-semibold text-red-600 dark:text-red-400 mb-2">Negar</p>
                                    <div class="rounded-lg bg-zinc-950 p-3 text-xs overflow-x-auto">
                                        <pre class="language-php"><code>Response::deny('Sem permissão');
Response::denyWithStatus(403, 'Proibido');
Response::denyAsNotFound('Não encontrado');</code></pre>
                                    </div>
                                </div>
                            </div>

                            <div class="rounded-xl bg-zinc-950 p-4 text-sm overflow-x-auto">
                                <pre class="language-php"><code>class DocumentPolicy extends Authorization
{
    public function access($document): Response
    {
        if (!$this->user->checkPriv('documents:view')) {
            return Response::deny('Sem permissão');
        }

        if ($document->is_confidential && !$this->user->checkPriv('documents:confidential')) {
            return Response::denyWithStatus(403, 'Documento confidencial');
        }

        if ($document->status === 'deleted') {
            return Response::denyAsNotFound('Documento não encontrado');
        }

        return Response::allow();
    }
}</code></pre>
                            </div>
                        </section>

                        {{-- Impersonation --}}
                        <section id="impersonation" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <div class="prose prose-zinc dark:prose-invert max-w-none">
                                <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                    <flux:icon.users class="size-6 text-amber-500" />
                                    Impersonation
                                </h2>
                                <p class="text-zinc-700 dark:text-zinc-300">
                                    Permite que um administrador "atue como" outro utilizador para debugging ou suporte:
                                </p>
                            </div>

                            <div class="rounded-xl bg-gradient-to-br from-zinc-900 to-zinc-950 p-6 overflow-x-auto">
                                <svg viewBox="0 0 600 300" class="w-full max-w-2xl mx-auto" xmlns="http://www.w3.org/2000/svg">
                                    <defs>
                                        <linearGradient id="imperGradient" x1="0%" y1="0%" x2="100%" y2="0%">
                                            <stop offset="0%" style="stop-color:#f59e0b;stop-opacity:1" />
                                            <stop offset="100%" style="stop-color:#3b82f6;stop-opacity:1" />
                                        </linearGradient>
                                    </defs>

                                    <!-- Admin Entity -->
                                    <circle cx="100" cy="80" r="30" fill="#78350f" stroke="#f59e0b" stroke-width="2"/>
                                    <text x="100" y="85" fill="#fbbf24" font-family="ui-monospace, monospace" font-size="12" text-anchor="middle" font-weight="600">Admin</text>
                                    <text x="100" y="130" fill="#fcd34d" font-family="ui-monospace, monospace" font-size="11" text-anchor="middle">ID: 1</text>
                                    <text x="100" y="145" fill="#fcd34d" font-family="ui-monospace, monospace" font-size="10" text-anchor="middle">(Real Identity)</text>

                                    <!-- Arrow Impersonate -->
                                    <path d="M 140 80 Q 300 30 460 80" fill="none" stroke="url(#imperGradient)" stroke-width="2" stroke-dasharray="5 5"/>
                                    <polygon points="460,80 450,75 450,85" fill="#3b82f6"/>
                                    <text x="300" y="50" fill="#94a3b8" font-family="ui-monospace, monospace" font-size="10" text-anchor="middle">impersonates</text>

                                    <!-- User Entity -->
                                    <circle cx="500" cy="80" r="30" fill="#1e3a5f" stroke="#3b82f6" stroke-width="2"/>
                                    <text x="500" y="85" fill="#93c5fd" font-family="ui-monospace, monospace" font-size="12" text-anchor="middle" font-weight="600">User</text>
                                    <text x="500" y="130" fill="#bfdbfe" font-family="ui-monospace, monospace" font-size="11" text-anchor="middle">ID: 42</text>
                                    <text x="500" y="145" fill="#bfdbfe" font-family="ui-monospace, monospace" font-size="10" text-anchor="middle">(Current Identity)</text>

                                    <!-- System View Box -->
                                    <rect x="50" y="170" width="500" height="100" rx="8" fill="#1e293b" stroke="#475569" stroke-width="1"/>
                                    <text x="70" y="190" fill="#cbd5e1" font-family="ui-monospace, monospace" font-size="11" font-weight="600">Visão do Sistema</text>

                                    <!-- Logic Row 1 -->
                                    <text x="70" y="215" fill="#94a3b8" font-family="ui-monospace, monospace" font-size="11">Auth::id()</text>
                                    <line x1="150" y1="212" x2="480" y2="212" stroke="#475569" stroke-width="1" stroke-dasharray="2 2"/>
                                    <text x="500" y="215" fill="#60a5fa" font-family="ui-monospace, monospace" font-size="11" font-weight="bold">42</text>

                                    <!-- Logic Row 2 -->
                                    <text x="70" y="240" fill="#94a3b8" font-family="ui-monospace, monospace" font-size="11">Audit Log</text>
                                    <line x1="150" y1="237" x2="480" y2="237" stroke="#475569" stroke-width="1" stroke-dasharray="2 2"/>
                                    <text x="500" y="240" fill="#fbbf24" font-family="ui-monospace, monospace" font-size="11" font-weight="bold">by Admin(1)</text>

                                    <!-- Logic Row 3 -->
                                    <text x="70" y="265" fill="#94a3b8" font-family="ui-monospace, monospace" font-size="11">check()</text>
                                    <line x1="150" y1="262" x2="480" y2="262" stroke="#475569" stroke-width="1" stroke-dasharray="2 2"/>
                                    <text x="500" y="265" fill="#22c55e" font-family="ui-monospace, monospace" font-size="11" font-weight="bold">true</text>

                                </svg>
                            </div>

                            <div class="feature-card">
                                <p class="font-semibold text-zinc-900 dark:text-white mb-2">Audit Log com Impersonation</p>
                                <div class="rounded-lg bg-zinc-950 p-3 text-xs overflow-x-auto">
                                    <pre class="language-php"><code>class AuditService
{
    public function log(string $action, $resource): void
    {
        $auth = app('auth');

        $data = [
            'action' => $action,
            'user_id' => $auth->id(),
        ];

        // Se em impersonation, registar admin original
        if ($auth->isImpersonating()) {
            $data['impersonator_id'] = $auth->impersonatorId();
            $data['is_impersonated'] = true;
        }

        AuditLog::create($data);
    }
}</code></pre>
                                </div>
                            </div>
                        </section>

                        {{-- Practical Examples --}}
                        <section id="exemplos" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <div class="prose prose-zinc dark:prose-invert max-w-none">
                                <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                    <flux:icon.code-bracket class="size-6 text-cyan-500" />
                                    Exemplos Práticos
                                </h2>
                            </div>

                            <div class="grid gap-4">
                                {{-- Example 1 --}}
                                <div class="feature-card">
                                    <p class="font-semibold text-zinc-900 dark:text-white mb-2">🎯 Proteger uma Action</p>
                                    <div class="rounded-lg bg-zinc-950 p-3 text-xs overflow-x-auto">
                                        <pre class="language-php"><code>class CreateInvoiceAction
{
    public function execute(array $data): Invoice
    {
        $auth = app('auth');

        if (!$auth->check()) {
            throw new AuthorizationException('Não autenticado');
        }

        if (!$auth->can('invoices:create')) {
            throw new AuthorizationException('Sem permissão');
        }

        return Invoice::create([
            ...$data,
            'created_by' => $auth->id(),
        ]);
    }
}</code></pre>
                                    </div>
                                </div>

                                {{-- Example 2 --}}
                                <div class="feature-card">
                                    <p class="font-semibold text-zinc-900 dark:text-white mb-2">🎯 Middleware de Autorização</p>
                                    <div class="rounded-lg bg-zinc-950 p-3 text-xs overflow-x-auto">
                                        <pre class="language-php"><code>class RequirePermissionMiddleware
{
    public function handle(Request $request, string $permission): Response
    {
        $auth = app('auth');

        if (!$auth->check()) {
            return redirect('/login');
        }

        if (!$auth->can($permission)) {
            return response()->error('Acesso negado', 403);
        }

        return $next($request);
    }
}

// Uso na rota:
Route::get('/invoices', [InvoiceController::class, 'index'])
    ->middleware('permission:invoices:view');</code></pre>
                                    </div>
                                </div>
                            </div>
                        </section>

                        {{-- Legacy Integration --}}
                        <section id="legacy" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <div class="prose prose-zinc dark:prose-invert max-w-none">
                                <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                    <flux:icon.link class="size-6 text-amber-500" />
                                    Integração com Legacy
                                </h2>
                            </div>

                            <div class="rounded-xl border-l-4 border-amber-500 bg-amber-50 dark:bg-amber-900/20 p-4">
                                <p class="text-sm text-amber-900 dark:text-amber-100">
                                    O AuthManager mantém sincronização bidirecional entre o sistema moderno e as variáveis globais legacy (<code>$u</code>).
                                </p>
                            </div>

                            <div class="rounded-xl bg-zinc-950 p-4 text-sm overflow-x-auto">
                                <pre class="language-php"><code>// Código legacy usa $u global
global $u;
if ($u->checkPriv('stocks:artigos')) {
    // ...
}

// Código moderno usa AuthManager
$auth = app('auth');
if ($auth->can('stocks:artigos')) {
    // ...
}

// ✅ Ambos funcionam e estão sincronizados!</code></pre>
                            </div>
                        </section>

                        {{-- Best Practices --}}
                        <section id="boas-praticas" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <div class="prose prose-zinc dark:prose-invert max-w-none">
                                <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                    <flux:icon.check-badge class="size-6 text-green-500" />
                                    Boas Práticas
                                </h2>
                            </div>

                            <div class="grid gap-4 md:grid-cols-2">
                                <div class="feature-card border-l-4 border-green-500">
                                    <p class="font-semibold text-green-600 dark:text-green-400 mb-2">✅ Verificar Cedo</p>
                                    <div class="rounded-lg bg-zinc-950 p-3 text-xs overflow-x-auto">
                                        <pre class="language-php"><code>public function execute(): void
{
    // BOM — verificar no início
    if (!app('auth')->can('perm')) {
        throw new Exception();
    }
    // ... resto do código
}</code></pre>
                                    </div>
                                </div>
                                <div class="feature-card border-l-4 border-green-500">
                                    <p class="font-semibold text-green-600 dark:text-green-400 mb-2">✅ Usar Policies</p>
                                    <div class="rounded-lg bg-zinc-950 p-3 text-xs overflow-x-auto">
                                        <pre class="language-php"><code>// BOM — lógica centralizada
$policy->authorize('edit', $doc);

// MAU — lógica espalhada
if ($user->id === $doc->owner
    || $user->checkPriv('admin')) {
}</code></pre>
                                    </div>
                                </div>
                                <div class="feature-card border-l-4 border-green-500">
                                    <p class="font-semibold text-green-600 dark:text-green-400 mb-2">✅ Logar Acessos Negados</p>
                                    <div class="rounded-lg bg-zinc-950 p-3 text-xs overflow-x-auto">
                                        <pre class="language-php"><code>if (!$auth->can($perm)) {
    Funcs::log('security',
        "Negado: user={$auth->id()}"
    );
    throw new Exception();
}</code></pre>
                                    </div>
                                </div>
                                <div class="feature-card border-l-4 border-red-500">
                                    <p class="font-semibold text-red-600 dark:text-red-400 mb-2">❌ Confiar no Cliente</p>
                                    <div class="rounded-lg bg-zinc-950 p-3 text-xs overflow-x-auto">
                                        <pre class="language-php"><code>// MAU — confiar no request
$inv->user_id = $request->input('user_id');

// BOM — usar ID autenticado
$inv->user_id = app('auth')->id();</code></pre>
                                    </div>
                                </div>
                            </div>
                        </section>

                        {{-- Navigation --}}
                        <div class="flex justify-between items-center pt-8 border-t border-zinc-200 dark:border-zinc-800">
                            <a href="{{ route('docs.show', 'service-providers') }}" class="inline-flex items-center gap-2 text-sm text-zinc-600 dark:text-zinc-400 hover:text-primary">
                                <flux:icon.arrow-left class="size-4" />
                                Service Providers
                            </a>
                            <a href="{{ route('docs.show', 'facades') }}" class="inline-flex items-center gap-2 text-sm font-medium text-primary hover:underline">
                                Facades
                                <flux:icon.arrow-right class="size-4" />
                            </a>
                        </div>
                    </div>

                </div>
            </div>

            {{-- Right Sidebar: Table of Contents --}}
            @include('docs.partials.toc', ['sections' => [
                'conceitos' => 'Conceitos',
                'quick-start' => 'Quick Start',
                'auth-manager' => 'AuthManager',
                'policies' => 'Policies',
                'access-response' => 'Access Response',
                'impersonation' => 'Impersonation',
                'exemplos' => 'Exemplos',
                'legacy' => 'Legacy',
                'boas-praticas' => 'Boas Práticas',
            ]])
        </div>
    </div>
@endsection
