@extends('layouts.docs')

@section('title', 'Container - OG Framework')

@section('body')
    <div class="relative min-h-screen bg-transparent px-4 pb-12 pt-6 sm:px-6 lg:px-8">
        <div class="floating-blur blue -left-10 -top-10 hidden lg:block"></div>
        <div class="floating-blur amber right-10 top-32 hidden lg:block"></div>

        <div class="mx-auto flex w-full max-w-7xl gap-8 bg-transparent">
            {{-- Left Sidebar: Navigation --}}
            @include('docs.partials.sidebar')

            {{-- Main Content --}}
            <div class="flex-1 min-w-0">
                <div class="relative overflow-hidden rounded-3xl border border-white/50 bg-white/85 p-8 shadow-xl shadow-primary/20 backdrop-blur dark:border-zinc-800 dark:bg-zinc-900/80">

                    {{-- Page Header --}}
                    <div class="mb-8">
                        <a href="{{ route('docs.index') }}" class="inline-flex items-center gap-1 text-sm text-primary hover:underline mb-4">
                            <flux:icon.arrow-left class="size-4" />
                            Voltar para Documentação
                        </a>
                        <p class="text-xs uppercase tracking-[0.3em] text-primary dark:text-blue-200">Arquitetura</p>
                        <h1 class="font-display text-4xl font-bold text-zinc-900 dark:text-white sm:text-5xl">Container de Dependências</h1>
                        <p class="mt-3 text-base text-zinc-700 dark:text-zinc-200 max-w-3xl">
                            O Container é o coração do framework — responsável por gerenciar dependências, instanciar classes automaticamente e servir como ponte entre o código moderno e o legado.
                        </p>
                    </div>

                    <div class="space-y-10">
                        {{-- What is DI --}}
                        <section id="dependency-injection" class="scroll-mt-28 space-y-4">
                            <div class="prose prose-zinc dark:prose-invert max-w-none">
                                <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                    <flux:icon.light-bulb class="size-6 text-amber-500" />
                                    O que é Dependency Injection?
                                </h2>
                                <p class="text-zinc-700 dark:text-zinc-300">
                                    <strong>Dependency Injection (DI)</strong> é um padrão onde as dependências de uma classe são "injetadas" em vez de criadas internamente. Isso torna o código testável e desacoplado:
                                </p>
                                <div class="rounded-xl bg-zinc-950 p-4 text-sm overflow-x-auto">
                                    <pre class="language-php"><code>// ❌ SEM DI — dependência acoplada
class UserService {
    public function __construct() {
        $this->database = new DatabaseConnection(); // Problema!
    }
}

// ✅ COM DI — dependência injetada
class UserService {
    public function __construct(DatabaseConnection $database) {
        $this->database = $database; // Flexível e testável
    }
}</code></pre>
                                </div>
                            </div>
                        </section>

                        {{-- Basic Example --}}
                        <section id="exemplo-basico" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <div class="prose prose-zinc dark:prose-invert max-w-none">
                                <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                    <flux:icon.code-bracket class="size-6 text-green-500" />
                                    Usando o Container
                                </h2>
                                <p class="text-zinc-700 dark:text-zinc-300">
                                    O helper <code>app()</code> é a forma principal de acessar o container:
                                </p>
                                <div class="rounded-xl bg-zinc-950 p-4 text-sm overflow-x-auto">
                                    <pre class="language-php"><code>// 1. OBTER o container
$container = app();

// 2. RESOLVER um serviço registrado
$cache = app('cache');
$database = app('database');

// 3. VERIFICAR se existe antes de usar
if (app()->has('database')) {
    $db = app('database');
}

// 4. RESOLVER com parâmetros
$service = app()->make(MyService::class, [
    'config' => $minhaConfig
]);</code></pre>
                                </div>
                            </div>

                            {{-- Services Available --}}
                            <div class="feature-card">
                                <p class="font-semibold text-zinc-900 dark:text-white mb-3">Serviços Disponíveis via app()</p>
                                <div class="grid gap-2 md:grid-cols-4 text-xs font-mono">
                                    <div class="bg-zinc-100 dark:bg-zinc-800 px-2 py-1 rounded"><code>app('database')</code></div>
                                    <div class="bg-zinc-100 dark:bg-zinc-800 px-2 py-1 rounded"><code>app('cache')</code></div>
                                    <div class="bg-zinc-100 dark:bg-zinc-800 px-2 py-1 rounded"><code>app('auth')</code></div>
                                    <div class="bg-zinc-100 dark:bg-zinc-800 px-2 py-1 rounded"><code>app('user')</code></div>
                                    <div class="bg-zinc-100 dark:bg-zinc-800 px-2 py-1 rounded"><code>app('session')</code></div>
                                    <div class="bg-zinc-100 dark:bg-zinc-800 px-2 py-1 rounded"><code>app('router')</code></div>
                                    <div class="bg-zinc-100 dark:bg-zinc-800 px-2 py-1 rounded"><code>app('email')</code></div>
                                    <div class="bg-zinc-100 dark:bg-zinc-800 px-2 py-1 rounded"><code>app('hooks')</code></div>
                                </div>
                            </div>
                        </section>

                        {{-- Auto-wiring --}}
                        <section id="auto-wiring" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <div class="prose prose-zinc dark:prose-invert max-w-none">
                                <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                    <flux:icon.sparkles class="size-6 text-amber-500" />
                                    Auto-Wiring
                                </h2>
                                <p class="text-zinc-700 dark:text-zinc-300">
                                    O container usa <strong>reflection</strong> para resolver dependências via type-hints automaticamente:
                                </p>
                                <div class="rounded-xl bg-zinc-950 p-4 text-sm overflow-x-auto">
                                    <pre class="language-php"><code>use Og\Modules\Common\Queue\Contracts\QueueManagerInterface;
use Og\Modules\Common\Cache\Cache;

class ReportGenerator
{
    // O container injeta TUDO automaticamente!
    public function __construct(
        private QueueManagerInterface $queue,  // Resolvido via DI
        private Cache $cache                    // Resolvido via DI
    ) {}

    public function generate(int $reportId): void
    {
        // Usar cache
        if ($this->cache->has("report:{$reportId}")) {
            return $this->cache->get("report:{$reportId}");
        }

        // Enfileirar job
        $this->queue->push(new GenerateReportJob($reportId));
    }
}

// Uso — o container faz a mágica:
$generator = app(ReportGenerator::class);  // Tudo injetado!</code></pre>
                                </div>
                            </div>
                        </section>

                        {{-- make() vs get() --}}
                        <section id="make-vs-get" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <div class="prose prose-zinc dark:prose-invert max-w-none">
                                <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                    <flux:icon.arrows-right-left class="size-6 text-purple-500" />
                                    make() vs get()
                                </h2>
                            </div>

                            <div class="grid gap-4 md:grid-cols-2">
                                <div class="feature-card border-l-4 border-blue-500">
                                    <p class="font-mono text-sm font-semibold text-blue-600 dark:text-blue-400 mb-2">get($id)</p>
                                    <p class="text-sm text-zinc-700 dark:text-zinc-300 mb-2">Obtém serviço registrado. Lança exceção se não existir.</p>
                                    <div class="rounded-lg bg-zinc-950 p-3 text-xs overflow-x-auto">
                                        <pre class="language-php"><code>$cache = app()->get('cache');
$db = app()->get('database');</code></pre>
                                    </div>
                                </div>
                                <div class="feature-card border-l-4 border-green-500">
                                    <p class="font-mono text-sm font-semibold text-green-600 dark:text-green-400 mb-2">make($class, $params)</p>
                                    <p class="text-sm text-zinc-700 dark:text-zinc-300 mb-2">Instancia classe com auto-wiring. Aceita parâmetros extras.</p>
                                    <div class="rounded-lg bg-zinc-950 p-3 text-xs overflow-x-auto">
                                        <pre class="language-php"><code>$svc = app()->make(MyService::class);
$svc = app()->make(MyService::class, [
    'config' => $cfg
]);</code></pre>
                                    </div>
                                </div>
                            </div>
                        </section>

                        {{-- Legacy Bridge --}}
                        <section id="legacy-bridge" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <div class="prose prose-zinc dark:prose-invert max-w-none">
                                <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                    <flux:icon.link class="size-6 text-amber-500" />
                                    Bridge com Legacy
                                </h2>
                            </div>
                            <div class="rounded-xl border-l-4 border-amber-500 bg-amber-50 dark:bg-amber-900/20 p-4">
                                <p class="text-sm text-amber-900 dark:text-amber-100">
                                    O <code>GlobalServiceProvider</code> expõe variáveis globais (<code>$u</code>, <code>$a</code>, <code>$s</code>, etc.) via container. Assim você pode acessar tanto via <code>$GLOBALS['u']</code> quanto via <code>app('user')</code> — facilitando a migração gradual.
                                </p>
                            </div>
                            <div class="rounded-xl bg-zinc-950 p-4 text-sm overflow-x-auto">
                                <pre class="language-php"><code>// Código legacy
global $u;
if ($u->checkPriv('stocks:artigos')) {
    // ...
}

// Código moderno (equivalente!)
$user = app('user');
if ($user->checkPriv('stocks:artigos')) {
    // ...
}

// Ambos funcionam e estão sincronizados!</code></pre>
                            </div>

                            {{-- Legacy mapping --}}
                            <div class="overflow-x-auto">
                                <table class="w-full text-sm text-left">
                                    <thead class="text-xs uppercase bg-zinc-100 dark:bg-zinc-800">
                                        <tr>
                                            <th class="px-4 py-3">Container</th>
                                            <th class="px-4 py-3">Global Legacy</th>
                                            <th class="px-4 py-3">Classe</th>
                                        </tr>
                                    </thead>
                                    <tbody class="divide-y divide-zinc-200 dark:divide-zinc-700 text-xs">
                                        <tr class="bg-white dark:bg-zinc-900">
                                            <td class="px-4 py-2 font-mono">app('database')</td>
                                            <td class="px-4 py-2 font-mono">$GLOBALS['db']</td>
                                            <td class="px-4 py-2">DatabaseConnector</td>
                                        </tr>
                                        <tr class="bg-zinc-50 dark:bg-zinc-800/50">
                                            <td class="px-4 py-2 font-mono">app('user')</td>
                                            <td class="px-4 py-2 font-mono">$GLOBALS['u']</td>
                                            <td class="px-4 py-2">User</td>
                                        </tr>
                                        <tr class="bg-white dark:bg-zinc-900">
                                            <td class="px-4 py-2 font-mono">app('app')</td>
                                            <td class="px-4 py-2 font-mono">$GLOBALS['a']</td>
                                            <td class="px-4 py-2">Aplication</td>
                                        </tr>
                                        <tr class="bg-zinc-50 dark:bg-zinc-800/50">
                                            <td class="px-4 py-2 font-mono">app('session')</td>
                                            <td class="px-4 py-2 font-mono">$GLOBALS['s']</td>
                                            <td class="px-4 py-2">Session</td>
                                        </tr>
                                        <tr class="bg-white dark:bg-zinc-900">
                                            <td class="px-4 py-2 font-mono">app('email')</td>
                                            <td class="px-4 py-2 font-mono">$GLOBALS['mail']</td>
                                            <td class="px-4 py-2">Email</td>
                                        </tr>
                                    </tbody>
                                </table>
                            </div>
                        </section>

                        {{-- Next Steps --}}
                        <div class="rounded-xl border border-blue-200 dark:border-blue-800 bg-blue-50 dark:bg-blue-900/20 p-4">
                            <p class="text-sm text-blue-900 dark:text-blue-100">
                                <strong>Próximo:</strong> Veja <a href="{{ route('docs.show', 'service-providers') }}" class="underline">Service Providers</a> para aprender como registrar serviços no container usando <code>singleton()</code>, <code>bind()</code>, e <code>instance()</code>.
                            </p>
                        </div>

                        {{-- Navigation --}}
                        <div class="flex justify-between items-center pt-8 border-t border-zinc-200 dark:border-zinc-800">
                            <a href="{{ route('docs.show', 'bootstrap') }}" class="inline-flex items-center gap-2 text-sm text-zinc-600 dark:text-zinc-400 hover:text-primary">
                                <flux:icon.arrow-left class="size-4" />
                                Bootstrap
                            </a>
                            <a href="{{ route('docs.show', 'service-providers') }}" class="inline-flex items-center gap-2 text-sm font-medium text-primary hover:underline">
                                Service Providers
                                <flux:icon.arrow-right class="size-4" />
                            </a>
                        </div>
                    </div>

                </div>
            </div>

            {{-- Right Sidebar: Table of Contents --}}
            @include('docs.partials.toc', ['sections' => [
                'dependency-injection' => 'O que é DI?',
                'exemplo-basico' => 'Usando o Container',
                'auto-wiring' => 'Auto-Wiring',
                'make-vs-get' => 'make() vs get()',
                'legacy-bridge' => 'Bridge com Legacy',
            ]])
        </div>
    </div>
@endsection
