@extends('layouts.docs')

@section('title', 'Database System - OG Framework')

@section('body')
    <div class="relative min-h-screen bg-transparent px-4 pb-12 pt-6 sm:px-6 lg:px-8">
        {{-- Floating shapes for styling --}}
        <div class="floating-blur orange -left-10 -top-10 hidden lg:block"></div>
        <div class="floating-blur purple right-10 top-32 hidden lg:block"></div>

        <div class="mx-auto flex w-full max-w-7xl gap-8 bg-transparent">
            {{-- Left Sidebar: Navigation --}}
            @include('docs.partials.sidebar')

            {{-- Main Content --}}
            <div class="flex-1 min-w-0">
                <div class="relative overflow-hidden rounded-3xl border border-white/50 bg-white/85 p-8 shadow-xl shadow-primary/20 backdrop-blur dark:border-zinc-800 dark:bg-zinc-900/80">

                    {{-- Page Header --}}
                    <div class="mb-8">
                        <a href="{{ route('docs.index') }}" class="inline-flex items-center gap-1 text-sm text-primary hover:underline mb-4">
                            <flux:icon.arrow-left class="size-4" />
                            Voltar para Documentação
                        </a>
                        <p class="text-xs uppercase tracking-[0.3em] text-primary dark:text-blue-200">System</p>
                        <h1 class="font-display text-4xl font-bold text-zinc-900 dark:text-white sm:text-5xl">Database System</h1>
                        <p class="mt-3 text-base text-zinc-700 dark:text-zinc-200 max-w-3xl">
                            ORM moderno e Schema Builder para OfficeGest — Modelos tipados, Query Builder expressivo, Casting automático e Migrações.
                        </p>
                    </div>

                    <div class="space-y-10">

                        {{-- Visão Geral --}}
                        <section id="visao-geral" class="scroll-mt-28 space-y-4">
                            <div class="prose prose-zinc dark:prose-invert max-w-none">
                                <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                    <flux:icon.server class="size-6 text-blue-500" />
                                    Visão Geral
                                </h2>
                                <p class="text-zinc-700 dark:text-zinc-300">
                                    O Sistema de Base de Dados do OfficeGest fornece uma camada moderna de ORM que <strong>encapsula</strong> o query builder legado existente (CodeIgniter 2).
                                </p>
                            </div>

                            {{-- Important Note --}}
	                            <div class="rounded-lg border border-blue-200 bg-blue-50 p-4 dark:border-blue-900/30 dark:bg-blue-900/10">
	                                <p class="font-semibold text-blue-800 dark:text-blue-300">📌 Arquitetura</p>
	                                <p class="text-sm text-blue-700 dark:text-blue-400 mt-1">
	                                    O <code>Model</code> e o <code>Builder</code> <strong>não substituem</strong> o query builder legado — eles o <strong>encapsulam</strong>. 
	                                    Internamente, todas as queries continuam a usar o <code>OGDB_query_builder</code> do CodeIgniter 2. 
	                                    A integração com o legado é isolada num adaptador interno (<code>LegacyQueryAdapter</code>), mantendo a API do Builder moderna e consistente.
	                                    O que adicionamos é uma camada de conveniência com tipagem forte, PHPDoc, PHP 8 Attributes, casting automático e proteções contra operações acidentais.
	                                </p>
	                            </div>

                            {{-- Feature Cards --}}
                            <div class="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
                                <div class="rounded-xl border border-zinc-200 bg-zinc-50 p-4 dark:border-zinc-800 dark:bg-zinc-900/30">
                                    <h3 class="font-semibold text-zinc-900 dark:text-white flex items-center gap-2 mb-2">
                                        <flux:icon.cube class="size-5 text-purple-500" />
                                        Modelos Tipados
                                    </h3>
                                    <p class="text-sm text-zinc-600 dark:text-zinc-400">PHPDoc e autocompletar para todas as propriedades.</p>
                                </div>
                                <div class="rounded-xl border border-zinc-200 bg-zinc-50 p-4 dark:border-zinc-800 dark:bg-zinc-900/30">
                                    <h3 class="font-semibold text-zinc-900 dark:text-white flex items-center gap-2 mb-2">
                                        <flux:icon.code-bracket class="size-5 text-green-500" />
                                        Query Builder
                                    </h3>
                                    <p class="text-sm text-zinc-600 dark:text-zinc-400">Interface fluente com proteções contra operações em massa.</p>
                                </div>
                                <div class="rounded-xl border border-zinc-200 bg-zinc-50 p-4 dark:border-zinc-800 dark:bg-zinc-900/30">
                                    <h3 class="font-semibold text-zinc-900 dark:text-white flex items-center gap-2 mb-2">
                                        <flux:icon.arrow-path class="size-5 text-orange-500" />
                                        Casting Automático
                                    </h3>
                                    <p class="text-sm text-zinc-600 dark:text-zinc-400">Conversão automática entre tipos PHP e tipos de banco.</p>
                                </div>
                            </div>

                            {{-- Architecture --}}
                            <div class="rounded-xl bg-gradient-to-br from-zinc-900 to-zinc-950 p-6 overflow-x-auto">
                                <pre class="language-php text-xs"><code>Modules/Common/Database/
├── Model.php           # Classe base para modelos
├── Builder.php         # Query builder moderno
├── Traits/
│   ├── Castable.php        # Sistema de casting
│   ├── HasAttributes.php   # Manipulação de atributos
│   ├── HasAccessors.php    # PHP 8 Attributes
│   ├── TracksChanges.php   # Dirty tracking
│   ├── Persistable.php     # CRUD operations
│   └── HasTimestamps.php   # Timestamps configuráveis
├── Casts/
│   ├── JsonCast.php        # JSON encode/decode
│   ├── DateTimeCast.php    # Carbon conversion
│   └── DecimalCast.php     # Números com precisão
└── Interfaces/
    ├── QueueableEntity.php # Serialização em jobs
    └── UrlRoutable.php     # Route model binding</code></pre>
                            </div>
                        </section>

                        {{-- Modelos --}}
                        <section id="modelos" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <div class="prose prose-zinc dark:prose-invert max-w-none">
                                <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                    <flux:icon.cube class="size-6 text-purple-500" />
                                    Modelos de Dados
                                </h2>
                                <p class="text-zinc-700 dark:text-zinc-300">
                                    Um <strong>Modelo</strong> é uma classe PHP que representa uma tabela da base de dados. Em vez de arrays associativos, trabalhamos com <strong>objetos tipados</strong>.
                                </p>
                            </div>

                            {{-- Basic Model Example --}}
                            <div class="rounded-xl border border-zinc-200 bg-zinc-50 p-4 dark:border-zinc-800 dark:bg-zinc-900/30">
                                <h3 class="font-mono text-sm font-bold text-primary mb-2">Estrutura Básica de um Modelo</h3>
                                <pre class="language-php text-xs rounded-lg bg-zinc-950 p-3"><code>&lt;?php

namespace Og\Modules\Tax\Models;

use Og\Modules\Common\Database\Model;

/**
 * Modelo de IVA (Taxas de Imposto).
 * 
 * @property-read string $codiva       Código do IVA
 * @property-read string $designacao   Descrição da taxa
 * @property-read float $iva           Percentagem de IVA
 * @property-read Carbon $date_reg     Data de registo
 */
class Vat extends Model
{
    protected string $table = TABLE_IVAS;
    
    protected string|array $primaryKey = 'CodIVA';
    
    protected array $casts = [
        'iva' => 'decimal:2',
        'date_reg' => 'datetime',
        'date_alter' => 'datetime',
    ];
}</code></pre>
                            </div>

                            {{-- Properties Table --}}
                            <div class="overflow-x-auto">
                                <table class="min-w-full text-sm">
                                    <thead class="bg-zinc-100 dark:bg-zinc-800">
                                        <tr>
                                            <th class="px-4 py-2 text-left font-semibold text-zinc-900 dark:text-white">Propriedade</th>
                                            <th class="px-4 py-2 text-left font-semibold text-zinc-900 dark:text-white">Tipo</th>
                                            <th class="px-4 py-2 text-left font-semibold text-zinc-900 dark:text-white">Default</th>
                                            <th class="px-4 py-2 text-left font-semibold text-zinc-900 dark:text-white">Descrição</th>
                                        </tr>
                                    </thead>
                                    <tbody class="divide-y divide-zinc-200 dark:divide-zinc-700">
                                        <tr><td class="px-4 py-2 font-mono text-xs text-purple-600 dark:text-purple-400">$table</td><td class="px-4 py-2">string</td><td class="px-4 py-2">—</td><td class="px-4 py-2">Nome da tabela</td></tr>
                                        <tr><td class="px-4 py-2 font-mono text-xs text-purple-600 dark:text-purple-400">$primaryKey</td><td class="px-4 py-2">string|array</td><td class="px-4 py-2">'id'</td><td class="px-4 py-2">Chave primária (simples ou composta)</td></tr>
                                        <tr><td class="px-4 py-2 font-mono text-xs text-purple-600 dark:text-purple-400">$timestamps</td><td class="px-4 py-2">bool</td><td class="px-4 py-2">true</td><td class="px-4 py-2">Ativa timestamps automáticos</td></tr>
                                        <tr><td class="px-4 py-2 font-mono text-xs text-purple-600 dark:text-purple-400">$casts</td><td class="px-4 py-2">array</td><td class="px-4 py-2">[]</td><td class="px-4 py-2">Mapeamento de casts</td></tr>
                                        <tr><td class="px-4 py-2 font-mono text-xs text-purple-600 dark:text-purple-400">$fillable</td><td class="px-4 py-2">array</td><td class="px-4 py-2">[]</td><td class="px-4 py-2">Campos permitidos em fill()</td></tr>
                                        <tr><td class="px-4 py-2 font-mono text-xs text-purple-600 dark:text-purple-400">$hidden</td><td class="px-4 py-2">array</td><td class="px-4 py-2">[]</td><td class="px-4 py-2">Campos ocultos em toArray()</td></tr>
                                    </tbody>
                                </table>
                            </div>

                            {{-- Usage Examples --}}
                            <div class="rounded-xl border border-zinc-200 bg-zinc-50 p-4 dark:border-zinc-800 dark:bg-zinc-900/30">
                                <h3 class="font-mono text-sm font-bold text-primary mb-2">Uso Básico</h3>
                                <pre class="language-php text-xs rounded-lg bg-zinc-950 p-3"><code>// Obter todos os IVAs
$vats = Vat::all();

// Obter um IVA específico por primary key
$vat = Vat::find('NOR');

	// Acessar propriedades com tipagem
	echo $vat->designacao;  // string
	echo $vat->iva;         // string (decimal:2, ex.: "23.00")
	echo $vat->date_reg;    // Carbon (objeto de data)

// Filtrar com where
$normalVats = Vat::where('tipotaxa', 'NOR')
    ->where('iva', '>', 0)
    ->get();</code></pre>
                            </div>

                            {{-- Multiple Syntax Options --}}
                            <div class="rounded-lg border border-green-200 bg-green-50 p-4 dark:border-green-900/30 dark:bg-green-900/10">
                                <p class="font-semibold text-green-800 dark:text-green-300">✨ Sintaxe Flexível</p>
                                <p class="text-sm text-green-700 dark:text-green-400 mt-1 mb-3">
                                    O Model suporta várias formas de interação. <strong>A forma estática é a recomendada</strong> para uso comum:
                                </p>
                                <pre class="language-php text-xs rounded-lg bg-zinc-950 p-3"><code>// ✅ RECOMENDADO: Métodos estáticos (mais limpo e direto)
$vats = Vat::where('iva', '>', 0)->get();
$vat = Vat::find('NOR');
$vat = Vat::findOrFail('NOR');
$count = Vat::where('tipotaxa', 'NOR')->count();

	// 🔧 Via query() - acesso ao Builder moderno (API camelCase)
	$results = Vat::query()
	    ->where('iva', '>', 0)
	    ->asArray()
	    ->get();
	
	// Escape hatch: acesso ao query builder legado (CodeIgniter 2)
	// Use apenas quando precisares de métodos ainda não mapeados no wrapper moderno.
	$legacy = Vat::query()->unwrap();
	$legacy->where_in('codiva', ['NOR', 'RED'])->get();

// Via instância (para casos específicos)
$model = new Vat();
$vats = $model->newQuery()->where('iva', '>', 0)->get();</code></pre>
                            </div>

                            <div class="rounded-lg border border-blue-200 bg-blue-50 p-4 dark:border-blue-900/30 dark:bg-blue-900/10">
                                <p class="font-semibold text-blue-800 dark:text-blue-300">💡 Quando usar query()?</p>
                                <p class="text-sm text-blue-700 dark:text-blue-400 mt-1">
	                                    Use <code>Model::query()</code> para obter um <code>Builder</code> moderno (API camelCase) e encadear queries.
	                                    Se precisares de um método específico do CodeIgniter 2 que ainda não existe no wrapper moderno, usa <code>->unwrap()</code> para acessar o <code>OGDB_query_builder</code> diretamente.
	                                </p>
	                            </div>
                        </section>

                        {{-- Sistema de Casting --}}
                        <section id="casting" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <div class="prose prose-zinc dark:prose-invert max-w-none">
                                <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                    <flux:icon.arrow-path class="size-6 text-orange-500" />
                                    Sistema de Casting
                                </h2>
                                <p class="text-zinc-700 dark:text-zinc-300">
                                    O sistema de casting converte automaticamente valores entre tipos PHP e tipos de base de dados.
                                </p>
                            </div>

                            {{-- Simple Casts --}}
                            <div class="rounded-xl border border-zinc-200 bg-zinc-50 p-4 dark:border-zinc-800 dark:bg-zinc-900/30">
                                <h3 class="font-mono text-sm font-bold text-primary mb-2">Casts Simples (String)</h3>
                                <pre class="language-php text-xs rounded-lg bg-zinc-950 p-3"><code>protected array $casts = [
    'is_active' => 'bool',       // "T" → true
    'price' => 'decimal:2',      // "23.000" → "23.00"
    'settings' => 'json',        // JSON string → array
    'created_at' => 'datetime',  // String → Carbon
    'quantity' => 'int',         // "123" → 123
];</code></pre>
                            </div>

                            {{-- Cast Types Table --}}
                            <div class="overflow-x-auto">
                                <table class="min-w-full text-sm">
                                    <thead class="bg-zinc-100 dark:bg-zinc-800">
                                        <tr>
                                            <th class="px-4 py-2 text-left font-semibold text-zinc-900 dark:text-white">Tipo</th>
                                            <th class="px-4 py-2 text-left font-semibold text-zinc-900 dark:text-white">Descrição</th>
                                            <th class="px-4 py-2 text-left font-semibold text-zinc-900 dark:text-white">Exemplo</th>
                                        </tr>
                                    </thead>
                                    <tbody class="divide-y divide-zinc-200 dark:divide-zinc-700">
                                        <tr><td class="px-4 py-2 font-mono text-xs">int, integer</td><td class="px-4 py-2">Inteiro</td><td class="px-4 py-2">"123" → 123</td></tr>
                                        <tr><td class="px-4 py-2 font-mono text-xs">float, double</td><td class="px-4 py-2">Decimal</td><td class="px-4 py-2">"12.5" → 12.5</td></tr>
                                        <tr><td class="px-4 py-2 font-mono text-xs">bool, boolean</td><td class="px-4 py-2">Booleano</td><td class="px-4 py-2">"T" → true</td></tr>
                                        <tr><td class="px-4 py-2 font-mono text-xs">array, json</td><td class="px-4 py-2">JSON → array</td><td class="px-4 py-2">'{"a":1}' → ['a' => 1]</td></tr>
                                        <tr><td class="px-4 py-2 font-mono text-xs">datetime, date</td><td class="px-4 py-2">Carbon instance</td><td class="px-4 py-2">"2024-01-15" → Carbon</td></tr>
                                        <tr><td class="px-4 py-2 font-mono text-xs">timestamp</td><td class="px-4 py-2">Unix timestamp</td><td class="px-4 py-2">1705276800 → Carbon</td></tr>
                                        <tr><td class="px-4 py-2 font-mono text-xs">collection</td><td class="px-4 py-2">Array → Collection</td><td class="px-4 py-2">[1,2,3] → Collection</td></tr>
                                        <tr><td class="px-4 py-2 font-mono text-xs">decimal:N</td><td class="px-4 py-2">Decimal com N casas</td><td class="px-4 py-2">"23.000" → "23.00"</td></tr>
                                    </tbody>
                                </table>
                            </div>

                            {{-- Custom Cast --}}
                            <div class="rounded-xl border border-zinc-200 bg-zinc-50 p-4 dark:border-zinc-800 dark:bg-zinc-900/30">
                                <h3 class="font-mono text-sm font-bold text-primary mb-2">Cast Customizado</h3>
                                <pre class="language-php text-xs rounded-lg bg-zinc-950 p-3"><code>use Og\Modules\Common\Database\Casts\CastInterface;

class MoneyCast implements CastInterface
{
    public function get(mixed $value, array $attributes = []): Money
    {
        return new Money((int) $value, $attributes['currency'] ?? 'EUR');
    }

    public function set(mixed $value, array $attributes = []): int
    {
        return $value instanceof Money ? $value->cents : (int) $value;
    }
}

// Uso no Model
protected array $casts = [
    'total_cents' => MoneyCast::class,
];</code></pre>
                            </div>
                        </section>

                        {{-- Accessors e Mutators --}}
                        <section id="accessors" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <div class="prose prose-zinc dark:prose-invert max-w-none">
                                <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                    <flux:icon.adjustments-horizontal class="size-6 text-teal-500" />
                                    Accessors e Mutators
                                </h2>
                                <p class="text-zinc-700 dark:text-zinc-300">
                                    Accessors (getters virtuais) e Mutators (setters com lógica) transformam dados ao ler ou escrever. 
                                    Suporta <strong>dois estilos</strong>: PHP 8 Attributes e Laravel-style.
                                </p>
                            </div>

                            <div class="space-y-4">
                                {{-- Modern Style --}}
                                <div class="rounded-xl border border-green-200 bg-green-50 p-4 dark:border-green-900/30 dark:bg-green-900/10">
                                    <h3 class="font-semibold text-green-800 dark:text-green-300 mb-2">✨ Estilo Moderno (PHP 8 Attributes)</h3>
                                    <pre class="language-php text-xs rounded-lg bg-zinc-950 p-3"><code>use Og\Modules\Common\Database\Attributes\Computed;

class TipoDocumento extends Model
{
    #[Computed]
    protected function nomeCompleto(): string
    {
        $nome = $this->langvar 
            ? lang($this->langvar) 
            : $this->codabreviado;
            
        return "{$nome} [{$this->codabreviado}]";
    }
}

// Uso - método convertido para snake_case
echo $doc->nome_completo;  // "Fatura [FT]"</code></pre>
                                </div>

                                {{-- Legacy Style --}}
                                <div class="rounded-xl border border-zinc-200 bg-zinc-50 p-4 dark:border-zinc-800 dark:bg-zinc-900/30">
                                    <h3 class="font-semibold text-zinc-800 dark:text-zinc-300 mb-2">Estilo Legacy (Laravel-style)</h3>
                                    <pre class="language-php text-xs rounded-lg bg-zinc-950 p-3"><code>class TipoDocumento extends Model
{
    // Accessor: getXxxAttribute()
    protected function getNomeCompletoAttribute(): string
    {
        return "{$this->designacao} [{$this->codabreviado}]";
    }
}

// Uso é o mesmo
echo $doc->nome_completo;  // "Fatura [FT]"</code></pre>
                                </div>
                            </div>

                            {{-- Mutator Example --}}
                            <div class="rounded-xl border border-zinc-200 bg-zinc-50 p-4 dark:border-zinc-800 dark:bg-zinc-900/30">
                                <h3 class="font-mono text-sm font-bold text-primary mb-2">Mutator (#[Mutator])</h3>
                                <pre class="language-php text-xs rounded-lg bg-zinc-950 p-3"><code>use Og\Modules\Common\Database\Attributes\Mutator;

class User extends Model
{
    #[Mutator]
    protected function password(string $value): void
    {
        $this->attributes['password'] = password_hash($value, PASSWORD_DEFAULT);
    }
}

$user->password = 'minhasenha';  // Automaticamente hashed!</code></pre>
                            </div>

                            <div class="rounded-lg border border-amber-200 bg-amber-50 p-4 dark:border-amber-900/30 dark:bg-amber-900/10">
                                <p class="font-semibold text-amber-800 dark:text-amber-300">💡 Dica</p>
                                <p class="text-sm text-amber-700 dark:text-amber-400 mt-1">Campos <code>password</code> e <code>senha</code> são hashed automaticamente via <code>password_hash()</code> sem precisar de mutator!</p>
                            </div>
                        </section>

                        {{-- Query Builder --}}
                        <section id="builder" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
	                            <div class="prose prose-zinc dark:prose-invert max-w-none">
	                                <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
	                                    <flux:icon.code-bracket class="size-6 text-green-500" />
	                                    Builder (Wrapper do Query Builder)
	                                </h2>
	                                <p class="text-zinc-700 dark:text-zinc-300">
	                                    O <code>Builder</code> fornece uma interface fluente e tipada que <strong>encapsula</strong> o <code>OGDB_query_builder</code> legado do CodeIgniter 2. 
	                                    Internamente, todos os métodos delegam para o query builder original, mas com camadas adicionais de segurança e conveniência.
	                                </p>
	                            </div>
	
	                            <div class="rounded-lg border border-amber-200 bg-amber-50 p-4 dark:border-amber-900/30 dark:bg-amber-900/10">
	                                <p class="font-semibold text-amber-800 dark:text-amber-300">ℹ️ API camelCase</p>
	                                <p class="text-sm text-amber-700 dark:text-amber-400 mt-1">
	                                    No wrapper moderno, priorizamos métodos em <code>camelCase</code> (ex.: <code>whereIn</code>). Chamadas diretas a métodos snake_case do legado
	                                    (ex.: <code>where_in</code>, <code>order_by</code>) devem ser feitas via <code>->unwrap()</code>.
	                                </p>
	                            </div>

                            {{-- Query Methods --}}
                            <div class="space-y-4">
                                <div class="rounded-xl border border-zinc-200 bg-zinc-50 p-4 dark:border-zinc-800 dark:bg-zinc-900/30">
                                    <h3 class="font-mono text-sm font-bold text-primary mb-2">Métodos de Consulta</h3>
                                    <pre class="language-php text-xs rounded-lg bg-zinc-950 p-3"><code>// WHERE básico
Vat::where('tipotaxa', 'NOR')
    ->where('iva', '>', 0)
    ->get();

	// WHERE IN
	Vat::whereIn('codiva', ['NOR', 'RED', 'ISE'])->get();

	// LIMIT
	Vat::query()->limit(10)->get();</code></pre>
	                                </div>

                                <div class="rounded-xl border border-zinc-200 bg-zinc-50 p-4 dark:border-zinc-800 dark:bg-zinc-900/30">
                                    <h3 class="font-mono text-sm font-bold text-primary mb-2">Métodos de Conveniência</h3>
                                    <pre class="language-php text-xs rounded-lg bg-zinc-950 p-3"><code>// Encontrar por ID
$vat = Vat::find('NOR');
$vat = Vat::findOrFail('NOR');

// Obter valores de uma coluna
$ids = Vat::pluck('codiva');  // ['NOR', 'RED', 'ISE']

// Verificar existência
if (Vat::where('codiva', 'NEW')->exists()) {
    // ...
}</code></pre>
                                </div>
                            </div>

                            {{-- FirstOrCreate / UpdateOrCreate --}}
                            <div class="rounded-xl border border-zinc-200 bg-zinc-50 p-4 dark:border-zinc-800 dark:bg-zinc-900/30">
                                <h3 class="font-mono text-sm font-bold text-primary mb-2">Métodos de Criação</h3>
                                <pre class="language-php text-xs rounded-lg bg-zinc-950 p-3"><code>// Encontrar ou criar (NÃO salva automaticamente)
$vat = Vat::firstOrNew(
    ['codiva' => 'NEW'],                        // Critério de busca
    ['iva' => 23, 'designacao' => 'Novo']       // Valores para nova instância
);

// Encontrar ou criar E SALVAR
$vat = Vat::firstOrCreate(
    ['codiva' => 'NEW'],
    ['iva' => 23, 'designacao' => 'Novo']
);

// Atualizar ou criar
$vat = Vat::updateOrCreate(
    ['codiva' => 'NEW'],
    ['iva' => 25]
);</code></pre>
                            </div>

                            {{-- Safety Guards --}}
                            <div class="rounded-lg border border-red-200 bg-red-50 p-4 dark:border-red-900/30 dark:bg-red-900/10">
	                                <p class="font-semibold text-red-800 dark:text-red-300">⚠️ Proteção Contra Operações em Massa</p>
	                                <p class="text-sm text-red-700 dark:text-red-400 mt-1 mb-3">
	                                    O Builder previne UPDATE/DELETE acidentais sem WHERE por padrão. Para operações em massa, o bypass precisa ser explícito.
	                                </p>
	                                <pre class="language-php text-xs rounded-lg bg-zinc-950 p-3"><code>// ❌ ERRO: RuntimeException - sem WHERE
	Vat::query()->delete();
	
	// ✅ CORRETO: DELETE com WHERE
	Vat::where('codiva', 'OLD')->delete();
	
	// ⚠️ Bypass explícito (usar com extrema cautela)
	Vat::query()->withoutSafetyGuards()->delete();
	
	// ✅ Segurança extra: whereIn([]) representa conjunto vazio e vira no-op
	$affected = Vat::query()->whereIn('codiva', [])->update(['iva' => 0]); // 0</code></pre>
	                            </div>

                            {{-- Cache --}}
                            <div class="rounded-xl border border-zinc-200 bg-zinc-50 p-4 dark:border-zinc-800 dark:bg-zinc-900/30">
                                <h3 class="font-mono text-sm font-bold text-primary mb-2">Cache de Queries e Modo Híbrido</h3>
                                <pre class="language-php text-xs rounded-lg bg-zinc-950 p-3"><code>// Cache por 5 minutos (300 segundos)
$vats = Vat::where('tipotaxa', 'NOR')
    ->cached('vats:normal', 300)
    ->get();

	// Modo híbrido para relatórios (sem hidratar Models)
	// Usa asArray() para receber arrays
	$data = Vat::query()->asArray()->get();</code></pre>
	                            </div>
                        </section>

	                        {{-- CRUD Operations --}}
	                        <section id="crud" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
	                            <div class="prose prose-zinc dark:prose-invert max-w-none">
	                                <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
	                                    <flux:icon.document-text class="size-6 text-blue-500" />
	                                    CRUD Operations
	                                </h2>
	                            </div>

                            <div class="space-y-4">
                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-mono text-sm font-bold text-green-600 mb-2">CREATE</h3>
                                    <pre class="language-php text-xs rounded-lg bg-zinc-950 p-3"><code>$vat = new Vat();
$vat->fill(['codiva' => 'NEW', 'iva' => 15]);
$vat->save();</code></pre>
                                </div>

                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-mono text-sm font-bold text-blue-600 mb-2">READ</h3>
                                    <pre class="language-php text-xs rounded-lg bg-zinc-950 p-3"><code>$vat = Vat::find('NOR');
$vats = Vat::where('iva', '>', 0)->get();</code></pre>
                                </div>

                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-mono text-sm font-bold text-orange-600 mb-2">UPDATE</h3>
                                    <pre class="language-php text-xs rounded-lg bg-zinc-950 p-3"><code>$vat->iva = 16;
$vat->save();  // UPDATE só do campo 'iva'

// Ou via update
$vat->update(['iva' => 17]);</code></pre>
                                </div>

                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-mono text-sm font-bold text-red-600 mb-2">DELETE</h3>
                                    <pre class="language-php text-xs rounded-lg bg-zinc-950 p-3"><code>$vat->delete();

// Via query (com proteção)
Vat::where('codiva', 'OLD')->delete();</code></pre>
                                </div>
	                            </div>
	                        </section>

	                        {{-- Dirty Tracking --}}
	                        <section id="dirty-tracking" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
	                            <div class="prose prose-zinc dark:prose-invert max-w-none">
	                                <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
	                                    <flux:icon.sparkles class="size-6 text-purple-500" />
	                                    Dirty Tracking (Mudanças em Memória)
	                                </h2>
	                                <p class="text-zinc-700 dark:text-zinc-300">
	                                    O <code>Model</code> mantém um snapshot dos valores originais vindos do banco. Isso permite detectar alterações em memória
	                                    e garantir que o <code>save()</code> atualize apenas os campos realmente modificados.
	                                </p>
	                            </div>

	                            <div class="flex gap-4">
	                                <div class="flex-1 min-w-0 rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
	                                    <h3 class="font-mono text-sm font-bold text-purple-600 mb-2">Estados</h3>
	                                    <pre class="language-php text-xs rounded-lg bg-zinc-950 p-3 overflow-x-auto"><code>$vat = Vat::find('NOR');

// Modelo vindo do banco
$vat->exists(); // true

// Snapshot do que veio do banco
$vat->getOriginal(); // array

// Mudanças desde o último syncOriginal()
$vat->getDirty(); // array
$vat->isDirty();  // bool
$vat->isClean();  // bool</code></pre>
	                                </div>

	                                <div class="flex-1 min-w-0 rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
	                                    <h3 class="font-mono text-sm font-bold text-purple-600 mb-2">Mudanças e Reversão</h3>
	                                    <pre class="language-php text-xs rounded-lg bg-zinc-950 p-3 overflow-x-auto"><code>$vat = Vat::find('NOR');

$vat->iva = '9.99';

// Lista apenas os campos alterados
$vat->getDirty(); // ['iva' =&gt; '9.99']

// Mudanças com original + current
$vat->getChanges(); // ['iva' =&gt; ['original' =&gt; '23.00', 'current' =&gt; '9.99']]

// Desfaz alterações em memória
$vat->discardChanges();</code></pre>
	                                </div>
	                            </div>

	                            <div class="rounded-lg border border-blue-200 bg-blue-50 p-4 dark:border-blue-900/30 dark:bg-blue-900/10">
	                                <p class="font-semibold text-blue-800 dark:text-blue-300">💡 Por que isso importa?</p>
	                                <p class="text-sm text-blue-700 dark:text-blue-400 mt-1">
	                                    Em produção, isso reduz writes desnecessários, evita atualizar colunas sem intenção e simplifica auditoria (só muda o que foi alterado).
	                                </p>
	                            </div>
	                        </section>

	                        {{-- Mass Assignment --}}
	                        <section id="mass-assignment" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
	                            <div class="prose prose-zinc dark:prose-invert max-w-none">
	                                <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
	                                    <flux:icon.shield-check class="size-6 text-red-500" />
                                    Mass Assignment Protection
                                </h2>
                            </div>

                            <div class="rounded-xl border border-zinc-200 bg-zinc-50 p-4 dark:border-zinc-800 dark:bg-zinc-900/30">
                                <pre class="language-php text-xs rounded-lg bg-zinc-950 p-3"><code>class User extends Model
{
    // Whitelist - campos permitidos em fill()
    protected array $fillable = ['name', 'email', 'password'];
    
    // Ou Blacklist - bloquear específicos
    protected array $guarded = ['id', 'is_admin', 'role'];
}</code></pre>
                            </div>

                            <div class="overflow-x-auto">
                                <table class="min-w-full text-sm">
                                    <thead class="bg-zinc-100 dark:bg-zinc-800">
                                        <tr>
                                            <th class="px-4 py-2 text-left font-semibold">Método</th>
                                            <th class="px-4 py-2 text-left font-semibold">Respeita $fillable</th>
                                            <th class="px-4 py-2 text-left font-semibold">Descrição</th>
                                        </tr>
                                    </thead>
                                    <tbody class="divide-y divide-zinc-200 dark:divide-zinc-700">
                                        <tr><td class="px-4 py-2 font-mono text-xs">fill($data)</td><td class="px-4 py-2">✅ Sim</td><td class="px-4 py-2">Ignora não-fillable silenciosamente</td></tr>
                                        <tr><td class="px-4 py-2 font-mono text-xs">fillOrFail($data)</td><td class="px-4 py-2">✅ Sim</td><td class="px-4 py-2">Lança exceção para não-fillable</td></tr>
                                        <tr><td class="px-4 py-2 font-mono text-xs">forceFill($data)</td><td class="px-4 py-2">❌ Não</td><td class="px-4 py-2">Bypass total (uso interno)</td></tr>
                                        <tr><td class="px-4 py-2 font-mono text-xs">update($data)</td><td class="px-4 py-2">✅ Sim</td><td class="px-4 py-2">fill() + save()</td></tr>
                                    </tbody>
                                </table>
                            </div>
                        </section>

                        {{-- OGDB Forge --}}
                        <section id="forge" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <div class="prose prose-zinc dark:prose-invert max-w-none">
                                <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                    <flux:icon.wrench-screwdriver class="size-6 text-amber-500" />
                                    OGDB Forge (Schema Builder)
                                </h2>
                                <p class="text-zinc-700 dark:text-zinc-300">
                                    O OGDB Forge fornece uma API de alto nível para operações de definição e evolução de esquema de banco de dados (DDL).
                                </p>
                            </div>

                            {{-- Create Table --}}
                            <div class="rounded-xl border border-zinc-200 bg-zinc-50 p-4 dark:border-zinc-800 dark:bg-zinc-900/30">
                                <h3 class="font-mono text-sm font-bold text-primary mb-2">Criar Tabela</h3>
                                <pre class="language-php text-xs rounded-lg bg-zinc-950 p-3"><code>$this->dbforge
    ->add_field([
        'id' => ['type' => 'INT', 'constraint' => 11, 'auto_increment' => true],
        'user_id' => ['type' => 'INT', 'constraint' => 11, 'null' => false],
        'email' => ['type' => 'VARCHAR', 'constraint' => 255, 'null' => false],
        'created_at' => ['type' => 'DATETIME', 'null' => true, 'default' => RawSql::currentTimestamp()],
    ])
    ->primary('id')
    ->unique('email')
    ->index(['user_id', 'created_at'])
    ->addForeignKey('user_id', 'users', 'id', 'CASCADE', 'CASCADE')
    ->create_table('user_emails', true);  // true = IF NOT EXISTS</code></pre>
                            </div>

                            {{-- Field Attributes --}}
                            <div class="overflow-x-auto">
                                <table class="min-w-full text-sm">
                                    <thead class="bg-zinc-100 dark:bg-zinc-800">
                                        <tr>
                                            <th class="px-4 py-2 text-left font-semibold">Atributo</th>
                                            <th class="px-4 py-2 text-left font-semibold">Tipo</th>
                                            <th class="px-4 py-2 text-left font-semibold">Descrição</th>
                                        </tr>
                                    </thead>
                                    <tbody class="divide-y divide-zinc-200 dark:divide-zinc-700">
                                        <tr><td class="px-4 py-2 font-mono text-xs">type</td><td class="px-4 py-2">string</td><td class="px-4 py-2">VARCHAR, INT, BIGINT, DATETIME, TEXT</td></tr>
                                        <tr><td class="px-4 py-2 font-mono text-xs">constraint</td><td class="px-4 py-2">int</td><td class="px-4 py-2">Tamanho/precisão (255, 11, etc.)</td></tr>
                                        <tr><td class="px-4 py-2 font-mono text-xs">null</td><td class="px-4 py-2">bool</td><td class="px-4 py-2">Permite NULL</td></tr>
                                        <tr><td class="px-4 py-2 font-mono text-xs">default</td><td class="px-4 py-2">mixed</td><td class="px-4 py-2">Valor padrão ou RawSql</td></tr>
                                        <tr><td class="px-4 py-2 font-mono text-xs">unsigned</td><td class="px-4 py-2">bool</td><td class="px-4 py-2">Para tipos numéricos</td></tr>
                                        <tr><td class="px-4 py-2 font-mono text-xs">auto_increment</td><td class="px-4 py-2">bool</td><td class="px-4 py-2">Incremento automático</td></tr>
                                    </tbody>
                                </table>
                            </div>

                            {{-- Patterns --}}
                            <div class="rounded-xl border border-zinc-200 bg-zinc-50 p-4 dark:border-zinc-800 dark:bg-zinc-900/30">
                                <h3 class="font-mono text-sm font-bold text-primary mb-2">Padrões de Campos (Atalhos)</h3>
                                <pre class="language-php text-xs rounded-lg bg-zinc-950 p-3"><code>// Timestamps: date_reg e date_alter com defaults
$this->dbforge->timestamps();

// Soft deletes: deleted_at + índice
$this->dbforge->softDeletes();

// User tracking: oper_reg e oper_alter com FKs
$this->dbforge->userTracking();</code></pre>
                            </div>

                            {{-- Add/Modify Columns --}}
                            <div class="space-y-4">
                                <div class="rounded-xl border border-zinc-200 bg-zinc-50 p-4 dark:border-zinc-800 dark:bg-zinc-900/30">
                                    <h3 class="font-mono text-sm font-bold text-primary mb-2">Adicionar Colunas</h3>
                                    <pre class="language-php text-xs rounded-lg bg-zinc-950 p-3"><code>$this->dbforge->addColumns(
    'orders',
    [
        'status' => [
            'type' => 'VARCHAR', 
            'constraint' => 30, 
            'default' => 'pending'
        ],
    ],
    null,
    true,  // verifyIfExists
    ['status' => 'index']
);</code></pre>
                                </div>

                                <div class="rounded-xl border border-zinc-200 bg-zinc-50 p-4 dark:border-zinc-800 dark:bg-zinc-900/30">
                                    <h3 class="font-mono text-sm font-bold text-primary mb-2">Modificar/Remover</h3>
                                    <pre class="language-php text-xs rounded-lg bg-zinc-950 p-3"><code>// Modificar coluna
$this->dbforge->modify_column('users', [
    'email' => [
        'name' => 'email', 
        'type' => 'VARCHAR', 
        'constraint' => 320
    ]
]);

// Remover coluna
$this->dbforge->drop_column('users', 'legacy');</code></pre>
                                </div>
                            </div>
                        </section>

                        {{-- Modernização --}}
                        <section id="modernizacao" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <div class="prose prose-zinc dark:prose-invert max-w-none">
                                <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                    <flux:icon.arrow-trending-up class="size-6 text-indigo-500" />
                                    Modernização: Legado vs. Moderno
                                </h2>
                            </div>

                            {{-- Comparison Table --}}
                            <div class="overflow-x-auto">
                                <table class="min-w-full text-sm">
                                    <thead class="bg-gradient-to-r from-zinc-800 to-zinc-900 text-white">
                                        <tr>
                                            <th class="px-4 py-3 text-left font-semibold">Métrica</th>
                                            <th class="px-4 py-3 text-left font-semibold">Legado</th>
                                            <th class="px-4 py-3 text-left font-semibold">Moderno</th>
                                            <th class="px-4 py-3 text-left font-semibold">Ganho</th>
                                        </tr>
                                    </thead>
                                    <tbody class="divide-y divide-zinc-200 dark:divide-zinc-700">
                                        <tr><td class="px-4 py-2 font-semibold">Linhas de código</td><td class="px-4 py-2">~170</td><td class="px-4 py-2">~80</td><td class="px-4 py-2 text-green-600 font-bold">-53%</td></tr>
                                        <tr><td class="px-4 py-2 font-semibold">Testabilidade</td><td class="px-4 py-2 text-red-600">❌ Difícil</td><td class="px-4 py-2 text-green-600">✅ Unit tests</td><td class="px-4 py-2 text-green-600 font-bold">100%</td></tr>
                                        <tr><td class="px-4 py-2 font-semibold">Type safety</td><td class="px-4 py-2 text-red-600">❌ Nenhuma</td><td class="px-4 py-2 text-green-600">✅ Forte</td><td class="px-4 py-2 text-green-600 font-bold">100%</td></tr>
                                        <tr><td class="px-4 py-2 font-semibold">Reutilização</td><td class="px-4 py-2 text-red-600">❌ Copy/paste</td><td class="px-4 py-2 text-green-600">✅ Components</td><td class="px-4 py-2 text-green-600 font-bold">Alta</td></tr>
                                        <tr><td class="px-4 py-2 font-semibold">API Docs</td><td class="px-4 py-2 text-red-600">❌ Manual</td><td class="px-4 py-2 text-green-600">✅ OpenAPI</td><td class="px-4 py-2 text-green-600 font-bold">Auto</td></tr>
                                    </tbody>
                                </table>
                            </div>

                            <div class="space-y-4">
                                {{-- Legacy --}}
                                <div class="rounded-xl border border-red-200 bg-red-50 p-4 dark:border-red-900/30 dark:bg-red-900/10">
                                    <h3 class="font-semibold text-red-800 dark:text-red-300 mb-2">❌ Código Legado</h3>
                                    <pre class="language-php text-xs rounded-lg bg-zinc-950 p-3"><code>// 170+ linhas, sem tipagem
public function list_report($request)
{
    global $db;  // Dependência global
    
    // Parsing manual
    if (Funcs::not_empty_verify($request['date'])) {
        $search['where']['date >='] = ...;
    }
    // +50 linhas de condicionais...
    
    // Formatação no loop
    foreach ($list as $k => $v) {
        $list[$k]['total_f'] = Funcs::number_format($v['total']);
    }
    
    return $list;  // Array sem tipo
}</code></pre>
                                </div>

                                {{-- Modern --}}
                                <div class="rounded-xl border border-green-200 bg-green-50 p-4 dark:border-green-900/30 dark:bg-green-900/10">
	                                    <h3 class="font-semibold text-green-800 dark:text-green-300 mb-2">✅ Código Moderno</h3>
	                                    <pre class="language-php text-xs rounded-lg bg-zinc-950 p-3"><code>readonly class HandlingList
	{
	    public function execute(
	        HandlingFilterDTO $filters
	    ): array {
	        $query = Handling::query();
	
	        if ($filters->statusIds) {
	            $query->whereIn('status_id', $filters->statusIds);
	        }
	
	        if ($filters->dateFrom) {
	            $query->where('date_reg', '>=', $filters->dateFrom);
	        }
	
	        return $query
	            ->limit($filters->limit)
	            ->asArray()
	            ->get();
	    }
	}</code></pre>
	                                </div>
                            </div>

                            <div class="rounded-lg border border-blue-200 bg-blue-50 p-4 dark:border-blue-900/30 dark:bg-blue-900/10">
                                <p class="font-semibold text-blue-800 dark:text-blue-300">💡 ROI Estimado</p>
                                <p class="text-sm text-blue-700 dark:text-blue-400 mt-1">Para cada hora investida na migração, economiza-se <strong>3-5 horas</strong> em manutenção futura e debugging.</p>
                            </div>
                        </section>

                        {{-- CLI make:model --}}
                        <section id="cli" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <div class="prose prose-zinc dark:prose-invert max-w-none">
                                <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                    <flux:icon.command-line class="size-6 text-zinc-500" />
                                    CLI: make:model
                                </h2>
                            </div>

                            <div class="rounded-xl bg-zinc-950 p-4 text-sm overflow-x-auto">
                                <pre class="language-bash"><code># Criar model básico
php og make:model User --module=Auth

# Criar model para tabela legada
php og make:model Vat --module=Tax --legacy --normalize --pk=CodIVA --table=ivas</code></pre>
                            </div>

                            <div class="overflow-x-auto">
                                <table class="min-w-full text-sm">
                                    <thead class="bg-zinc-100 dark:bg-zinc-800">
                                        <tr>
                                            <th class="px-4 py-2 text-left font-semibold">Opção</th>
                                            <th class="px-4 py-2 text-left font-semibold">Descrição</th>
                                        </tr>
                                    </thead>
                                    <tbody class="divide-y divide-zinc-200 dark:divide-zinc-700">
                                        <tr><td class="px-4 py-2 font-mono text-xs">--module=</td><td class="px-4 py-2">Módulo destino</td></tr>
                                        <tr><td class="px-4 py-2 font-mono text-xs">--addon=</td><td class="px-4 py-2">Addon destino</td></tr>
                                        <tr><td class="px-4 py-2 font-mono text-xs">--table=</td><td class="px-4 py-2">Nome da tabela</td></tr>
                                        <tr><td class="px-4 py-2 font-mono text-xs">--pk=</td><td class="px-4 py-2">Primary key</td></tr>
                                        <tr><td class="px-4 py-2 font-mono text-xs">--legacy</td><td class="px-4 py-2">Timestamps legados (date_reg/date_alter)</td></tr>
                                        <tr><td class="px-4 py-2 font-mono text-xs">--normalize</td><td class="px-4 py-2">Normalização de keys</td></tr>
                                    </tbody>
                                </table>
                            </div>
                        </section>

                        {{-- Timestamps Configuráveis --}}
                        <section id="timestamps" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <div class="prose prose-zinc dark:prose-invert max-w-none">
                                <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                    <flux:icon.clock class="size-6 text-cyan-500" />
                                    Timestamps Configuráveis
                                </h2>
                                <p class="text-zinc-700 dark:text-zinc-300">
                                    Controle flexível sobre colunas de timestamp para tabelas modernas e legadas.
                                </p>
                            </div>

                            <div class="space-y-4">
                                <div class="rounded-xl border border-zinc-200 bg-zinc-50 p-4 dark:border-zinc-800 dark:bg-zinc-900/30">
                                    <h3 class="font-mono text-sm font-bold text-primary mb-2">Tabelas Modernas</h3>
                                    <pre class="language-php text-xs rounded-lg bg-zinc-950 p-3"><code>class Order extends Model
{
    // Padrão: created_at e updated_at
    protected bool $timestamps = true;
    
    // Formato (opcional)
    protected string $timestampFormat = 'Y-m-d H:i:s';
}</code></pre>
                                </div>

                                <div class="rounded-xl border border-zinc-200 bg-zinc-50 p-4 dark:border-zinc-800 dark:bg-zinc-900/30">
                                    <h3 class="font-mono text-sm font-bold text-primary mb-2">Tabelas Legadas</h3>
                                    <pre class="language-php text-xs rounded-lg bg-zinc-950 p-3"><code>class Vat extends Model
{
    // Para tabelas com date_reg/date_alter
    protected string $createdAtColumn = 'date_reg';
    protected string $updatedAtColumn = 'date_alter';
}</code></pre>
                                </div>
                            </div>

                            <div class="rounded-xl border border-zinc-200 bg-zinc-50 p-4 dark:border-zinc-800 dark:bg-zinc-900/30">
                                <h3 class="font-mono text-sm font-bold text-primary mb-2">Desativar Timestamps</h3>
                                <pre class="language-php text-xs rounded-lg bg-zinc-950 p-3"><code>// Permanentemente no modelo
class LogEntry extends Model
{
    protected bool $timestamps = false;
}

// Temporariamente para uma operação
$model->withoutTimestamps()->save();</code></pre>
                            </div>
                        </section>

                        {{-- Visibilidade de Atributos --}}
                        <section id="visibilidade" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <div class="prose prose-zinc dark:prose-invert max-w-none">
                                <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                    <flux:icon.eye class="size-6 text-violet-500" />
                                    Visibilidade de Atributos
                                </h2>
                                <p class="text-zinc-700 dark:text-zinc-300">
                                    Controle quais atributos aparecem na serialização (<code>toArray()</code>, <code>toJson()</code>).
                                </p>
                            </div>

                            <div class="rounded-xl border border-zinc-200 bg-zinc-50 p-4 dark:border-zinc-800 dark:bg-zinc-900/30">
                                <pre class="language-php text-xs rounded-lg bg-zinc-950 p-3"><code>class User extends Model
{
    // Ocultar sempre (excluídos de toArray, retornam null no acesso)
    protected array $hidden = ['password', 'api_token', 'remember_token'];
    
    // Modo whitelist: SÓ estes aparecem em toArray()
    protected array $visible = ['id', 'name', 'email', 'avatar'];
    
    // Computed properties a incluir automaticamente
    protected array $appends = ['full_name', 'profile_url'];
}</code></pre>
                            </div>

                            <div class="space-y-4">
                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-mono text-sm font-bold text-primary mb-2">Alterações Temporárias</h3>
                                    <pre class="language-php text-xs rounded-lg bg-zinc-950 p-3"><code>// Tornar visível temporariamente
$user->makeVisible(['password'])->toArray();

// Ocultar temporariamente
$user->makeHidden(['email'])->toArray();</code></pre>
                                </div>

                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-mono text-sm font-bold text-primary mb-2">Acesso Interno</h3>
                                    <pre class="language-php text-xs rounded-lg bg-zinc-950 p-3"><code>// Acesso bloqueado (retorna null)
$user->password;  // null

// Acesso raw para uso interno
$user->getRawAttribute('password');  // valor real</code></pre>
                                </div>
                            </div>
                        </section>

                        {{-- Normalização de Keys --}}
                        <section id="normalizacao" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <div class="prose prose-zinc dark:prose-invert max-w-none">
                                <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                    <flux:icon.key class="size-6 text-amber-500" />
                                    Normalização de Keys Legadas
                                </h2>
                                <p class="text-zinc-700 dark:text-zinc-300">
                                    Tabelas antigas podem ter colunas duplicadas com case diferente (ex: <code>SAFTTaxCode</code> e <code>safttaxcode</code>). O Model normaliza automaticamente.
                                </p>
                            </div>

                            <div class="rounded-xl border border-zinc-200 bg-zinc-50 p-4 dark:border-zinc-800 dark:bg-zinc-900/30">
                                <pre class="language-php text-xs rounded-lg bg-zinc-950 p-3"><code>class Vat extends Model
{
    // Opções: null (desativado), 'lower', 'snake'
    protected ?string $normalizeKeys = 'lower';
}

// SAFTTaxCode e safttaxcode → apenas safttaxcode
// Evita duplicação no toArray()</code></pre>
                            </div>

                            <div class="overflow-x-auto">
                                <table class="min-w-full text-sm">
                                    <thead class="bg-zinc-100 dark:bg-zinc-800">
                                        <tr>
                                            <th class="px-4 py-2 text-left font-semibold">Opção</th>
                                            <th class="px-4 py-2 text-left font-semibold">Input</th>
                                            <th class="px-4 py-2 text-left font-semibold">Resultado</th>
                                        </tr>
                                    </thead>
                                    <tbody class="divide-y divide-zinc-200 dark:divide-zinc-700">
                                        <tr><td class="px-4 py-2 font-mono text-xs">null</td><td class="px-4 py-2">SAFTTaxCode</td><td class="px-4 py-2">SAFTTaxCode</td></tr>
                                        <tr><td class="px-4 py-2 font-mono text-xs">'lower'</td><td class="px-4 py-2">SAFTTaxCode</td><td class="px-4 py-2">safttaxcode</td></tr>
                                        <tr><td class="px-4 py-2 font-mono text-xs">'snake'</td><td class="px-4 py-2">SAFTTaxCode</td><td class="px-4 py-2">s_a_f_t_tax_code</td></tr>
                                    </tbody>
                                </table>
                            </div>
                        </section>

                        {{-- Interfaces --}}
                        <section id="interfaces" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <div class="prose prose-zinc dark:prose-invert max-w-none">
                                <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                    <flux:icon.puzzle-piece class="size-6 text-pink-500" />
                                    Interfaces Implementadas
                                </h2>
                            </div>

                            <div class="space-y-4">
                                <div class="rounded-xl border border-zinc-200 bg-zinc-50 p-4 dark:border-zinc-800 dark:bg-zinc-900/30">
                                    <h3 class="font-semibold text-zinc-900 dark:text-white mb-2">QueueableEntity</h3>
                                    <p class="text-sm text-zinc-600 dark:text-zinc-400 mb-3">Serialização eficiente em jobs de queue. Apenas o ID é guardado.</p>
                                    <pre class="language-php text-xs rounded-lg bg-zinc-950 p-3"><code>// Dispatch com modelo
Queue::dispatch(new ProcessOrderJob($order));

// No job, carrega automaticamente do banco
class ProcessOrderJob {
    public function __construct(
        private Order $order  // Só ID serializado
    ) {}
}</code></pre>
                                </div>

                                <div class="rounded-xl border border-zinc-200 bg-zinc-50 p-4 dark:border-zinc-800 dark:bg-zinc-900/30">
                                    <h3 class="font-semibold text-zinc-900 dark:text-white mb-2">UrlRoutable</h3>
                                    <p class="text-sm text-zinc-600 dark:text-zinc-400 mb-3">Route Model Binding — converte parâmetros de URL em modelos.</p>
                                    <pre class="language-php text-xs rounded-lg bg-zinc-950 p-3"><code>// Route: /orders/{order}
Route::get('/orders/{order}', [OrderController::class, 'show']);

// Controller - $order já carregado!
public function show(Order $order): Response
{
    return view('orders.show', compact('order'));
}</code></pre>
                                </div>
                            </div>

                            <div class="rounded-xl border border-zinc-200 bg-zinc-50 p-4 dark:border-zinc-800 dark:bg-zinc-900/30">
                                <h3 class="font-semibold text-zinc-900 dark:text-white mb-2">ArrayAccess</h3>
                                <p class="text-sm text-zinc-600 dark:text-zinc-400 mb-3">Acesso estilo array para compatibilidade com código legado.</p>
                                <pre class="language-php text-xs rounded-lg bg-zinc-950 p-3"><code>$user = User::query()->find(1);

// Acesso por array (equivalente a ->name)
echo $user['name'];
$user['email'] = 'new@example.com';

// Verificação
isset($user['name']);  // true
unset($user['temp']);  // remove atributo</code></pre>
                            </div>
                        </section>

                        {{-- Entidades Ricas (DDD) --}}
                        <section id="ddd" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <div class="prose prose-zinc dark:prose-invert max-w-none">
                                <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                    <flux:icon.building-office class="size-6 text-emerald-500" />
                                    Entidades Ricas (DDD)
                                </h2>
                                <p class="text-zinc-700 dark:text-zinc-300">
                                    Evolução de modelos anémicos (só dados) para entidades ricas com comportamento de negócio encapsulado.
                                </p>
                            </div>

                            <div class="space-y-4">
                                <div class="rounded-xl border border-red-200 bg-red-50 p-4 dark:border-red-900/30 dark:bg-red-900/10">
                                    <h3 class="font-semibold text-red-800 dark:text-red-300 mb-2">❌ Modelo Anémico</h3>
                                    <pre class="language-php text-xs rounded-lg bg-zinc-950 p-3"><code>// Só getters, sem lógica
class Order extends Model {}

// Lógica espalhada pelo código
$total = $order->subtotal + $order->tax;
if ($order->status === 'pending') { ... }</code></pre>
                                </div>

                                <div class="rounded-xl border border-green-200 bg-green-50 p-4 dark:border-green-900/30 dark:bg-green-900/10">
                                    <h3 class="font-semibold text-green-800 dark:text-green-300 mb-2">✅ Entidade Rica</h3>
                                    <pre class="language-php text-xs rounded-lg bg-zinc-950 p-3"><code>class Order extends Model
{
    #[Computed]
    protected function total(): float {
        return $this->subtotal + $this->tax;
    }
    
    public function isPending(): bool {
        return $this->status === 'pending';
    }
}</code></pre>
                                </div>
                            </div>

                            <div class="rounded-xl border border-zinc-200 bg-zinc-50 p-4 dark:border-zinc-800 dark:bg-zinc-900/30">
                                <h3 class="font-mono text-sm font-bold text-primary mb-2">Exemplo Completo: Entidade Rica</h3>
                                <pre class="language-php text-xs rounded-lg bg-zinc-950 p-3"><code>class Order extends Model
{
    #[Computed]
    protected function total(): float {
        return $this->subtotal + $this->tax - $this->discount;
    }
    
    public function isPending(): bool { return $this->status === 'pending'; }
    public function isPaid(): bool { return $this->status === 'paid'; }
    public function canBeCancelled(): bool {
        return in_array($this->status, ['pending', 'confirmed']);
    }
    
    public function confirm(): void {
        if (!$this->isPending()) {
            throw new \DomainException('Só pedidos pendentes podem ser confirmados');
        }
        $this->status = 'confirmed';
        $this->confirmed_at = now();
        $this->save();
    }
    
    public function cancel(string $reason): void {
        if (!$this->canBeCancelled()) {
            throw new \DomainException('Este pedido não pode ser cancelado');
        }
        $this->status = 'cancelled';
        $this->cancellation_reason = $reason;
        $this->save();
    }
}</code></pre>
                            </div>

                            <div class="rounded-lg border border-blue-200 bg-blue-50 p-4 dark:border-blue-900/30 dark:bg-blue-900/10">
                                <p class="font-semibold text-blue-800 dark:text-blue-300">📈 Evolução Gradual</p>
                                <p class="text-sm text-blue-700 dark:text-blue-400 mt-1">
                                    <strong>Etapa 1:</strong> Arrays (legado) → <strong>Etapa 2:</strong> Modelos tipados (atual) → <strong>Etapa 3:</strong> Entidades ricas + Value Objects (futuro)
                                </p>
                            </div>
                        </section>

                        {{-- Navigation --}}
                        <div class="flex justify-between items-center pt-8 border-t border-zinc-200 dark:border-zinc-800">
                            <a href="{{ route('docs.show', 'cache') }}" class="inline-flex items-center gap-2 text-sm text-zinc-600 dark:text-zinc-400 hover:text-primary">
                                <flux:icon.arrow-left class="size-4" />
                                Cache
                            </a>
                            <a href="{{ route('docs.show', 'validation') }}" class="inline-flex items-center gap-2 text-sm font-medium text-primary hover:underline">
                                Validação
                                <flux:icon.arrow-right class="size-4" />
                            </a>
                        </div>
                    </div>

                </div>
            </div>

            {{-- Right Sidebar: Table of Contents --}}
            @include('docs.partials.toc', ['sections' => [
                'visao-geral' => 'Visão Geral',
                'modelos' => 'Modelos de Dados',
                'casting' => 'Sistema de Casting',
                'accessors' => 'Accessors e Mutators',
                'builder' => 'Query Builder',
                'crud' => 'CRUD Operations',
                'mass-assignment' => 'Mass Assignment',
                'timestamps' => 'Timestamps',
                'visibilidade' => 'Visibilidade',
                'normalizacao' => 'Normalização de Keys',
                'interfaces' => 'Interfaces',
                'ddd' => 'Entidades Ricas (DDD)',
                'forge' => 'OGDB Forge',
                'modernizacao' => 'Modernização',
                'cli' => 'CLI: make:model',
            ]])
        </div>
    </div>
@endsection
