@extends('layouts.docs')

@section('title', 'Resources (Output DTOs) - OG Framework')

@section('body')
    <div class="relative min-h-screen bg-transparent px-4 pb-12 pt-6 sm:px-6 lg:px-8">
        {{-- Floating shapes for styling --}}
        <div class="floating-blur cyan -left-10 -top-10 hidden lg:block"></div>
        <div class="floating-blur purple right-10 top-32 hidden lg:block"></div>

        <div class="mx-auto flex w-full max-w-7xl gap-8 bg-transparent">
            {{-- Left Sidebar: Navigation --}}
            @include('docs.partials.sidebar')

            {{-- Main Content --}}
            <div class="flex-1 min-w-0">
                <div class="relative overflow-hidden rounded-3xl border border-white/50 bg-white/85 p-8 shadow-xl shadow-primary/20 backdrop-blur dark:border-zinc-800 dark:bg-zinc-900/80">

                    {{-- Page Header --}}
                    <div class="mb-8">
                        <a href="{{ route('docs.index') }}" class="inline-flex items-center gap-1 text-sm text-primary hover:underline mb-4">
                            <flux:icon.arrow-left class="size-4" />
                            Voltar para Documentação
                        </a>
                        <p class="text-xs uppercase tracking-[0.3em] text-primary dark:text-blue-200">API</p>
                        <h1 class="font-display text-4xl font-bold text-zinc-900 dark:text-white sm:text-5xl">Resources (Output DTOs)</h1>
                        <p class="mt-3 text-base text-zinc-700 dark:text-zinc-200 max-w-3xl">
                            O sistema de Resources implementa o padrão <strong>Resource/Transformer</strong> para serialização consistente de dados em APIs. Transforma dados brutos de entidades, base de dados e arrays numa representação JSON estruturada e previsível.
                        </p>
                    </div>

                    <div class="space-y-10">

                        {{-- Architecture Diagram --}}
                        <section id="arquitectura" class="scroll-mt-28 space-y-4">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.cube-transparent class="size-6 text-sky-500" />
                                Arquitectura do Sistema
                            </h2>
                            <p class="text-zinc-700 dark:text-zinc-300">
                                O fluxo de transformação passa pelos seguintes componentes:
                            </p>

                            {{-- SVG: Data Flow Diagram --}}
                            <div class="rounded-xl border border-zinc-200 bg-white p-6 dark:border-zinc-800 dark:bg-zinc-900/50 overflow-x-auto">
                                <svg viewBox="0 0 800 200" class="w-full max-w-3xl mx-auto" xmlns="http://www.w3.org/2000/svg">
                                    {{-- Input Group --}}
                                    <g transform="translate(20,30)">
                                        <rect x="0" y="0" width="160" height="140" rx="12" fill="#f8fafc" stroke="#e2e8f0" stroke-width="2"/>
                                        <text x="80" y="20" text-anchor="middle" class="text-xs font-semibold" fill="#64748b">Dados de Entrada</text>
                                        
                                        {{-- DB --}}
                                        <ellipse cx="40" cy="55" rx="30" ry="18" fill="#dbeafe" stroke="#3b82f6" stroke-width="1.5"/>
                                        <text x="40" y="60" text-anchor="middle" class="text-[10px]" fill="#1e40af">Base de Dados</text>
                                        
                                        {{-- Entity --}}
                                        <rect x="80" y="70" width="70" height="28" rx="4" fill="#fef3c7" stroke="#f59e0b" stroke-width="1.5"/>
                                        <text x="115" y="88" text-anchor="middle" class="text-[10px]" fill="#92400e">Entity/Model</text>
                                        
                                        {{-- Array --}}
                                        <rect x="10" y="100" width="60" height="28" rx="4" fill="#f3e8ff" stroke="#a855f7" stroke-width="1.5"/>
                                        <text x="40" y="118" text-anchor="middle" class="text-[10px]" fill="#7e22ce">Array</text>
                                    </g>
                                    
                                    {{-- Arrows to DTO --}}
                                    <path d="M190,100 L280,100" stroke="#94a3b8" stroke-width="2" marker-end="url(#arrowhead)"/>
                                    
                                    {{-- Output DTO Box --}}
                                    <g transform="translate(290,25)">
                                        <rect x="0" y="0" width="220" height="150" rx="12" fill="#e0f2fe" stroke="#0284c7" stroke-width="2"/>
                                        <text x="110" y="25" text-anchor="middle" class="text-sm font-bold" fill="#0369a1">Output DTO</text>
                                        
                                        <rect x="15" y="40" width="190" height="100" rx="8" fill="#ffffff" stroke="#bae6fd" stroke-width="1"/>
                                        <text x="110" y="60" text-anchor="middle" class="text-[11px] font-semibold" fill="#0c4a6e">JsonResource</text>
                                        <text x="110" y="78" text-anchor="middle" class="text-[9px]" fill="#64748b">+ ConditionalAttributesTrait</text>
                                        <text x="110" y="92" text-anchor="middle" class="text-[9px]" fill="#64748b">+ DateSerializerTrait</text>
                                        <text x="110" y="106" text-anchor="middle" class="text-[9px]" fill="#64748b">+ DelegatesToResource</text>
                                        <text x="110" y="120" text-anchor="middle" class="text-[9px]" fill="#64748b">+ InteractsWithType</text>
                                    </g>
                                    
                                    {{-- Arrow to Output --}}
                                    <path d="M520,100 L600,100" stroke="#94a3b8" stroke-width="2" marker-end="url(#arrowhead)"/>
                                    
                                    {{-- JSON Output --}}
                                    <g transform="translate(610,35)">
                                        <rect x="0" y="0" width="160" height="130" rx="12" fill="#f0fdf4" stroke="#22c55e" stroke-width="2"/>
                                        <text x="80" y="22" text-anchor="middle" class="text-xs font-semibold" fill="#166534">Saída JSON</text>
                                        <rect x="15" y="35" width="130" height="85" rx="6" fill="#ffffff" stroke="#86efac" stroke-width="1"/>
                                        <text x="30" y="55" class="text-[10px] font-mono" fill="#166534">{</text>
                                        <text x="40" y="70" class="text-[10px] font-mono" fill="#166534">"id": 1,</text>
                                        <text x="40" y="85" class="text-[10px] font-mono" fill="#166534">"name": "João",</text>
                                        <text x="40" y="100" class="text-[10px] font-mono" fill="#166534">"email": "..."</text>
                                        <text x="30" y="115" class="text-[10px] font-mono" fill="#166534">}</text>
                                    </g>
                                    
                                    {{-- Arrowhead marker --}}
                                    <defs>
                                        <marker id="arrowhead" markerWidth="10" markerHeight="7" refX="9" refY="3.5" orient="auto">
                                            <polygon points="0 0, 10 3.5, 0 7" fill="#94a3b8"/>
                                        </marker>
                                    </defs>
                                </svg>
                            </div>

                            {{-- Design Patterns Table --}}
                            <div class="overflow-hidden rounded-xl border border-zinc-200 dark:border-zinc-800 bg-white dark:bg-zinc-900/50 mt-6">
                                <table class="min-w-full divide-y divide-zinc-200 dark:divide-zinc-800">
                                    <thead class="bg-zinc-50 dark:bg-zinc-900/80">
                                        <tr>
                                            <th class="px-6 py-3 text-left text-xs font-medium text-zinc-500 uppercase">Padrão</th>
                                            <th class="px-6 py-3 text-left text-xs font-medium text-zinc-500 uppercase">Implementação</th>
                                        </tr>
                                    </thead>
                                    <tbody class="divide-y divide-zinc-200 dark:divide-zinc-800">
                                        <tr>
                                            <td class="px-6 py-3 font-semibold text-xs text-zinc-700 dark:text-zinc-300">Resource Pattern</td>
                                            <td class="px-6 py-3 text-xs text-zinc-600 dark:text-zinc-400">Classe base <code>JsonResource</code> com método <code>toArray()</code></td>
                                        </tr>
                                        <tr>
                                            <td class="px-6 py-3 font-semibold text-xs text-zinc-700 dark:text-zinc-300">Factory Pattern</td>
                                            <td class="px-6 py-3 text-xs text-zinc-600 dark:text-zinc-400">Métodos estáticos <code>make()</code> e <code>collection()</code></td>
                                        </tr>
                                        <tr>
                                            <td class="px-6 py-3 font-semibold text-xs text-zinc-700 dark:text-zinc-300">Decorator Pattern</td>
                                            <td class="px-6 py-3 text-xs text-zinc-600 dark:text-zinc-400">Traits adicionam comportamentos (validação, serialização)</td>
                                        </tr>
                                        <tr>
                                            <td class="px-6 py-3 font-semibold text-xs text-zinc-700 dark:text-zinc-300">Singleton Pattern</td>
                                            <td class="px-6 py-3 text-xs text-zinc-600 dark:text-zinc-400"><code>MissingValue::getInstance()</code> para valores ausentes</td>
                                        </tr>
                                    </tbody>
                                </table>
                            </div>
                        </section>

                        {{-- JsonResource Class --}}
                        <section id="json-resource" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.code-bracket class="size-6 text-blue-500" />
                                Classe Base: JsonResource
                            </h2>
                            <p class="text-zinc-700 dark:text-zinc-300">
                                A classe <code>JsonResource</code> é o coração do sistema, fornecendo a estrutura base para transformação de dados.
                            </p>

                            <div class="rounded-xl bg-zinc-950 p-4 text-sm overflow-x-auto">
                                <pre class="language-php"><code>use Og\Modules\Common\Resource\JsonResource;

final class UserOutputDTO extends JsonResource
{
    public function toArray(): array
    {
        return [
            'id' => $this->codempr,
            'email' => $this->email,
            'name' => $this->nome,
            'short_name' => $this->nomeabreviado,
        ];
    }
}

// Uso
$result = UserOutputDTO::make($userData);
// Resultado: ['id' => 1, 'email' => 'joao@ex.com', ...]</code></pre>
                            </div>

                            <div class="grid gap-4 md:grid-cols-2 mt-4">
                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-bold text-zinc-900 dark:text-white mb-2 text-sm flex items-center gap-2">
                                        <flux:icon.document class="size-4 text-blue-500" />
                                        make($data)
                                    </h3>
                                    <p class="text-xs text-zinc-600 dark:text-zinc-400">
                                        Transforma um único objecto/array. Retorna array vazio se <code>$data</code> for null.
                                    </p>
                                </div>
                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-bold text-zinc-900 dark:text-white mb-2 text-sm flex items-center gap-2">
                                        <flux:icon.rectangle-stack class="size-4 text-green-500" />
                                        collection($data)
                                    </h3>
                                    <p class="text-xs text-zinc-600 dark:text-zinc-400">
                                        Transforma listas. Detecta paginação automaticamente e inclui metadata.
                                    </p>
                                </div>
                            </div>
                        </section>

                        {{-- Conditional Fields --}}
                        <section id="campos-condicionais" class="scroll-mt-28 space-y-6 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.adjustments-horizontal class="size-6 text-purple-500" />
                                Campos Condicionais
                            </h2>
                            <p class="text-zinc-700 dark:text-zinc-300">
                                O trait <code>ConditionalAttributesTrait</code> permite controlar quais campos aparecem na saída JSON.
                            </p>

                            {{-- whenHas --}}
                            <div class="space-y-4">
                                <h3 class="font-bold text-lg text-zinc-900 dark:text-white border-b border-zinc-200 dark:border-zinc-700 pb-2">
                                    whenHas() — Campo Existe e Tem Valor
                                </h3>
                                <div class="rounded-xl bg-zinc-950 p-4 text-sm overflow-x-auto">
                                    <pre class="language-php"><code>public function toArray(): array
{
    return [
        // Sintaxe simples
        'name' => $this->whenHas('name'),
        
        // Com callback de transformação
        'created_at' => $this->whenHas('created_at', 
            fn ($v) => Carbon::parse($v)->toIso8601String()
        ),
        
        // Dot notation para dados aninhados
        'city' => $this->whenHas('address.city'),
        
        // DTO aninhado com lazy loading
        'customer' => $this->whenHas('customer', 
            fn () => CustomerOutputDTO::make($this->customer)
        ),
        
        // Com valor default
        'status' => $this->whenHas('status', null, 'active'),
    ];
}</code></pre>
                                </div>

                                <div class="overflow-hidden rounded-xl border border-zinc-200 dark:border-zinc-800 bg-white dark:bg-zinc-900/50">
                                    <table class="min-w-full divide-y divide-zinc-200 dark:divide-zinc-800">
                                        <thead class="bg-zinc-50 dark:bg-zinc-900/80">
                                            <tr>
                                                <th class="px-6 py-3 text-left text-xs font-medium text-zinc-500">Entrada</th>
                                                <th class="px-6 py-3 text-left text-xs font-medium text-zinc-500">Resultado</th>
                                                <th class="px-6 py-3 text-left text-xs font-medium text-zinc-500">Campo na Saída?</th>
                                            </tr>
                                        </thead>
                                        <tbody class="divide-y divide-zinc-200 dark:divide-zinc-800">
                                            <tr>
                                                <td class="px-6 py-2 font-mono text-xs">['name' => 'João']</td>
                                                <td class="px-6 py-2 font-mono text-xs text-green-600">'João'</td>
                                                <td class="px-6 py-2 text-xs">✅ Sim</td>
                                            </tr>
                                            <tr>
                                                <td class="px-6 py-2 font-mono text-xs">['name' => '']</td>
                                                <td class="px-6 py-2 font-mono text-xs text-red-600">MissingValue</td>
                                                <td class="px-6 py-2 text-xs">❌ Não</td>
                                            </tr>
                                            <tr>
                                                <td class="px-6 py-2 font-mono text-xs">['name' => null]</td>
                                                <td class="px-6 py-2 font-mono text-xs text-red-600">MissingValue</td>
                                                <td class="px-6 py-2 text-xs">❌ Não</td>
                                            </tr>
                                            <tr>
                                                <td class="px-6 py-2 font-mono text-xs">[] (ausente)</td>
                                                <td class="px-6 py-2 font-mono text-xs text-red-600">MissingValue</td>
                                                <td class="px-6 py-2 text-xs">❌ Não</td>
                                            </tr>
                                        </tbody>
                                    </table>
                                </div>
                            </div>

                            {{-- when --}}
                            <div class="space-y-4">
                                <h3 class="font-bold text-lg text-zinc-900 dark:text-white border-b border-zinc-200 dark:border-zinc-700 pb-2">
                                    when() — Condição Booleana Livre
                                </h3>
                                <div class="rounded-xl bg-zinc-950 p-4 text-sm overflow-x-auto">
                                    <pre class="language-php"><code>return [
    'id' => $this->id,
    
    // Só aparece se documento estiver anulado
    'cancelReason' => $this->when(
        $this->resource['anulado'] === 'T',
        $this->resource['cancel_reason']
    ),
    
    // Só aparece para admins
    'adminNotes' => $this->when(
        auth()->user()?->isAdmin(),
        fn () => $this->fetchAdminNotes()
    ),
    
    // Com valor default
    'badge' => $this->when($this->score >= 100, 'gold', 'standard'),
];</code></pre>
                                </div>
                            </div>

                            {{-- merge --}}
                            <div class="space-y-4">
                                <h3 class="font-bold text-lg text-zinc-900 dark:text-white border-b border-zinc-200 dark:border-zinc-700 pb-2">
                                    mergeWhen() — Múltiplos Campos Condicionais
                                </h3>
                                <div class="rounded-xl border border-amber-200 bg-amber-50 p-4 dark:border-amber-900/50 dark:bg-amber-900/20">
                                    <p class="text-xs text-amber-800 dark:text-amber-200 mb-3">
                                        <strong>Importante:</strong> O operador spread (<code>...</code>) é <strong>obrigatório</strong> antes de <code>merge()</code> e <code>mergeWhen()</code>.
                                    </p>
                                    <pre class="text-xs bg-white dark:bg-black/20 p-2 rounded"><code>return [
    'id' => $this->id,
    
    // Mescla múltiplos campos se for admin
    ...$this->mergeWhen($this->isAdmin, [
        'email' => $this->email,
        'permissions' => $this->permissions,
        'last_login' => $this->last_login,
    ]),
    
    // Com closure para lazy loading
    ...$this->mergeWhen($this->includeFinancials, fn () => [
        'total_spent' => $this->orders()->sum('total'),
        'credit_limit' => $this->credit_limit,
    ]),
];</code></pre>
                                </div>
                            </div>
                        </section>

                        {{-- Type Validation --}}
                        <section id="validacao-tipos" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.shield-check class="size-6 text-green-500" />
                                Validação de Tipos
                            </h2>
                            <p class="text-zinc-700 dark:text-zinc-300">
                                O sistema inclui validação robusta de tipos para garantir integridade de dados na saída.
                            </p>

                            {{-- SVG: Type Validation Hierarchy --}}
                            <div class="rounded-xl border border-zinc-200 bg-white p-6 dark:border-zinc-800 dark:bg-zinc-900/50 overflow-x-auto">
                                <svg viewBox="0 0 600 280" class="w-full max-w-2xl mx-auto" xmlns="http://www.w3.org/2000/svg">
                                    {{-- Main coordinator --}}
                                    <rect x="200" y="10" width="200" height="50" rx="8" fill="#dbeafe" stroke="#3b82f6" stroke-width="2"/>
                                    <text x="300" y="30" text-anchor="middle" class="text-[11px] font-bold" fill="#1e40af">InteractsWithType</text>
                                    <text x="300" y="45" text-anchor="middle" class="text-[9px]" fill="#3b82f6">(coordinador)</text>
                                    
                                    {{-- Lines --}}
                                    <line x1="300" y1="60" x2="300" y2="90" stroke="#94a3b8" stroke-width="2"/>
                                    <line x1="100" y1="90" x2="500" y2="90" stroke="#94a3b8" stroke-width="2"/>
                                    <line x1="100" y1="90" x2="100" y2="110" stroke="#94a3b8" stroke-width="2"/>
                                    <line x1="233" y1="90" x2="233" y2="110" stroke="#94a3b8" stroke-width="2"/>
                                    <line x1="366" y1="90" x2="366" y2="110" stroke="#94a3b8" stroke-width="2"/>
                                    <line x1="500" y1="90" x2="500" y2="110" stroke="#94a3b8" stroke-width="2"/>
                                    
                                    {{-- Primitive Types --}}
                                    <rect x="20" y="110" width="160" height="80" rx="8" fill="#fef3c7" stroke="#f59e0b" stroke-width="1.5"/>
                                    <text x="100" y="130" text-anchor="middle" class="text-[10px] font-semibold" fill="#92400e">ValidatesPrimitiveTypes</text>
                                    <text x="100" y="150" text-anchor="middle" class="text-[9px]" fill="#a16207">isInteger(), isString(),</text>
                                    <text x="100" y="165" text-anchor="middle" class="text-[9px]" fill="#a16207">isBoolean(), isArray(),</text>
                                    <text x="100" y="180" text-anchor="middle" class="text-[9px]" fill="#a16207">isFloat(), isNumeric()</text>
                                    
                                    {{-- Nullable Types --}}
                                    <rect x="153" y="110" width="160" height="80" rx="8" fill="#f3e8ff" stroke="#a855f7" stroke-width="1.5"/>
                                    <text x="233" y="130" text-anchor="middle" class="text-[10px] font-semibold" fill="#7e22ce">ValidatesNullableTypes</text>
                                    <text x="233" y="150" text-anchor="middle" class="text-[9px]" fill="#9333ea">isNullOrInteger(),</text>
                                    <text x="233" y="165" text-anchor="middle" class="text-[9px]" fill="#9333ea">isNullOrString(),</text>
                                    <text x="233" y="180" text-anchor="middle" class="text-[9px]" fill="#9333ea">isNullOrBoolean()</text>
                                    
                                    {{-- Class Types --}}
                                    <rect x="286" y="110" width="160" height="80" rx="8" fill="#dcfce7" stroke="#22c55e" stroke-width="1.5"/>
                                    <text x="366" y="130" text-anchor="middle" class="text-[10px] font-semibold" fill="#166534">ValidatesClassTypes</text>
                                    <text x="366" y="150" text-anchor="middle" class="text-[9px]" fill="#15803d">isDateTime(), isCarbon(),</text>
                                    <text x="366" y="165" text-anchor="middle" class="text-[9px]" fill="#15803d">isCollection(),</text>
                                    <text x="366" y="180" text-anchor="middle" class="text-[9px]" fill="#15803d">isEnum()</text>
                                    
                                    {{-- Advanced Types --}}
                                    <rect x="420" y="110" width="160" height="80" rx="8" fill="#fee2e2" stroke="#ef4444" stroke-width="1.5"/>
                                    <text x="500" y="130" text-anchor="middle" class="text-[10px] font-semibold" fill="#991b1b">ValidatesAdvancedTypes</text>
                                    <text x="500" y="150" text-anchor="middle" class="text-[9px]" fill="#b91c1c">isArrayOf(), isEmail(),</text>
                                    <text x="500" y="165" text-anchor="middle" class="text-[9px]" fill="#b91c1c">isUrl(), isUuid(),</text>
                                    <text x="500" y="180" text-anchor="middle" class="text-[9px]" fill="#b91c1c">isPositiveInteger()</text>
                                    
                                    {{-- Example usage --}}
                                    <rect x="100" y="210" width="400" height="60" rx="8" fill="#f8fafc" stroke="#cbd5e1" stroke-width="1"/>
                                    <text x="300" y="230" text-anchor="middle" class="text-[10px] font-mono" fill="#475569">'user_id' => $this->isInteger()->whenHas('user_id')</text>
                                    <text x="300" y="250" text-anchor="middle" class="text-[10px] font-mono" fill="#475569">'email' => $this->isEmail('email')</text>
                                </svg>
                            </div>

                            <div class="grid gap-4 md:grid-cols-2">
                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-bold text-zinc-900 dark:text-white mb-2 text-sm">Tipos Primitivos</h3>
                                    <pre class="text-xs bg-zinc-950 p-3 rounded-lg text-zinc-300"><code>'id' => $this->isInteger('id'),
'name' => $this->isString('name'),
'active' => $this->isBoolean('active'),
'settings' => $this->isArray('settings'),</code></pre>
                                </div>

                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-bold text-zinc-900 dark:text-white mb-2 text-sm">Tipos Avançados</h3>
                                    <pre class="text-xs bg-zinc-950 p-3 rounded-lg text-zinc-300"><code>'tag_ids' => $this->isArrayOf('integer', 'tag_ids'),
'email' => $this->isEmail('email'),
'website' => $this->isNullOrUrl('website'),
'quantity' => $this->isPositiveInteger('quantity'),</code></pre>
                                </div>
                            </div>
                        </section>

                        {{-- Collections & Pagination --}}
                        <section id="coleccoes" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.rectangle-stack class="size-6 text-indigo-500" />
                                Colecções e Paginação
                            </h2>
                            <p class="text-zinc-700 dark:text-zinc-300">
                                O método <code>collection()</code> detecta automaticamente dados paginados e inclui metadata.
                            </p>

                            <div class="grid gap-6 lg:grid-cols-2">
                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-bold text-zinc-900 dark:text-white mb-2 text-sm">Colecção Simples</h3>
                                    <pre class="text-xs bg-zinc-950 p-3 rounded-lg text-zinc-300"><code>$users = [
    ['codempr' => 1, 'nome' => 'João'],
    ['codempr' => 2, 'nome' => 'Maria'],
];

UserOutputDTO::collection($users);
// [
//   ['id' => 1, 'name' => 'João'],
//   ['id' => 2, 'name' => 'Maria'],
// ]</code></pre>
                                </div>

                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-bold text-zinc-900 dark:text-white mb-2 text-sm">Com Paginação</h3>
                                    <pre class="text-xs bg-zinc-950 p-3 rounded-lg text-zinc-300"><code>$paginated = [
    'items' => [...],
    'per_page' => 10,
    'total' => 50,
    'current_page' => 1,
];

// Resultado inclui 'meta':
// {
//   "items": [...],
//   "meta": {
//     "current_page": 1,
//     "per_page": 10,
//     "total": 50,
//     ...
//   }
// }</code></pre>
                                </div>
                            </div>
                        </section>

                        {{-- Best Practices --}}
                        <section id="boas-praticas" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.light-bulb class="size-6 text-amber-500" />
                                Boas Práticas
                            </h2>

                            <div class="grid gap-4 md:grid-cols-2">
                                <div class="rounded-xl border border-green-200 bg-green-50 p-4 dark:border-green-900/50 dark:bg-green-900/20">
                                    <h3 class="font-semibold text-green-900 dark:text-green-200 mb-2 text-sm flex items-center gap-2">
                                        ✅ Correcto
                                    </h3>
                                    <pre class="text-xs bg-white dark:bg-black/20 p-2 rounded"><code>// Usar whenHas() para segurança
'name' => $this->whenHas('name'),

// DTO aninhado com lazy loading
'customer' => $this->whenHas('customer', 
    fn () => CustomerDTO::make($this->customer)
),</code></pre>
                                </div>

                                <div class="rounded-xl border border-red-200 bg-red-50 p-4 dark:border-red-900/50 dark:bg-red-900/20">
                                    <h3 class="font-semibold text-red-900 dark:text-red-200 mb-2 text-sm flex items-center gap-2">
                                        ❌ Evitar
                                    </h3>
                                    <pre class="text-xs bg-white dark:bg-black/20 p-2 rounded"><code>// Acesso directo pode falhar
'name' => $this->resource['name'],

// DTO sem lazy loading pode lançar erro
'customer' => CustomerDTO::make($this->customer),</code></pre>
                                </div>
                            </div>

                            <div class="rounded-xl border border-sky-200 bg-sky-50 p-4 dark:border-sky-900/50 dark:bg-sky-900/20">
                                <h3 class="font-semibold text-sky-900 dark:text-sky-200 mb-2 text-sm">Convenção de Nomenclatura</h3>
                                <ul class="text-xs text-sky-800 dark:text-sky-300 space-y-1 list-disc list-inside">
                                    <li><strong>Nome da classe:</strong> <code>{Domain}{Entity}OutputDTO</code> (ex: <code>CrmAppointmentOutputDTO</code>)</li>
                                    <li><strong>Namespace:</strong> <code>Og\Modules\{Module}\{Domain}\DTOs</code></li>
                                    <li><strong>Campos de saída:</strong> <code>snake_case</code> (ex: <code>employee_id</code>, <code>created_at</code>)</li>
                                </ul>
                            </div>
                        </section>

                        {{-- Navigation --}}
                        <div class="flex justify-between items-center pt-8 border-t border-zinc-200 dark:border-zinc-800">
                            <a href="{{ route('docs.show', 'helpers') }}" class="inline-flex items-center gap-2 text-sm text-zinc-600 dark:text-zinc-400 hover:text-primary">
                                <flux:icon.arrow-left class="size-4" />
                                Helpers
                            </a>
                            <a href="{{ route('docs.show', 'cache') }}" class="inline-flex items-center gap-2 text-sm font-medium text-primary hover:underline">
                                Cache System
                                <flux:icon.arrow-right class="size-4" />
                            </a>
                        </div>
                    </div>

                </div>
            </div>

            {{-- Right Sidebar: Table of Contents --}}
            @include('docs.partials.toc', ['sections' => [
                'arquitectura' => 'Arquitectura',
                'json-resource' => 'JsonResource',
                'campos-condicionais' => 'Campos Condicionais',
                'validacao-tipos' => 'Validação de Tipos',
                'coleccoes' => 'Colecções',
                'boas-praticas' => 'Boas Práticas',
            ]])
        </div>
    </div>
@endsection
