@extends('layouts.docs')

@section('title', 'Routing System - OG Framework')

@section('body')
    <div class="relative min-h-screen bg-transparent px-4 pb-12 pt-6 sm:px-6 lg:px-8">
        {{-- Floating shapes for styling --}}
        <div class="floating-blur cyan -left-10 -top-10 hidden lg:block"></div>
        <div class="floating-blur purple right-10 top-32 hidden lg:block"></div>

        <div class="mx-auto flex w-full max-w-7xl gap-8 bg-transparent">
            {{-- Left Sidebar: Navigation --}}
            @include('docs.partials.sidebar')

            {{-- Main Content --}}
            <div class="flex-1 min-w-0">
                <div class="relative overflow-hidden rounded-3xl border border-white/50 bg-white/85 p-8 shadow-xl shadow-primary/20 backdrop-blur dark:border-zinc-800 dark:bg-zinc-900/80">

                    {{-- Page Header --}}
                    <div class="mb-8">
                        <a href="{{ route('docs.index') }}" class="inline-flex items-center gap-1 text-sm text-primary hover:underline mb-4">
                            <flux:icon.arrow-left class="size-4" />
                            Voltar para Documentação
                        </a>
                        <p class="text-xs uppercase tracking-[0.3em] text-primary dark:text-blue-200">Core</p>
                        <h1 class="font-display text-4xl font-bold text-zinc-900 dark:text-white sm:text-5xl">Routing System</h1>
                        <p class="mt-3 text-base text-zinc-700 dark:text-zinc-200 max-w-3xl">
                            API fluente e expressiva para definição de endpoints. Mapeamento de URLs, grupos, middlewares e injeção de dependência automática.
                        </p>
                    </div>

                    <div class="space-y-10">
                        {{-- Introduction --}}
                        <section id="introducao" class="scroll-mt-28 space-y-4">
                            <div class="prose prose-zinc dark:prose-invert max-w-none">
                                <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                    <flux:icon.book-open class="size-6 text-orange-500" />
                                    Introdução
                                </h2>
                                <p class="text-zinc-700 dark:text-zinc-300">
                                    O sistema de rotas é a porta de entrada da aplicação. Mapeia URLs para Controllers, suportando gestão de middleware e validação de parâmetros.
                                    As rotas são definidas na pasta <code>Routes/</code> de cada módulo:
                                </p>
                                <ul class="list-disc list-inside text-zinc-600 dark:text-zinc-400 text-sm marker:text-primary">
                                    <li><strong>Web.php</strong>: Interfaces browser (Sessão, CSRF, Cookies).</li>
                                    <li><strong>Api.php</strong>: Endpoints stateless (JWT/Sanctum, Rate Limiting).</li>
                                </ul>
                            </div>

                            {{-- Architecture Diagram --}}
                            <div class="rounded-xl border border-zinc-200 bg-zinc-50 p-4 dark:border-zinc-800 dark:bg-zinc-900/30">
                                <h3 class="font-semibold text-zinc-900 dark:text-white mb-3 text-sm flex items-center gap-2">
                                    <flux:icon.arrows-right-left class="size-4 text-primary" />
                                    Fluxo do Request
                                </h3>
                                <div class="bg-white dark:bg-zinc-950 rounded-lg p-4 overflow-x-auto">
                                    <svg viewBox="0 0 800 180" class="w-full max-w-3xl mx-auto" xmlns="http://www.w3.org/2000/svg">
                                        <defs>
                                            <marker id="arrowhead" markerWidth="10" markerHeight="7" refX="9" refY="3.5" orient="auto">
                                                <polygon points="0 0, 10 3.5, 0 7" class="fill-primary" />
                                            </marker>
                                            <linearGradient id="boxGradient" x1="0%" y1="0%" x2="0%" y2="100%">
                                                <stop offset="0%" style="stop-color:#3b82f6;stop-opacity:0.1" />
                                                <stop offset="100%" style="stop-color:#3b82f6;stop-opacity:0.05" />
                                            </linearGradient>
                                        </defs>
                                        
                                        <!-- Request Box -->
                                        <rect x="10" y="40" width="100" height="50" rx="8" class="fill-emerald-100 dark:fill-emerald-900/30 stroke-emerald-500" stroke-width="2"/>
                                        <text x="60" y="70" text-anchor="middle" class="fill-emerald-700 dark:fill-emerald-300 text-xs font-semibold">Request</text>
                                        
                                        <!-- Arrow 1 -->
                                        <line x1="115" y1="65" x2="150" y2="65" class="stroke-primary" stroke-width="2" marker-end="url(#arrowhead)"/>
                                        
                                        <!-- RouterCollector Box -->
                                        <rect x="155" y="30" width="130" height="70" rx="8" class="fill-blue-100 dark:fill-blue-900/30 stroke-blue-500" stroke-width="2"/>
                                        <text x="220" y="55" text-anchor="middle" class="fill-blue-700 dark:fill-blue-300 text-xs font-semibold">RouterCollector</text>
                                        <text x="220" y="75" text-anchor="middle" class="fill-blue-600 dark:fill-blue-400 text-[10px]">(Route Match)</text>
                                        
                                        <!-- Arrow 2 -->
                                        <line x1="290" y1="65" x2="325" y2="65" class="stroke-primary" stroke-width="2" marker-end="url(#arrowhead)"/>
                                        
                                        <!-- Middleware Box -->
                                        <rect x="330" y="30" width="110" height="70" rx="8" class="fill-amber-100 dark:fill-amber-900/30 stroke-amber-500" stroke-width="2"/>
                                        <text x="385" y="55" text-anchor="middle" class="fill-amber-700 dark:fill-amber-300 text-xs font-semibold">Middleware</text>
                                        <text x="385" y="75" text-anchor="middle" class="fill-amber-600 dark:fill-amber-400 text-[10px]">Stack</text>
                                        
                                        <!-- Arrow 3 -->
                                        <line x1="445" y1="65" x2="480" y2="65" class="stroke-primary" stroke-width="2" marker-end="url(#arrowhead)"/>
                                        
                                        <!-- Controller Box -->
                                        <rect x="485" y="30" width="110" height="70" rx="8" class="fill-purple-100 dark:fill-purple-900/30 stroke-purple-500" stroke-width="2"/>
                                        <text x="540" y="55" text-anchor="middle" class="fill-purple-700 dark:fill-purple-300 text-xs font-semibold">Controller</text>
                                        <text x="540" y="75" text-anchor="middle" class="fill-purple-600 dark:fill-purple-400 text-[10px]">Action</text>
                                        
                                        <!-- Arrow 4 -->
                                        <line x1="600" y1="65" x2="635" y2="65" class="stroke-primary" stroke-width="2" marker-end="url(#arrowhead)"/>
                                        
                                        <!-- Response Box -->
                                        <rect x="640" y="40" width="100" height="50" rx="8" class="fill-emerald-100 dark:fill-emerald-900/30 stroke-emerald-500" stroke-width="2"/>
                                        <text x="690" y="70" text-anchor="middle" class="fill-emerald-700 dark:fill-emerald-300 text-xs font-semibold">Response</text>
                                        
                                        <!-- Arrow down to Dispatcher -->
                                        <line x1="220" y1="105" x2="220" y2="125" class="stroke-primary" stroke-width="2" marker-end="url(#arrowhead)"/>
                                        
                                        <!-- RouterDispatcher Box -->
                                        <rect x="155" y="130" width="130" height="45" rx="8" class="fill-indigo-100 dark:fill-indigo-900/30 stroke-indigo-500" stroke-width="2"/>
                                        <text x="220" y="150" text-anchor="middle" class="fill-indigo-700 dark:fill-indigo-300 text-xs font-semibold">RouterDispatcher</text>
                                        <text x="220" y="165" text-anchor="middle" class="fill-indigo-600 dark:fill-indigo-400 text-[10px]">(DI Resolution)</text>
                                    </svg>
                                </div>
                            </div>
                        </section>

                        {{-- CLI Commands --}}
                        <section id="cli" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <div class="prose prose-zinc dark:prose-invert max-w-none">
                                <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                    <flux:icon.command-line class="size-6 text-zinc-500" />
                                    CLI Commands
                                </h2>
                                <p class="text-zinc-700 dark:text-zinc-300">
                                    Ferramentas para acelerar o desenvolvimento e otimizar produção.
                                </p>
                            </div>

                            <div class="space-y-4">
                                <div class="space-y-2">
                                    <h3 class="font-bold text-zinc-900 dark:text-white text-sm">Criar Ficheiros de Rotas</h3>
                                    <div class="rounded-xl bg-zinc-950 p-4 text-zinc-300 text-xs overflow-x-auto">
                                        <pre class="language-bash"># Criar rotas API (Modules/Wmso/Routes/Api.php)
./og make:route Wmso --api

# Criar rotas Web
./og make:route Wmso --web

# Criar para sub-módulo
./og make:route Workshop/Os --api</pre>
                                    </div>
                                    <p class="text-xs text-zinc-500">Gera a classe com namespace correto e implementa <code>RouterInterface</code>.</p>
                                </div>

                                <div class="space-y-2">
                                    <h3 class="font-bold text-zinc-900 dark:text-white text-sm">Cache (Produção)</h3>
                                    <div class="rounded-xl bg-zinc-950 p-4 text-zinc-300 text-xs overflow-x-auto">
                                        <pre class="language-bash"># Compilar rotas num único JSON
./og route:cache

# Limpar cache (obrigatório após alterações)
./og route:clear

# Listar todas as rotas registadas
./og route:list</pre>
                                    </div>
                                    <p class="text-xs text-amber-600 dark:text-amber-400">
                                        <strong>Crítico:</strong> Em produção, o uso do cache evita o processamento pesado de dezenas de ficheiros a cada request.
                                    </p>
                                </div>
                            </div>

                            <div class="rounded-lg border border-blue-200 bg-blue-50 p-4 dark:border-blue-900/30 dark:bg-blue-900/10">
                                <p class="font-semibold text-blue-800 dark:text-blue-300">💡 route:list</p>
                                <p class="text-sm text-blue-700 dark:text-blue-400 mt-1">
                                    O comando <code>./og route:list</code> mostra todas as rotas com nome, método HTTP, URI, controller e middlewares aplicados. Útil para debugging.
                                </p>
                            </div>
                        </section>

                        {{-- Basic Routing --}}
                        <section id="basico" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.rocket-launch class="size-6 text-green-500" />
                                Roteamento Básico
                            </h2>
                            
                            <div class="rounded-xl bg-zinc-950 p-4 text-sm overflow-x-auto">
                                <pre class="language-php"><code>// Definição simples com Closure
Route::get('/status', function () {
    return 'Sistema Operacional';
});

// Suporte para todos os verbos HTTP
Route::get($uri, $callback);
Route::post($uri, $callback);
Route::put($uri, $callback);
Route::patch($uri, $callback);
Route::delete($uri, $callback);

// Múltiplos verbos numa única rota
Route::match(['get', 'post'], '/analytics', [AnalyticsController::class, 'track']);

// Aceitar qualquer verbo HTTP
Route::any('/webhook', [WebhookController::class, 'handle']);</code></pre>
                            </div>

                            <div class="rounded-xl border border-blue-200 bg-blue-50 p-4 dark:border-blue-900/50 dark:bg-blue-900/20">
                                <h3 class="font-semibold text-blue-900 dark:text-blue-200 mb-2 flex items-center gap-2">
                                    <flux:icon.cube class="size-4" />
                                    Injeção de Dependências
                                </h3>
                                <p class="text-sm text-blue-800 dark:text-blue-300 mb-2">
                                    O <code>RouterDispatcher</code> resolve automaticamente qualquer classe tipada na assinatura da função ou método do controller.
                                </p>
                                <pre class="text-xs bg-white dark:bg-black/20 p-2 rounded"><code>use Og\Modules\Sales\Services\InvoiceService;

Route::get('/invoice/{id}', function (InvoiceService $service, $id) {
    return $service->download($id);
});</code></pre>
                            </div>
                        </section>

                        {{-- Parameters --}}
                        <section id="parametros" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.variable class="size-6 text-cyan-500" />
                                Parâmetros de Rota
                            </h2>
                            <p class="text-zinc-700 dark:text-zinc-300">
                                Capture segmentos da URI definindo parâmetros entre chavetas <code>{}</code>.
                            </p>
                            
                            <div class="space-y-4">
                                {{-- Required --}}
                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <div class="flex items-center justify-between mb-2">
                                        <h3 class="font-bold text-zinc-900 dark:text-white">Obrigatórios</h3>
                                    </div>
                                    <pre class="text-xs bg-zinc-950 p-3 rounded-lg text-zinc-300"><code>Route::get('/user/{id}', function ($id) {
    return 'User '.$id;
});

// Múltiplos parâmetros
Route::get('/posts/{post}/comments/{comment}', 
    function ($postId, $commentId) {
        // $postId e $commentId injetados por ordem
    }
);</code></pre>
                                </div>

                                {{-- Optional --}}
                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <div class="flex items-center justify-between mb-2">
                                        <h3 class="font-bold text-zinc-900 dark:text-white">Opcionais (?)</h3>
                                    </div>
                                    <pre class="text-xs bg-zinc-950 p-3 rounded-lg text-zinc-300"><code>// Obrigatório definir valor default
Route::get('/posts/{page?}', function ($page = 1) {
    return "Página: $page";
});

// Múltiplos opcionais
Route::get('/search/{query?}/{limit?}', 
    function ($query = '', $limit = 10) {
        // ...
    }
);</code></pre>
                                </div>
                            </div>
                            
                            <div class="rounded-xl border border-purple-200 bg-purple-50 p-4 dark:border-purple-900/50 dark:bg-purple-900/20">
                                <h3 class="font-semibold text-purple-900 dark:text-purple-200 text-sm mb-1">⚠️ Ordem dos Parâmetros</h3>
                                <p class="text-xs text-purple-800 dark:text-purple-300">
                                    Em <strong>Closures</strong>, a ordem dos argumentos importa (são mapeados pela posição). 
                                    Em <strong>Controllers</strong>, o <code>RouterDispatcher</code> pode mapear por nome ou tipo-hint automaticamente.
                                </p>
                            </div>
                        </section>

                        {{-- Constraints --}}
                        <section id="constraints" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.shield-check class="size-6 text-red-500" />
                                Validação de Parâmetros (Constraints)
                            </h2>
                            <p class="text-zinc-700 dark:text-zinc-300">
                                Restrinja os valores dos parâmetros usando expressões regulares com <code>->where()</code>.
                            </p>

                            <div class="rounded-xl bg-zinc-950 p-4 text-sm overflow-x-auto">
                                <pre class="language-php"><code>// Constraint simples - apenas números
Route::get('/user/{id}', [UserController::class, 'show'])
    ->where('id', '[0-9]+');

// Múltiplos constraints
Route::get('/user/{id}/{name}', [UserController::class, 'profile'])
    ->where(['id' => '[0-9]+', 'name' => '[a-zA-Z]+']);

// Shortcuts comuns
Route::get('/user/{id}', ...)->whereNumber('id');
Route::get('/user/{name}', ...)->whereAlpha('name');
Route::get('/user/{slug}', ...)->whereAlphaNumeric('slug');
Route::get('/category/{uuid}', ...)->whereUuid('uuid');</code></pre>
                            </div>

                            <div class="space-y-4">
                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-bold text-zinc-900 dark:text-white mb-2 text-sm">Global Constraints</h3>
                                    <p class="text-xs text-zinc-500 mb-2">Definir padrões globais no <code>RouteServiceProvider</code>:</p>
                                    <pre class="text-xs bg-zinc-950 p-3 rounded-lg text-zinc-300"><code>// Em boot()
Route::pattern('id', '[0-9]+');
Route::pattern('uuid', '[a-f0-9-]{36}');</code></pre>
                                </div>
                                
                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-bold text-zinc-900 dark:text-white mb-2 text-sm">Tipagem Forte</h3>
                                    <p class="text-xs text-zinc-500 mb-2">PHP valida o tipo automaticamente:</p>
                                    <pre class="text-xs bg-zinc-950 p-3 rounded-lg text-zinc-300"><code>Route::get('/user/{id}', function (int $id) {
    // Se {id} não for int, erro lançado
    return User::find($id);
});</code></pre>
                                </div>
                            </div>
                        </section>

                        {{-- Named Routes --}}
                        <section id="nomes" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.tag class="size-6 text-purple-500" />
                                Rotas Nomeadas
                            </h2>
                            
                            <div class="rounded-xl bg-zinc-950 p-4 text-sm overflow-x-auto">
                                <pre class="language-php"><code>Route::get('/user/profile', [ProfileController::class, 'show'])
    ->name('profile.show');

// Gerar URL pelo nome
$url = route('profile.show');

// Redirecionar para rota nomeada
return redirect()->route('profile.show');</code></pre>
                            </div>

                            <div class="rounded-xl border-l-4 border-primary bg-zinc-50 p-4 dark:bg-zinc-900/50">
                                <h4 class="font-bold text-zinc-900 dark:text-white text-sm mb-1">📛 Dot Notation (.)</h4>
                                <p class="text-sm text-zinc-600 dark:text-zinc-400">
                                    Use sempre pontos para separar contexto, módulo e ação. Exemplo: <code>api.wmso.articles.show</code>.
                                </p>
                                <div class="mt-3 space-y-2 text-xs">
                                    <div class="bg-white dark:bg-zinc-900 rounded p-2">
                                        <span class="text-purple-600 font-semibold">api.</span><span class="text-zinc-400">wmso.articles.index</span>
                                        <p class="text-zinc-500 mt-1">Contexto</p>
                                    </div>
                                    <div class="bg-white dark:bg-zinc-900 rounded p-2">
                                        <span class="text-zinc-400">api.</span><span class="text-purple-600 font-semibold">wmso.</span><span class="text-zinc-400">articles.index</span>
                                        <p class="text-zinc-500 mt-1">Módulo</p>
                                    </div>
                                    <div class="bg-white dark:bg-zinc-900 rounded p-2">
                                        <span class="text-zinc-400">api.wmso.</span><span class="text-purple-600 font-semibold">articles.index</span>
                                        <p class="text-zinc-500 mt-1">Recurso.Ação</p>
                                    </div>
                                </div>
                            </div>
                        </section>

                        {{-- URL Generation --}}
                        <section id="urls" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.link class="size-6 text-emerald-500" />
                                Geração de URLs
                            </h2>
                            <p class="text-zinc-700 dark:text-zinc-300">
                                O helper <code>route()</code> gera URLs baseadas nos nomes das rotas, garantindo flexibilidade e evitando hardcoding.
                            </p>

                            <div class="rounded-xl bg-zinc-950 p-4 text-sm overflow-x-auto">
                                <pre class="language-php"><code>// URL simples
$url = route('profile.show'); 
// → http://app.com/user/profile

// Com parâmetros
$url = route('user.show', ['id' => 123]); 
// → http://app.com/user/123

// Múltiplos parâmetros
$url = route('post.comment', ['post' => 1, 'comment' => 5]); 
// → http://app.com/posts/1/comments/5

// Parâmetros extras → query string
$url = route('user.index', ['page' => 2, 'sort' => 'name']); 
// → http://app.com/users?page=2&sort=name</code></pre>
                            </div>

                            <div class="space-y-4">
                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-bold text-zinc-900 dark:text-white mb-2 text-sm">Em Views Blade</h3>
                                    <pre class="text-xs bg-zinc-950 p-3 rounded-lg text-zinc-300"><code>&lt;a href="@{{ route('profile.show') }}"&gt;
    Ver Perfil
&lt;/a&gt;

&lt;form action="@{{ route('user.update', $user) }}"&gt;
    ...
&lt;/form&gt;</code></pre>
                                </div>
                                
                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-bold text-zinc-900 dark:text-white mb-2 text-sm">Absolute vs Relative</h3>
                                    <pre class="text-xs bg-zinc-950 p-3 rounded-lg text-zinc-300"><code>// URL absoluta (padrão)
route('home'); 
// → http://app.com/

// Path relativo
route('home', [], false); 
// → /</code></pre>
                                </div>
                            </div>
                        </section>

                        {{-- Groups --}}
                        <section id="grupos" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.squares-plus class="size-6 text-amber-500" />
                                Grupos e Organização
                            </h2>
                            <p class="text-zinc-700 dark:text-zinc-300">
                                Evite repetição agrupando rotas com prefixos, middlewares ou controllers comuns.
                            </p>

                            <div class="space-y-4">
                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-bold text-zinc-900 dark:text-white mb-2 text-sm">Estrutura Hierárquica</h3>
                                    <pre class="text-xs bg-zinc-950 p-3 rounded-lg text-zinc-300"><code>Route::name('admin.')
    ->prefix('admin')
    ->middleware('auth')
    ->group(function () {
        
        Route::get('/users', 'Users@index')
            ->name('users'); 
            // URL: /admin/users
            // Nome: admin.users
            
    });</code></pre>
                                </div>

                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-bold text-zinc-900 dark:text-white mb-2 text-sm">Controller Grouping</h3>
                                    <pre class="text-xs bg-zinc-950 p-3 rounded-lg text-zinc-300"><code>Route::controller(OrderController::class)
    ->group(function () {
        Route::get('/orders', 'index');
        Route::post('/orders', 'store');
        Route::get('/orders/{id}', 'show');
        Route::put('/orders/{id}', 'update');
        Route::delete('/orders/{id}', 'destroy');
    });</code></pre>
                                </div>
                            </div>

                            {{-- Nested Groups Example --}}
                            <div class="rounded-xl bg-zinc-950 p-4 text-sm overflow-x-auto">
                                <p class="text-zinc-400 text-xs mb-2">// Exemplo real: API do módulo WMSO</p>
                                <pre class="language-php"><code>Route::name('api.')
    ->prefix('api')
    ->middleware(['api', 'throttle:60,1'])
    ->group(function () {
        
        Route::name('wmso.')
            ->prefix('wmso')
            ->group(function () {
                
                Route::name('articles.')
                    ->controller(ArticleController::class)
                    ->group(function () {
                        Route::get('/', 'index')->name('index');       // api.wmso.articles.index
                        Route::get('/{id}', 'show')->name('show');     // api.wmso.articles.show
                        Route::post('/', 'store')->name('store');      // api.wmso.articles.store
                    });
                    
            });
    });</code></pre>
                            </div>
                        </section>

                        {{-- Middleware --}}
                        <section id="middleware" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.funnel class="size-6 text-indigo-500" />
                                Middleware
                            </h2>
                            <p class="text-zinc-700 dark:text-zinc-300">
                                Middlewares filtram e processam requests antes de chegarem ao controller. Podem ser aplicados a rotas individuais ou grupos.
                            </p>

                            <div class="space-y-4">
                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-bold text-zinc-900 dark:text-white mb-2 text-sm">Rota Individual</h3>
                                    <pre class="text-xs bg-zinc-950 p-3 rounded-lg text-zinc-300"><code>Route::get('/admin', [AdminController::class, 'index'])
    ->middleware('auth');

// Múltiplos middlewares
Route::get('/dashboard', ...)
    ->middleware(['auth', 'verified']);</code></pre>
                                </div>
                                
                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-bold text-zinc-900 dark:text-white mb-2 text-sm">Em Grupos</h3>
                                    <pre class="text-xs bg-zinc-950 p-3 rounded-lg text-zinc-300"><code>Route::middleware(['auth', 'admin'])
    ->group(function () {
        Route::get('/users', ...);
        Route::get('/settings', ...);
    });</code></pre>
                                </div>
                            </div>

                            <div class="rounded-xl bg-zinc-950 p-4 text-sm overflow-x-auto">
                                <p class="text-zinc-400 text-xs mb-2">// Middlewares com parâmetros</p>
                                <pre class="language-php"><code>// Rate limiting: 60 requests por minuto
Route::middleware('throttle:60,1')->group(...);

// Verificar role específica
Route::middleware('role:admin')->group(...);

// Excluir middleware específico
Route::withoutMiddleware('csrf')->post('/webhook', ...);</code></pre>
                            </div>

                            {{-- Common Middlewares Table --}}
                            <div class="overflow-x-auto">
                                <table class="min-w-full text-sm">
                                    <thead class="bg-zinc-100 dark:bg-zinc-800">
                                        <tr>
                                            <th class="px-4 py-2 text-left font-semibold text-zinc-900 dark:text-white">Middleware</th>
                                            <th class="px-4 py-2 text-left font-semibold text-zinc-900 dark:text-white">Descrição</th>
                                        </tr>
                                    </thead>
                                    <tbody class="divide-y divide-zinc-200 dark:divide-zinc-700">
                                        <tr><td class="px-4 py-2 font-mono text-xs">auth</td><td class="px-4 py-2">Requer autenticação</td></tr>
                                        <tr><td class="px-4 py-2 font-mono text-xs">guest</td><td class="px-4 py-2">Apenas visitantes (não autenticados)</td></tr>
                                        <tr><td class="px-4 py-2 font-mono text-xs">throttle:X,Y</td><td class="px-4 py-2">Rate limit (X requests por Y minutos)</td></tr>
                                        <tr><td class="px-4 py-2 font-mono text-xs">verified</td><td class="px-4 py-2">Email verificado</td></tr>
                                        <tr><td class="px-4 py-2 font-mono text-xs">signed</td><td class="px-4 py-2">URL com assinatura válida</td></tr>
                                    </tbody>
                                </table>
                            </div>
                        </section>

                        {{-- Convenience Routes --}}
                        <section id="conveniencia" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.bolt class="size-6 text-yellow-500" />
                                Rotas de Conveniência
                            </h2>
                            <p class="text-zinc-700 dark:text-zinc-300">
                                Shortcuts para casos comuns que não precisam de controller.
                            </p>

                            <div class="space-y-4">
                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-mono text-xs font-bold text-primary mb-2">Route::redirect()</h3>
                                    <pre class="text-xs bg-zinc-950 p-3 rounded-lg text-zinc-300"><code>// Redirect 302 (temporário)
Route::redirect('/here', '/there');

// Redirect 301 (permanente)
Route::permanentRedirect(
    '/old', 
    '/new'
);</code></pre>
                                </div>
                                
                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-mono text-xs font-bold text-primary mb-2">Route::view()</h3>
                                    <pre class="text-xs bg-zinc-950 p-3 rounded-lg text-zinc-300"><code>// Renderizar view diretamente
Route::view('/about', 'pages.about');

// Com dados
Route::view('/terms', 'legal.terms', [
    'version' => '2.1'
]);</code></pre>
                                </div>
                                
                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-mono text-xs font-bold text-primary mb-2">Route::fallback()</h3>
                                    <pre class="text-xs bg-zinc-950 p-3 rounded-lg text-zinc-300"><code>// 404 customizado
Route::fallback(function () {
    return response()
        ->view('errors.404')
        ->setStatusCode(404);
});</code></pre>
                                </div>
                            </div>
                        </section>

                        {{-- Route Helpers --}}
                        <section id="helpers" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <div class="prose prose-zinc dark:prose-invert max-w-none">
                                <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                    <flux:icon.magnifying-glass class="size-6 text-blue-500" />
                                    Helpers e Introspecção
                                </h2>
                                <p class="text-zinc-700 dark:text-zinc-300">
                                    O trait <code>RouteHelper</code> no Request permite verificar onde estamos e aceder a parâmetros da rota.
                                </p>
                            </div>

                            <div class="space-y-4">
                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-mono text-xs font-bold text-primary mb-1">routeIs($pattern)</h3>
                                    <p class="text-xs text-zinc-500 mb-2">Verifica nome da rota (suporta wildcards).</p>
                                    <pre class="text-[10px] bg-zinc-100 dark:bg-zinc-900 p-2 rounded"><code>if ($req->routeIs('admin.*')) { }
if ($req->routeIs('*.store')) { }</code></pre>
                                </div>
                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-mono text-xs font-bold text-primary mb-1">is($pattern)</h3>
                                    <p class="text-xs text-zinc-500 mb-2">Verifica URL (path).</p>
                                    <pre class="text-[10px] bg-zinc-100 dark:bg-zinc-900 p-2 rounded"><code>if ($req->is('api/v1/*')) { }</code></pre>
                                </div>
                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-mono text-xs font-bold text-primary mb-1">route($param)</h3>
                                    <p class="text-xs text-zinc-500 mb-2">Obtém valor de parâmetro.</p>
                                    <pre class="text-[10px] bg-zinc-100 dark:bg-zinc-900 p-2 rounded"><code>$id = $req->route('id');</code></pre>
                                </div>
                            </div>
                            
                            <div class="rounded-xl bg-zinc-950 p-4 text-sm overflow-x-auto">
                                <pre class="language-php"><code>// URL: /api/wmso/articles/10?page=2

// Path sem query string
$request->path();          // "api/wmso/articles/10"

// URL completa
$request->url();           // "http://domain.com/api/wmso/articles/10"

// URL com query string
$request->fullUrl();       // "http://domain.com/api/wmso/articles/10?page=2"

// Segmentos da URL (array)
$request->segments();      // ['api', 'wmso', 'articles', '10']

// Segmento específico (1-based index)
$request->segment(2);      // 'wmso'
$request->segment(4);      // '10'

// Todos os parâmetros da rota
$request->routeParameters(); // ['id' => '10']</code></pre>
                            </div>
                        </section>

                        {{-- Method Spoofing --}}
                        <section id="spoofing" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.code-bracket-square class="size-6 text-gray-500" />
                                Form Method Spoofing
                            </h2>
                            <div class="rounded-xl border border-amber-200 bg-amber-50 p-4 dark:border-amber-900/50 dark:bg-amber-900/20">
                                <p class="text-sm text-amber-800 dark:text-amber-300 mb-2">
                                    Formulários HTML não suportam <code>PUT</code>, <code>PATCH</code> ou <code>DELETE</code> nativamente. Use a diretiva <code>@@method()</code>.
                                </p>
                                <pre class="text-xs bg-white dark:bg-black/20 p-3 rounded"><code>&lt;form action="/users/1" method="POST"&gt;
    @@method('PUT')
    @@csrf
    
    &lt;!-- campos do formulário --&gt;
&lt;/form&gt;

&lt;!-- Para DELETE --&gt;
&lt;form action="/users/1" method="POST"&gt;
    @@method('DELETE')
    @@csrf
    &lt;button type="submit"&gt;Eliminar&lt;/button&gt;
&lt;/form&gt;</code></pre>
                            </div>
                        </section>

                        {{-- Troubleshooting --}}
                        <section id="troubleshooting" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.wrench-screwdriver class="size-6 text-red-500" />
                                Troubleshooting
                            </h2>
                            <div class="space-y-4">
                                <div class="rounded-xl border border-red-200 bg-red-50 p-4 dark:border-red-900/50 dark:bg-red-900/20">
                                    <h3 class="font-semibold text-red-900 dark:text-red-200 mb-2 text-sm">❌ Route [name] not defined</h3>
                                    <p class="text-xs text-red-800 dark:text-red-300">
                                        Verifique se definiu o <code>->name()</code>. Lembre-se que em grupos, os prefixos concatenam-se. Use <code>./og route:list</code> para confirmar os nomes finais.
                                    </p>
                                </div>
                                <div class="rounded-xl border border-amber-200 bg-amber-50 p-4 dark:border-amber-900/50 dark:bg-amber-900/20">
                                    <h3 class="font-semibold text-amber-900 dark:text-amber-200 mb-2 text-sm">⚠️ Parâmetro Nulo</h3>
                                    <p class="text-xs text-amber-800 dark:text-amber-300">
                                        Se <code>$request->route('id')</code> for null, certifique-se que a rota tem <code>{id}</code>. Query strings (<code>?id=1</code>) usam <code>$request->input('id')</code>.
                                    </p>
                                </div>
                                <div class="rounded-xl border border-blue-200 bg-blue-50 p-4 dark:border-blue-900/50 dark:bg-blue-900/20">
                                    <h3 class="font-semibold text-blue-900 dark:text-blue-200 mb-2 text-sm">🔄 Cache Desatualizado</h3>
                                    <p class="text-xs text-blue-800 dark:text-blue-300">
                                        Após alterar rotas em produção, execute <code>./og route:clear && ./og route:cache</code>. O cache antigo não reflete as novas alterações.
                                    </p>
                                </div>
                                <div class="rounded-xl border border-purple-200 bg-purple-50 p-4 dark:border-purple-900/50 dark:bg-purple-900/20">
                                    <h3 class="font-semibold text-purple-900 dark:text-purple-200 mb-2 text-sm">🔍 Debug de Rotas</h3>
                                    <p class="text-xs text-purple-800 dark:text-purple-300">
                                        Para ver todas as rotas registadas: <code>dd(RouterCollector::getInstance()->getRoutes())</code> ou use <code>./og route:list</code>.
                                    </p>
                                </div>
                            </div>
                        </section>

                        {{-- Best Practices --}}
                        <section id="boas-praticas" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.light-bulb class="size-6 text-yellow-500" />
                                Boas Práticas
                            </h2>
                            
                            <div class="space-y-4">
                                <div class="rounded-xl border border-green-200 bg-green-50 p-4 dark:border-green-900/30 dark:bg-green-900/10">
                                    <h3 class="font-semibold text-green-800 dark:text-green-300 mb-2 text-sm">✅ Faça</h3>
                                    <ul class="text-xs text-green-700 dark:text-green-400 space-y-1 list-disc list-inside">
                                        <li>Use <strong>dot notation</strong> consistente nos nomes</li>
                                        <li>Agrupe rotas por <strong>controller</strong> sempre que possível</li>
                                        <li>Aplique <strong>constraints</strong> em IDs numéricos</li>
                                        <li>Use <code>route()</code> em vez de URLs hardcoded</li>
                                        <li>Mantenha um ficheiro de rotas por <strong>contexto</strong> (Api vs Web)</li>
                                        <li>Execute <code>route:cache</code> em produção</li>
                                    </ul>
                                </div>
                                
                                <div class="rounded-xl border border-red-200 bg-red-50 p-4 dark:border-red-900/30 dark:bg-red-900/10">
                                    <h3 class="font-semibold text-red-800 dark:text-red-300 mb-2 text-sm">❌ Evite</h3>
                                    <ul class="text-xs text-red-700 dark:text-red-400 space-y-1 list-disc list-inside">
                                        <li>Closures complexas nas rotas (dificulta cache)</li>
                                        <li>Nomes genéricos como <code>index</code> sem prefixo</li>
                                        <li>Lógica de negócio dentro de route files</li>
                                        <li>Misturar convenções de nomenclatura</li>
                                        <li>Rotas duplicadas (mesmo método + URI)</li>
                                        <li>Ignorar middlewares de segurança</li>
                                    </ul>
                                </div>
                            </div>
                        </section>

                        {{-- Reference Table --}}
                        <section id="referencia" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.table-cells class="size-6 text-zinc-500" />
                                Referência Rápida
                            </h2>
                            <div class="overflow-hidden rounded-xl border border-zinc-200 dark:border-zinc-800 bg-white dark:bg-zinc-900/50">
                                <table class="min-w-full divide-y divide-zinc-200 dark:divide-zinc-800">
                                    <thead class="bg-zinc-50 dark:bg-zinc-900/80">
                                        <tr>
                                            <th class="px-6 py-3 text-left text-xs font-medium text-zinc-500 uppercase">Método/Helper</th>
                                            <th class="px-6 py-3 text-left text-xs font-medium text-zinc-500 uppercase">Descrição</th>
                                            <th class="px-6 py-3 text-left text-xs font-medium text-zinc-500 uppercase">Exemplo</th>
                                        </tr>
                                    </thead>
                                    <tbody class="divide-y divide-zinc-200 dark:divide-zinc-800">
                                        <tr><td class="px-6 py-4 font-mono text-xs text-purple-600">Route::get/post/put/...</td><td class="px-6 py-4 text-xs">Definir rota com verbo HTTP</td><td class="px-6 py-4 font-mono text-xs">Route::get('/users', ...)</td></tr>
                                        <tr><td class="px-6 py-4 font-mono text-xs text-purple-600">->name()</td><td class="px-6 py-4 text-xs">Atribuir nome à rota</td><td class="px-6 py-4 font-mono text-xs">->name('users.index')</td></tr>
                                        <tr><td class="px-6 py-4 font-mono text-xs text-purple-600">->where()</td><td class="px-6 py-4 text-xs">Constraint regex</td><td class="px-6 py-4 font-mono text-xs">->where('id', '[0-9]+')</td></tr>
                                        <tr><td class="px-6 py-4 font-mono text-xs text-purple-600">->middleware()</td><td class="px-6 py-4 text-xs">Aplicar middleware</td><td class="px-6 py-4 font-mono text-xs">->middleware('auth')</td></tr>
                                        <tr><td class="px-6 py-4 font-mono text-xs text-purple-600">route($name, $params)</td><td class="px-6 py-4 text-xs">Gerar URL por nome</td><td class="px-6 py-4 font-mono text-xs">route('user.show', ['id' => 1])</td></tr>
                                        <tr><td class="px-6 py-4 font-mono text-xs text-purple-600">$req->routeIs()</td><td class="px-6 py-4 text-xs">Verificar nome da rota</td><td class="px-6 py-4 font-mono text-xs">$req->routeIs('admin.*')</td></tr>
                                        <tr><td class="px-6 py-4 font-mono text-xs text-purple-600">$req->route($param)</td><td class="px-6 py-4 text-xs">Obter parâmetro da rota</td><td class="px-6 py-4 font-mono text-xs">$req->route('id')</td></tr>
                                        <tr><td class="px-6 py-4 font-mono text-xs text-purple-600">$req->segment($n)</td><td class="px-6 py-4 text-xs">Segmento N da URL (1-based)</td><td class="px-6 py-4 font-mono text-xs">$req->segment(2)</td></tr>
                                    </tbody>
                                </table>
                            </div>
                        </section>

                        {{-- Navigation --}}
                        <div class="flex justify-between items-center pt-8 border-t border-zinc-200 dark:border-zinc-800">
                            <a href="{{ route('docs.show', 'filesystem') }}" class="inline-flex items-center gap-2 text-sm text-zinc-600 dark:text-zinc-400 hover:text-primary">
                                <flux:icon.arrow-left class="size-4" />
                                Filesystem
                            </a>
                            <a href="{{ route('docs.show', 'queue') }}" class="inline-flex items-center gap-2 text-sm font-medium text-primary hover:underline">
                                Queues
                                <flux:icon.arrow-right class="size-4" />
                            </a>
                        </div>
                    </div>

                </div>
            </div>

            {{-- Right Sidebar: Table of Contents --}}
            @include('docs.partials.toc', ['sections' => [
                'introducao' => 'Introdução',
                'cli' => 'CLI Commands',
                'basico' => 'Roteamento Básico',
                'parametros' => 'Parâmetros',
                'constraints' => 'Constraints',
                'nomes' => 'Rotas Nomeadas',
                'urls' => 'Geração de URLs',
                'grupos' => 'Grupos',
                'middleware' => 'Middleware',
                'conveniencia' => 'Conveniência',
                'helpers' => 'Helpers',
                'spoofing' => 'Method Spoofing',
                'troubleshooting' => 'Troubleshooting',
                'boas-praticas' => 'Boas Práticas',
                'referencia' => 'Referência',
            ]])
        </div>
    </div>
@endsection
