@extends('layouts.docs')

@section('title', 'Service Providers - OG Framework')

@section('body')
    <div class="relative min-h-screen bg-transparent px-4 pb-12 pt-6 sm:px-6 lg:px-8">
        <div class="floating-blur blue -left-10 -top-10 hidden lg:block"></div>
        <div class="floating-blur amber right-10 top-32 hidden lg:block"></div>

        <div class="mx-auto flex w-full max-w-7xl gap-8 bg-transparent">
            {{-- Left Sidebar: Navigation --}}
            @include('docs.partials.sidebar')

            {{-- Main Content --}}
            <div class="flex-1 min-w-0">
                <div class="relative overflow-hidden rounded-3xl border border-white/50 bg-white/85 p-8 shadow-xl shadow-primary/20 backdrop-blur dark:border-zinc-800 dark:bg-zinc-900/80">

                    {{-- Page Header --}}
                    <div class="mb-8">
                        <a href="{{ route('docs.index') }}" class="inline-flex items-center gap-1 text-sm text-primary hover:underline mb-4">
                            <flux:icon.arrow-left class="size-4" />
                            Voltar para Documentação
                        </a>
                        <p class="text-xs uppercase tracking-[0.3em] text-primary dark:text-blue-200">Arquitetura</p>
                        <h1 class="font-display text-4xl font-bold text-zinc-900 dark:text-white sm:text-5xl">Service Providers</h1>
                        <p class="mt-3 text-base text-zinc-700 dark:text-zinc-200 max-w-3xl">
                            Service Providers são classes que registam serviços no Container de DI. São o ponto central de configuração de toda a aplicação — centralizando bindings, eventos, middlewares e comandos CLI.
                        </p>
                    </div>

                    <div class="space-y-10">

                        {{-- Introduction --}}
                        <section id="introducao" class="scroll-mt-28 space-y-4">
                            <div class="prose prose-zinc dark:prose-invert max-w-none">
                                <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                    <flux:icon.light-bulb class="size-6 text-amber-500" />
                                    Por que usar Service Providers?
                                </h2>
                            </div>

                            <div class="overflow-x-auto">
                                <table class="w-full text-sm text-left">
                                    <thead class="text-xs uppercase bg-zinc-100 dark:bg-zinc-800">
                                        <tr>
                                            <th class="px-4 py-3 text-red-600 dark:text-red-400">❌ Sem Providers</th>
                                            <th class="px-4 py-3 text-green-600 dark:text-green-400">✅ Com Providers</th>
                                        </tr>
                                    </thead>
                                    <tbody class="divide-y divide-zinc-200 dark:divide-zinc-700">
                                        <tr class="bg-white dark:bg-zinc-900">
                                            <td class="px-4 py-3">Configuração espalhada</td>
                                            <td class="px-4 py-3">Configuração centralizada</td>
                                        </tr>
                                        <tr class="bg-zinc-50 dark:bg-zinc-800/50">
                                            <td class="px-4 py-3">Difícil de testar</td>
                                            <td class="px-4 py-3">Fácil mock/swap</td>
                                        </tr>
                                        <tr class="bg-white dark:bg-zinc-900">
                                            <td class="px-4 py-3">Acoplamento forte</td>
                                            <td class="px-4 py-3">Desacoplado via interfaces</td>
                                        </tr>
                                        <tr class="bg-zinc-50 dark:bg-zinc-800/50">
                                            <td class="px-4 py-3">Ordem manual</td>
                                            <td class="px-4 py-3">Ordem controlada</td>
                                        </tr>
                                    </tbody>
                                </table>
                            </div>

                            <div class="rounded-xl bg-zinc-950 p-4 text-sm overflow-x-auto">
                                <pre class="language-php"><code>// ❌ Sem Providers — configuração espalhada
$database = new Database($config);
$cache = new Cache($redis);
$logger = new Logger($path);

// ✅ Com Providers — tudo centralizado
class MyServiceProvider extends ServiceProvider
{
    public function register(): void
    {
        $this->app->singleton('database', fn() => new Database($config));
        $this->app->singleton('cache', fn() => new Cache($redis));
        $this->app->singleton('logger', fn() => new Logger($path));
    }
}</code></pre>
                            </div>
                        </section>

                        {{-- Lifecycle --}}
                        <section id="ciclo-de-vida" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <div class="prose prose-zinc dark:prose-invert max-w-none">
                                <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                    <flux:icon.arrow-path class="size-6 text-primary" />
                                    Ciclo de Vida: Register & Boot
                                </h2>
                                <p class="text-zinc-700 dark:text-zinc-300">
                                    Todo Provider tem duas fases distintas que são executadas em momentos diferentes:
                                </p>
                            </div>

                            <div class="rounded-xl bg-gradient-to-br from-zinc-900 to-zinc-950 p-6 overflow-x-auto">
                                <svg viewBox="0 0 700 350" class="w-full max-w-3xl mx-auto" xmlns="http://www.w3.org/2000/svg">
                                    <defs>
                                        <linearGradient id="registerGradient" x1="0%" y1="0%" x2="0%" y2="100%">
                                            <stop offset="0%" style="stop-color:#f59e0b;stop-opacity:1" />
                                            <stop offset="100%" style="stop-color:#b45309;stop-opacity:1" />
                                        </linearGradient>
                                        <linearGradient id="bootGradient" x1="0%" y1="0%" x2="0%" y2="100%">
                                            <stop offset="0%" style="stop-color:#22c55e;stop-opacity:1" />
                                            <stop offset="100%" style="stop-color:#15803d;stop-opacity:1" />
                                        </linearGradient>
                                    </defs>

                                    <!-- Phase 1: Register -->
                                    <rect x="150" y="10" width="400" height="140" rx="6" fill="#1e293b" stroke="#f59e0b" stroke-width="2" stroke-dasharray="4 4"/>
                                    
                                    <rect x="250" y="0" width="200" height="30" rx="6" fill="#f59e0b" stroke="#b45309" stroke-width="2"/>
                                    <text x="350" y="20" fill="#78350f" font-family="ui-monospace, monospace" font-size="12" text-anchor="middle" font-weight="800">FASE 1: REGISTER</text>
                                    
                                    <!-- Register Items -->
                                    <text x="200" y="55" fill="#fcd34d" font-family="ui-monospace, monospace" font-size="11">ProviderManager.register()</text>
                                    
                                    <line x1="220" y1="65" x2="220" y2="135" stroke="#475569" stroke-width="2"/>
                                    
                                    <circle cx="220" cy="80" r="4" fill="#f59e0b"/>
                                    <text x="240" y="85" fill="#cbd5e1" font-family="ui-monospace, monospace" font-size="11">Provider1->register()</text>
                                    <text x="450" y="85" fill="#94a3b8" font-family="ui-monospace, monospace" font-size="10">→ bindings ✓</text>

                                    <circle cx="220" cy="105" r="4" fill="#f59e0b"/>
                                    <text x="240" y="110" fill="#cbd5e1" font-family="ui-monospace, monospace" font-size="11">Provider2->register()</text>
                                    <text x="450" y="110" fill="#94a3b8" font-family="ui-monospace, monospace" font-size="10">→ bindings ✓</text>

                                    <circle cx="220" cy="130" r="4" fill="#f59e0b"/>
                                    <text x="240" y="135" fill="#cbd5e1" font-family="ui-monospace, monospace" font-size="11">Provider3->register()</text>
                                    <text x="450" y="135" fill="#94a3b8" font-family="ui-monospace, monospace" font-size="10">→ bindings ✓</text>

                                    <!-- Transition Arrow -->
                                    <line x1="350" y1="150" x2="350" y2="180" stroke="#cbd5e1" stroke-width="2"/>
                                    <polygon points="350,190 345,180 355,180" fill="#cbd5e1"/>

                                    <!-- Phase 2: Boot -->
                                    <rect x="150" y="200" width="400" height="140" rx="6" fill="#1e293b" stroke="#22c55e" stroke-width="2"/>
                                    
                                    <rect x="250" y="190" width="200" height="30" rx="6" fill="#22c55e" stroke="#14532d" stroke-width="2"/>
                                    <text x="350" y="210" fill="#064e3b" font-family="ui-monospace, monospace" font-size="12" text-anchor="middle" font-weight="800">FASE 2: BOOT</text>

                                    <!-- Boot Items -->
                                    <text x="200" y="245" fill="#86efac" font-family="ui-monospace, monospace" font-size="11">ProviderManager.boot()</text>
                                    
                                    <line x1="220" y1="255" x2="220" y2="325" stroke="#475569" stroke-width="2"/>
                                    
                                    <circle cx="220" cy="270" r="4" fill="#22c55e"/>
                                    <text x="240" y="275" fill="#cbd5e1" font-family="ui-monospace, monospace" font-size="11">Provider1->boot()</text>
                                    <text x="450" y="275" fill="#94a3b8" font-family="ui-monospace, monospace" font-size="10">→ events ✓</text>

                                    <circle cx="220" cy="295" r="4" fill="#22c55e"/>
                                    <text x="240" y="300" fill="#cbd5e1" font-family="ui-monospace, monospace" font-size="11">Provider2->boot()</text>
                                    <text x="450" y="300" fill="#94a3b8" font-family="ui-monospace, monospace" font-size="10">→ middleware ✓</text>

                                    <circle cx="220" cy="320" r="4" fill="#22c55e"/>
                                    <text x="240" y="325" fill="#cbd5e1" font-family="ui-monospace, monospace" font-size="11">Provider3->boot()</text>
                                    <text x="450" y="325" fill="#94a3b8" font-family="ui-monospace, monospace" font-size="10">→ routes ✓</text>

                                </svg>
                            </div>

                            <div class="grid gap-4 md:grid-cols-2">
                                <div class="feature-card border-l-4 border-amber-500">
                                    <p class="font-mono text-sm font-semibold text-amber-600 dark:text-amber-400">register()</p>
                                    <div class="text-sm text-zinc-700 dark:text-zinc-300 mt-2 space-y-1">
                                        <p class="text-green-600 dark:text-green-400">✅ Fazer:</p>
                                        <ul class="list-disc list-inside text-xs">
                                            <li>Registar singletons e bindings</li>
                                            <li>Criar aliases</li>
                                            <li>Registar comandos CLI</li>
                                        </ul>
                                        <p class="text-red-600 dark:text-red-400 mt-2">❌ Não fazer:</p>
                                        <ul class="list-disc list-inside text-xs">
                                            <li>Resolver dependências</li>
                                            <li>Usar outros serviços</li>
                                        </ul>
                                    </div>
                                </div>
                                <div class="feature-card border-l-4 border-green-500">
                                    <p class="font-mono text-sm font-semibold text-green-600 dark:text-green-400">boot()</p>
                                    <div class="text-sm text-zinc-700 dark:text-zinc-300 mt-2 space-y-1">
                                        <p class="text-green-600 dark:text-green-400">✅ Fazer:</p>
                                        <ul class="list-disc list-inside text-xs">
                                            <li>Registar eventos e listeners</li>
                                            <li>Configurar middleware</li>
                                            <li>Carregar rotas</li>
                                        </ul>
                                        <p class="text-red-600 dark:text-red-400 mt-2">❌ Não fazer:</p>
                                        <ul class="list-disc list-inside text-xs">
                                            <li>Trabalho pesado (usar lazy loading)</li>
                                        </ul>
                                    </div>
                                </div>
                            </div>
                        </section>

                        {{-- Quick Start --}}
                        <section id="criar-provider" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <div class="prose prose-zinc dark:prose-invert max-w-none">
                                <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                    <flux:icon.rocket-launch class="size-6 text-green-500" />
                                    Criar um Provider em 3 Passos
                                </h2>
                            </div>

                            <div class="space-y-4">
                                {{-- Step 1 --}}
                                <div class="feature-card">
                                    <div class="flex items-center gap-2 mb-3">
                                        <span class="inline-flex size-6 items-center justify-center rounded-full bg-primary text-white text-xs font-bold">1</span>
                                        <p class="font-semibold text-zinc-900 dark:text-white">Criar a classe</p>
                                    </div>
                                    <div class="rounded-xl bg-zinc-950 p-4 text-sm overflow-x-auto">
                                        <pre class="language-php"><code>namespace Og\Modules\MeuModulo\Providers;

use Og\Modules\Common\Providers\ServiceProvider;
use Og\Modules\MeuModulo\Services\MeuServico;
use Og\Modules\MeuModulo\Contracts\MeuServicoInterface;

class MeuModuloServiceProvider extends ServiceProvider
{
    public function register(): void
    {
        $this->app->singleton(MeuServicoInterface::class, function($app) {
            return new MeuServico(
                $app->get('database'),
                $app->get('config')
            );
        });
    }

    public function boot(): void
    {
        // Inicialização após todos os serviços registados
    }
}</code></pre>
                                    </div>
                                </div>

                                {{-- Step 2 --}}
                                <div class="feature-card">
                                    <div class="flex items-center gap-2 mb-3">
                                        <span class="inline-flex size-6 items-center justify-center rounded-full bg-primary text-white text-xs font-bold">2</span>
                                        <p class="font-semibold text-zinc-900 dark:text-white">Registar no Container</p>
                                    </div>
                                    <div class="rounded-xl bg-zinc-950 p-4 text-sm overflow-x-auto">
                                        <pre class="language-php"><code>// Em Container.php:registerServiceProviders()
$this->providerManager->register([
    // ... outros providers
    MeuModuloServiceProvider::class,
]);</code></pre>
                                    </div>
                                </div>

                                {{-- Step 3 --}}
                                <div class="feature-card">
                                    <div class="flex items-center gap-2 mb-3">
                                        <span class="inline-flex size-6 items-center justify-center rounded-full bg-primary text-white text-xs font-bold">3</span>
                                        <p class="font-semibold text-zinc-900 dark:text-white">Usar o serviço</p>
                                    </div>
                                    <div class="rounded-xl bg-zinc-950 p-4 text-sm overflow-x-auto">
                                        <pre class="language-php"><code>// Via container
$servico = app(MeuServicoInterface::class);

// Via DI no construtor
class MeuController {
    public function __construct(
        private MeuServicoInterface $servico
    ) {}
}</code></pre>
                                    </div>
                                </div>
                            </div>
                        </section>

                        {{-- 16 System Providers --}}
                        <section id="providers-sistema" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <div class="prose prose-zinc dark:prose-invert max-w-none">
                                <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                    <flux:icon.server-stack class="size-6 text-purple-500" />
                                    16 Providers do Sistema
                                </h2>
                                <p class="text-zinc-700 dark:text-zinc-300">
                                    Os providers são registados nesta ordem específica em <code>Container.php</code>:
                                </p>
                            </div>

                            <div class="rounded-xl bg-zinc-950 p-4 text-sm overflow-x-auto">
                                <pre class="language-php"><code>$this->providerManager->register([
    EventServiceProvider::class,        // 1.  Sistema de eventos
    HelperServiceProvider::class,       // 2.  Helpers globais
    ConfigServiceProvider::class,       // 3.  Configuração
    QueueServiceProvider::class,        // 4.  Sistema de filas
    RedisServiceProvider::class,        // 5.  Conexões Redis
    CacheServiceProvider::class,        // 6.  Sistema de cache
    HttpClientServiceProvider::class,   // 7.  Cliente HTTP
    SaftPtServiceProvider::class,       // 8.  SAFT Portugal
    SaftAoServiceProvider::class,       // 9.  SAFT Angola
    ValidationServiceProvider::class,   // 10. Validação
    GlobalServiceProvider::class,       // 11. Bridge legacy ⚠️
    ViewServiceProvider::class,         // 12. Templates
    NotificationServiceProvider::class, // 13. Notificações
    RoutingServiceProvider::class,      // 14. Routing
    GuardServiceProvider::class,        // 15. Autenticação
    QueueMasterServiceProvider::class,  // 16. Queue Master
]);</code></pre>
                            </div>

                            <div class="overflow-x-auto">
                                <table class="w-full text-sm text-left">
                                    <thead class="text-xs uppercase bg-zinc-100 dark:bg-zinc-800">
                                        <tr>
                                            <th class="px-4 py-3">Grupo</th>
                                            <th class="px-4 py-3">Providers</th>
                                            <th class="px-4 py-3">Razão da Ordem</th>
                                        </tr>
                                    </thead>
                                    <tbody class="divide-y divide-zinc-200 dark:divide-zinc-700">
                                        <tr class="bg-white dark:bg-zinc-900">
                                            <td class="px-4 py-3 font-medium text-blue-600">Infraestrutura</td>
                                            <td class="px-4 py-3">Events, Helpers, Config, Redis</td>
                                            <td class="px-4 py-3">Base para tudo</td>
                                        </tr>
                                        <tr class="bg-zinc-50 dark:bg-zinc-800/50">
                                            <td class="px-4 py-3 font-medium text-green-600">Core</td>
                                            <td class="px-4 py-3">Queue, Cache, HTTP</td>
                                            <td class="px-4 py-3">Usados por outros providers</td>
                                        </tr>
                                        <tr class="bg-white dark:bg-zinc-900">
                                            <td class="px-4 py-3 font-medium text-purple-600">Externos</td>
                                            <td class="px-4 py-3">SAFT PT/AO</td>
                                            <td class="px-4 py-3">Independentes</td>
                                        </tr>
                                        <tr class="bg-zinc-50 dark:bg-zinc-800/50">
                                            <td class="px-4 py-3 font-medium text-amber-600">Legacy</td>
                                            <td class="px-4 py-3">GlobalServiceProvider</td>
                                            <td class="px-4 py-3">Bridge $u, $a, $s</td>
                                        </tr>
                                        <tr class="bg-white dark:bg-zinc-900">
                                            <td class="px-4 py-3 font-medium text-cyan-600">Aplicação</td>
                                            <td class="px-4 py-3">View, Notification, Routing</td>
                                            <td class="px-4 py-3">Dependem do legacy</td>
                                        </tr>
                                        <tr class="bg-zinc-50 dark:bg-zinc-800/50">
                                            <td class="px-4 py-3 font-medium text-red-600">Segurança</td>
                                            <td class="px-4 py-3">Guard</td>
                                            <td class="px-4 py-3">Após routing</td>
                                        </tr>
                                    </tbody>
                                </table>
                            </div>

                            {{-- GlobalServiceProvider highlight --}}
                            <div class="rounded-xl border-l-4 border-amber-500 bg-amber-50 dark:bg-amber-900/20 p-4">
                                <div class="flex items-center gap-2 mb-2">
                                    <flux:icon.exclamation-triangle class="size-5 text-amber-600 dark:text-amber-400" />
                                    <p class="font-semibold text-amber-800 dark:text-amber-200">GlobalServiceProvider — O Mais Importante</p>
                                </div>
                                <p class="text-sm text-amber-900 dark:text-amber-100 mb-3">
                                    Este provider faz a ponte entre o sistema moderno e o código legacy:
                                </p>
                                <div class="grid gap-2 md:grid-cols-3 text-xs font-mono">
                                    <div class="bg-amber-100 dark:bg-amber-800/30 px-2 py-1 rounded"><code>'database'</code> → $GLOBALS['db']</div>
                                    <div class="bg-amber-100 dark:bg-amber-800/30 px-2 py-1 rounded"><code>'user'</code> → $GLOBALS['u']</div>
                                    <div class="bg-amber-100 dark:bg-amber-800/30 px-2 py-1 rounded"><code>'app'</code> → $GLOBALS['a']</div>
                                    <div class="bg-amber-100 dark:bg-amber-800/30 px-2 py-1 rounded"><code>'session'</code> → $GLOBALS['s']</div>
                                    <div class="bg-amber-100 dark:bg-amber-800/30 px-2 py-1 rounded"><code>'email'</code> → $GLOBALS['mail']</div>
                                    <div class="bg-amber-100 dark:bg-amber-800/30 px-2 py-1 rounded"><code>'hooks'</code> → $GLOBALS['hooks']</div>
                                </div>
                            </div>
                        </section>

                        {{-- Binding Types --}}
                        <section id="tipos-binding" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <div class="prose prose-zinc dark:prose-invert max-w-none">
                                <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                    <flux:icon.link class="size-6 text-blue-500" />
                                    Tipos de Bindings
                                </h2>
                            </div>

                            <div class="grid gap-4 md:grid-cols-2">
                                <div class="feature-card">
                                    <p class="font-mono text-sm font-semibold text-blue-600 dark:text-blue-400 mb-2">singleton() — Instância Única</p>
                                    <div class="rounded-lg bg-zinc-950 p-3 text-xs overflow-x-auto">
                                        <pre class="language-php"><code>$this->app->singleton('database', fn($app) =>
    new DatabaseConnection($app->get('config'))
);

// Mesma instância sempre
$db1 = app('database');
$db2 = app('database');
$db1 === $db2;  // true</code></pre>
                                    </div>
                                </div>
                                <div class="feature-card">
                                    <p class="font-mono text-sm font-semibold text-green-600 dark:text-green-400 mb-2">bind() — Nova Instância</p>
                                    <div class="rounded-lg bg-zinc-950 p-3 text-xs overflow-x-auto">
                                        <pre class="language-php"><code>$this->app->bind(RequestHandler::class, fn() =>
    new RequestHandler()
);

// Instâncias diferentes
$h1 = app(RequestHandler::class);
$h2 = app(RequestHandler::class);
$h1 === $h2;  // false</code></pre>
                                    </div>
                                </div>
                                <div class="feature-card">
                                    <p class="font-mono text-sm font-semibold text-purple-600 dark:text-purple-400 mb-2">instance() — Já Criada</p>
                                    <div class="rounded-lg bg-zinc-950 p-3 text-xs overflow-x-auto">
                                        <pre class="language-php"><code>// Registar instância existente
$user = new User($id);
$this->app->instance('current-user', $user);</code></pre>
                                    </div>
                                </div>
                                <div class="feature-card">
                                    <p class="font-mono text-sm font-semibold text-amber-600 dark:text-amber-400 mb-2">alias() — Nome Alternativo</p>
                                    <div class="rounded-lg bg-zinc-950 p-3 text-xs overflow-x-auto">
                                        <pre class="language-php"><code>$this->app->singleton(DatabaseConnection::class, ...);
$this->app->alias(DatabaseConnection::class, 'db');

// Ambos funcionam:
app(DatabaseConnection::class);
app('db');</code></pre>
                                    </div>
                                </div>
                            </div>
                        </section>

                        {{-- Full Example --}}
                        <section id="exemplo-completo" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <div class="prose prose-zinc dark:prose-invert max-w-none">
                                <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                    <flux:icon.academic-cap class="size-6 text-purple-500" />
                                    Exemplo Real: NotificationServiceProvider
                                </h2>
                                <p class="text-sm text-zinc-700 dark:text-zinc-300">
                                    <code>Modules/Common/Notifications/Providers/NotificationServiceProvider.php</code>
                                </p>
                            </div>

                            <div class="rounded-xl bg-zinc-950 p-4 text-sm overflow-x-auto">
                                <pre class="language-php"><code>namespace Og\Modules\Common\Notifications\Providers;

use Og\Modules\Common\Notifications\Channels\DatabaseChannel;
use Og\Modules\Common\Notifications\Channels\EmailChannel;
use Og\Modules\Common\Notifications\Channels\MattermostChannel;
use Og\Modules\Common\Notifications\Formatters\ContextFormatter;
use Og\Modules\Common\Notifications\Formatters\JobContextFormatter;
use Og\Modules\Common\Notifications\Formatters\StackTraceFormatter;
use Og\Modules\Common\Notifications\Services\NotificationService;
use Og\Modules\Common\Providers\ServiceProvider;

class NotificationServiceProvider extends ServiceProvider
{
    public function register(): void
    {
        $this->app->singleton(NotificationService::class, function ($app) {
            // Configuração via parâmetro da aplicação
            $debugEnabled = forceBoolean($app['app']->getParam('NOTIFICATION_DEBUG', true));
            $service = new NotificationService(isDebugEnabled: $debugEnabled);

            // Registar formatters
            $service->addFormatter('context', new ContextFormatter());
            $service->addFormatter('job_context', new JobContextFormatter());
            $service->addFormatter('exception', new StackTraceFormatter());

            // Canais sempre disponíveis
            $service->addChannel(new DatabaseChannel($app->get('message-center')));
            $service->addChannel(new EmailChannel($app->get('email')));

            // Canal condicional (Mattermost só se configurado)
            if (config('notifications.channels.mattermost.enabled', false)) {
                $service->addChannel(
                    new MattermostChannel(
                        config('notifications.channels.mattermost.api_url'),
                        config('notifications.channels.mattermost.token'),
                        config('notifications.channels.mattermost.channel_id')
                    )
                );
            }

            return $service;
        });
    }
}</code></pre>
                            </div>

                            <div class="grid gap-3 md:grid-cols-3">
                                <div class="feature-card border-l-4 border-blue-500">
                                    <p class="font-semibold text-blue-600 dark:text-blue-400 text-sm mb-1">Formatters</p>
                                    <p class="text-xs text-zinc-600 dark:text-zinc-400">Adiciona formatters para diferentes tipos de contexto</p>
                                </div>
                                <div class="feature-card border-l-4 border-green-500">
                                    <p class="font-semibold text-green-600 dark:text-green-400 text-sm mb-1">Canais Base</p>
                                    <p class="text-xs text-zinc-600 dark:text-zinc-400">Database e Email sempre disponíveis</p>
                                </div>
                                <div class="feature-card border-l-4 border-amber-500">
                                    <p class="font-semibold text-amber-600 dark:text-amber-400 text-sm mb-1">Canal Condicional</p>
                                    <p class="text-xs text-zinc-600 dark:text-zinc-400">Mattermost só se habilitado no config</p>
                                </div>
                            </div>
                        </section>

                        {{-- Best Practices --}}
                        <section id="boas-praticas" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <div class="prose prose-zinc dark:prose-invert max-w-none">
                                <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                    <flux:icon.check-badge class="size-6 text-green-500" />
                                    Boas Práticas
                                </h2>
                            </div>

                            <div class="grid gap-4 md:grid-cols-2">
                                <div class="feature-card border-l-4 border-green-500">
                                    <p class="font-semibold text-green-600 dark:text-green-400 mb-2">✅ Usar Interfaces</p>
                                    <div class="rounded-lg bg-zinc-950 p-3 text-xs overflow-x-auto">
                                        <pre class="language-php"><code>// BOM - depende de interface
$this->app->bind(
    CacheInterface::class,
    RedisCache::class
);</code></pre>
                                    </div>
                                </div>
                                <div class="feature-card border-l-4 border-green-500">
                                    <p class="font-semibold text-green-600 dark:text-green-400 mb-2">✅ Lazy Loading</p>
                                    <div class="rounded-lg bg-zinc-950 p-3 text-xs overflow-x-auto">
                                        <pre class="language-php"><code>// BOM - só cria quando usado
$this->app->singleton('heavy', fn($app) =>
    new HeavyService()
);</code></pre>
                                    </div>
                                </div>
                                <div class="feature-card border-l-4 border-red-500">
                                    <p class="font-semibold text-red-600 dark:text-red-400 mb-2">❌ Resolver no Register</p>
                                    <div class="rounded-lg bg-zinc-950 p-3 text-xs overflow-x-auto">
                                        <pre class="language-php"><code>public function register(): void
{
    // MAU - não fazer!
    $service = $this->app->make(MyService::class);
    $service->setup();
}</code></pre>
                                    </div>
                                </div>
                                <div class="feature-card border-l-4 border-red-500">
                                    <p class="font-semibold text-red-600 dark:text-red-400 mb-2">❌ Instance Pesada</p>
                                    <div class="rounded-lg bg-zinc-950 p-3 text-xs overflow-x-auto">
                                        <pre class="language-php"><code>// MAU - cria imediatamente
$this->app->instance(
    'heavy',
    new HeavyService() // Sem lazy!
);</code></pre>
                                    </div>
                                </div>
                            </div>
                        </section>

                        {{-- Navigation --}}
                        <div class="flex justify-between items-center pt-8 border-t border-zinc-200 dark:border-zinc-800">
                            <a href="{{ route('docs.show', 'container') }}" class="inline-flex items-center gap-2 text-sm text-zinc-600 dark:text-zinc-400 hover:text-primary">
                                <flux:icon.arrow-left class="size-4" />
                                Container
                            </a>
                            <a href="{{ route('docs.show', 'auth') }}" class="inline-flex items-center gap-2 text-sm font-medium text-primary hover:underline">
                                Autenticação
                                <flux:icon.arrow-right class="size-4" />
                            </a>
                        </div>
                    </div>

                </div>
            </div>

            {{-- Right Sidebar: Table of Contents --}}
            @include('docs.partials.toc', ['sections' => [
                'introducao' => 'Por que usar?',
                'ciclo-de-vida' => 'Register & Boot',
                'criar-provider' => 'Criar Provider',
                'providers-sistema' => '16 Providers',
                'tipos-binding' => 'Tipos de Binding',
                'exemplo-completo' => 'Exemplo Completo',
                'boas-praticas' => 'Boas Práticas',
            ]])
        </div>
    </div>
@endsection
