@extends('layouts.docs')

@section('title', 'Sistema de Validação - OG Framework')

@section('body')
    <div class="relative min-h-screen bg-transparent px-4 pb-12 pt-6 sm:px-6 lg:px-8">
        {{-- Floating shapes for styling --}}
        <div class="floating-blur cyan -left-10 -top-10 hidden lg:block"></div>
        <div class="floating-blur purple right-10 top-32 hidden lg:block"></div>

        <div class="mx-auto flex w-full max-w-7xl gap-8 bg-transparent">
            {{-- Left Sidebar: Navigation --}}
            @include('docs.partials.sidebar')

            {{-- Main Content --}}
            <div class="flex-1 min-w-0">
                <div class="relative overflow-hidden rounded-3xl border border-white/50 bg-white/85 p-8 shadow-xl shadow-primary/20 backdrop-blur dark:border-zinc-800 dark:bg-zinc-900/80">

                    {{-- Page Header --}}
                    <div class="mb-8">
                        <a href="{{ route('docs.index') }}" class="inline-flex items-center gap-1 text-sm text-primary hover:underline mb-4">
                            <flux:icon.arrow-left class="size-4" />
                            Voltar para Documentação
                        </a>
                        <p class="text-xs uppercase tracking-[0.3em] text-primary dark:text-blue-200">Forms & Data</p>
                        <h1 class="font-display text-4xl font-bold text-zinc-900 dark:text-white sm:text-5xl">Sistema de Validação</h1>
                        <p class="mt-3 text-base text-zinc-700 dark:text-zinc-200 max-w-3xl">
                            Motor robusto e extensível para validação de dados. Baseado nos padrões <strong>Strategy</strong>, <strong>Rule Pattern</strong> e <strong>Chain of Responsibility</strong>, oferece uma API fluente para validação de requests HTTP, forms e dados de API.
                        </p>
                    </div>

                    <div class="space-y-10">

                        {{-- Architecture SVG Diagram --}}
                        <section id="arquitectura" class="scroll-mt-28 space-y-4">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.cube-transparent class="size-6 text-sky-500" />
                                Arquitectura
                            </h2>

                            {{-- SVG: Validation Flow --}}
                            <div class="rounded-xl border border-zinc-200 bg-white p-6 dark:border-zinc-800 dark:bg-zinc-900/50 overflow-x-auto">
                                <svg viewBox="0 0 750 180" class="w-full max-w-3xl mx-auto" xmlns="http://www.w3.org/2000/svg">
                                    {{-- Input Sources --}}
                                    <g transform="translate(10,20)">
                                        <rect x="0" y="0" width="130" height="140" rx="10" fill="#fef3c7" stroke="#f59e0b" stroke-width="1.5"/>
                                        <text x="65" y="20" text-anchor="middle" class="text-[10px] font-semibold" fill="#92400e">Origem dos Dados</text>
                                        
                                        <rect x="10" y="35" width="110" height="28" rx="4" fill="#ffffff" stroke="#fcd34d"/>
                                        <text x="65" y="53" text-anchor="middle" class="text-[9px]" fill="#78350f">HTTP Request</text>
                                        
                                        <rect x="10" y="70" width="110" height="28" rx="4" fill="#ffffff" stroke="#fcd34d"/>
                                        <text x="65" y="88" text-anchor="middle" class="text-[9px]" fill="#78350f">Form Data</text>
                                        
                                        <rect x="10" y="105" width="110" height="28" rx="4" fill="#ffffff" stroke="#fcd34d"/>
                                        <text x="65" y="123" text-anchor="middle" class="text-[9px]" fill="#78350f">API Payload</text>
                                    </g>
                                    
                                    {{-- Arrow --}}
                                    <path d="M150,90 L220,90" stroke="#94a3b8" stroke-width="2" marker-end="url(#arrowheadV)"/>
                                    
                                    {{-- Validator Core --}}
                                    <g transform="translate(230,20)">
                                        <rect x="0" y="0" width="250" height="140" rx="12" fill="#e0f2fe" stroke="#0284c7" stroke-width="2"/>
                                        <text x="125" y="22" text-anchor="middle" class="text-sm font-bold" fill="#0369a1">Motor de Validação</text>
                                        
                                        {{-- Validator --}}
                                        <rect x="15" y="35" width="100" height="45" rx="6" fill="#ffffff" stroke="#bae6fd"/>
                                        <text x="65" y="55" text-anchor="middle" class="text-[10px] font-semibold" fill="#0c4a6e">Validator</text>
                                        <text x="65" y="70" text-anchor="middle" class="text-[8px]" fill="#64748b">702 lines</text>
                                        
                                        {{-- Rule Sets --}}
                                        <rect x="130" y="35" width="105" height="95" rx="6" fill="#ffffff" stroke="#bae6fd"/>
                                        <text x="182" y="52" text-anchor="middle" class="text-[10px] font-semibold" fill="#0c4a6e">Rule Sets</text>
                                        <text x="182" y="68" text-anchor="middle" class="text-[8px]" fill="#64748b">• CommonRule</text>
                                        <text x="182" y="81" text-anchor="middle" class="text-[8px]" fill="#64748b">• DateRule</text>
                                        <text x="182" y="94" text-anchor="middle" class="text-[8px]" fill="#64748b">• FormatRule</text>
                                        <text x="182" y="107" text-anchor="middle" class="text-[8px]" fill="#64748b">• FileRule</text>
                                        <text x="182" y="120" text-anchor="middle" class="text-[8px]" fill="#64748b">• Custom Rules</text>
                                        
                                        {{-- BaseRequest --}}
                                        <rect x="15" y="90" width="100" height="35" rx="6" fill="#ffffff" stroke="#bae6fd"/>
                                        <text x="65" y="112" text-anchor="middle" class="text-[9px]" fill="#0c4a6e">BaseRequest</text>
                                    </g>
                                    
                                    {{-- Arrow --}}
                                    <path d="M490,90 L560,90" stroke="#94a3b8" stroke-width="2" marker-end="url(#arrowheadV)"/>
                                    
                                    {{-- Output --}}
                                    <g transform="translate(570,30)">
                                        <rect x="0" y="0" width="160" height="120" rx="10" fill="#f0fdf4" stroke="#22c55e" stroke-width="1.5"/>
                                        <text x="80" y="20" text-anchor="middle" class="text-[10px] font-semibold" fill="#166534">Resultado</text>
                                        
                                        <rect x="15" y="35" width="130" height="30" rx="4" fill="#dcfce7" stroke="#86efac"/>
                                        <text x="80" y="55" text-anchor="middle" class="text-[10px] font-mono" fill="#166534">validated()</text>
                                        
                                        <rect x="15" y="75" width="130" height="30" rx="4" fill="#fee2e2" stroke="#fca5a5"/>
                                        <text x="80" y="95" text-anchor="middle" class="text-[10px] font-mono" fill="#991b1b">errors()</text>
                                    </g>
                                    
                                    {{-- Arrowhead --}}
                                    <defs>
                                        <marker id="arrowheadV" markerWidth="10" markerHeight="7" refX="9" refY="3.5" orient="auto">
                                            <polygon points="0 0, 10 3.5, 0 7" fill="#94a3b8"/>
                                        </marker>
                                    </defs>
                                </svg>
                            </div>
                        </section>

                        {{-- Helper Functions --}}
                        <section id="helpers" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.code-bracket class="size-6 text-blue-500" />
                                Helpers Globais
                            </h2>

                            <div class="space-y-4">
                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-bold text-zinc-900 dark:text-white mb-1 text-sm font-mono">validate()</h3>
                                    <p class="text-xs text-zinc-600 dark:text-zinc-400 mb-2">Valida dados contra regras</p>
                                    <pre class="text-xs bg-zinc-950 p-2 rounded text-zinc-300"><code>$passes = validate($data, [
    'email' => 'required|email',
    'name' => 'required|min_length[2]'
]);</code></pre>
                                </div>

                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-bold text-zinc-900 dark:text-white mb-1 text-sm font-mono">validationErrors()</h3>
                                    <p class="text-xs text-zinc-600 dark:text-zinc-400 mb-2">Obtém erros de validação</p>
                                    <pre class="text-xs bg-zinc-950 p-2 rounded text-zinc-300"><code>if (!$passes) {
    $errors = validationErrors();
    // ['email' => 'O email é inválido']
}</code></pre>
                                </div>

                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-bold text-zinc-900 dark:text-white mb-1 text-sm font-mono">validated()</h3>
                                    <p class="text-xs text-zinc-600 dark:text-zinc-400 mb-2">Obtém dados validados</p>
                                    <pre class="text-xs bg-zinc-950 p-2 rounded text-zinc-300"><code>$data = validated();
// ['email' => 'test@ex.com', 
//  'name' => 'João']</code></pre>
                                </div>
                            </div>
                        </section>

                        {{-- BaseRequest --}}
                        <section id="base-request" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.inbox-arrow-down class="size-6 text-green-500" />
                                BaseRequest
                            </h2>
                            <p class="text-zinc-700 dark:text-zinc-300">
                                Classe abstracta para validação automática de HTTP requests:
                            </p>

                            <div class="rounded-xl bg-zinc-950 p-4 text-sm overflow-x-auto">
                                <pre class="language-php"><code>class AppointmentsCreateRequest extends BaseRequest
{
    public function authorize(): bool
    {
        return true; // Ou lógica de permissão
    }

    public function rules(): array
    {
        return [
            'subject'    => ['required', 'string'],
            'start_date' => ['required', 'date_format[Y-m-d]'],
            'end_date'   => ['required', 'after_or_equal[start_date]'],
            
            // Condicionais
            'seller_id'  => ['if_exist', 'integer', 'exists[comerciais,codrepres]'],
            
            // Arrays
            'contacts'   => ['if_exist', 'array'],
            'contacts.*' => ['string'],
        ];
    }
}

// Uso no Controller
public function store()
{
    $request = new AppointmentsCreateRequest();
    return Appointment::create($request->validated());
}</code></pre>
                            </div>
                        </section>

                        {{-- Rule Factory --}}
                        <section id="rule-factory" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.wrench-screwdriver class="size-6 text-purple-500" />
                                Rule Factory
                            </h2>
                            <p class="text-zinc-700 dark:text-zinc-300">
                                A classe <code>Rule</code> fornece criação fluente de rules complexas:
                            </p>

                            <div class="space-y-4">
                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-bold text-zinc-900 dark:text-white mb-2 text-sm">Unicidade com Ignore</h3>
                                    <pre class="text-xs bg-zinc-950 p-3 rounded-lg text-zinc-300"><code>use Og\Modules\Common\Validation\Rules\Rule;

$rules = [
    'email' => [
        'required',
        'email',
        Rule::unique('users', 'email')
            ->ignore($userId, 'id')
    ]
];</code></pre>
                                </div>

                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-bold text-zinc-900 dark:text-white mb-2 text-sm">Existência e Enums</h3>
                                    <pre class="text-xs bg-zinc-950 p-3 rounded-lg text-zinc-300"><code>use Og\Modules\Common\Validation\Rules\Rule;

$rules = [
    'category_id' => [
        'required',
        'exists[categories,id,{active:1}]'
    ],
    'status' => Rule::enum(OrderStatus::class),
    'type'   => Rule::in(['A', 'B', 'C']),
];</code></pre>
                                </div>
                            </div>
                        </section>

                        {{-- Rule Catalog --}}
                        <section id="catalogo-rules" class="scroll-mt-28 space-y-6 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.book-open class="size-6 text-amber-500" />
                                Catálogo de Rules
                            </h2>

                            {{-- CommonRule --}}
                            <div class="rounded-xl border border-blue-200 bg-blue-50 p-4 dark:border-blue-900/50 dark:bg-blue-900/20">
                                <h3 class="font-semibold text-blue-900 dark:text-blue-200 mb-3 text-sm">CommonRule — Regras Básicas</h3>
                                <div class="overflow-x-auto">
                                    <table class="min-w-full text-xs">
                                        <thead>
                                            <tr class="text-left text-blue-700 dark:text-blue-300">
                                                <th class="pr-4 py-1">Rule</th>
                                                <th class="pr-4 py-1">Sintaxe</th>
                                                <th class="py-1">Descrição</th>
                                            </tr>
                                        </thead>
                                        <tbody class="text-blue-800 dark:text-blue-200 font-mono">
                                            <tr><td class="pr-4 py-1">required</td><td class="pr-4">required</td><td class="font-sans">Campo obrigatório</td></tr>
                                            <tr><td class="pr-4 py-1">min_length</td><td class="pr-4">min_length[5]</td><td class="font-sans">Mínimo de caracteres</td></tr>
                                            <tr><td class="pr-4 py-1">max_length</td><td class="pr-4">max_length[255]</td><td class="font-sans">Máximo de caracteres</td></tr>
                                            <tr><td class="pr-4 py-1">in_list</td><td class="pr-4">in_list[A,B,C]</td><td class="font-sans">Valor na lista</td></tr>
                                            <tr><td class="pr-4 py-1">is_unique</td><td class="pr-4">is_unique[table.col]</td><td class="font-sans">Único na tabela</td></tr>
                                            <tr><td class="pr-4 py-1">exists</td><td class="pr-4">exists[table,col]</td><td class="font-sans">Registo existe</td></tr>
                                            <tr><td class="pr-4 py-1">greater_than</td><td class="pr-4">greater_than[0]</td><td class="font-sans">Maior que valor</td></tr>
                                            <tr><td class="pr-4 py-1">required_with</td><td class="pr-4">required_with[f1,f2]</td><td class="font-sans">Obrigatório se campos preenchidos</td></tr>
                                        </tbody>
                                    </table>
                                </div>
                            </div>

                            {{-- DateRule --}}
                            <div class="rounded-xl border border-green-200 bg-green-50 p-4 dark:border-green-900/50 dark:bg-green-900/20">
                                <h3 class="font-semibold text-green-900 dark:text-green-200 mb-3 text-sm">DateRule — Regras Temporais</h3>
                                <div class="overflow-x-auto">
                                    <table class="min-w-full text-xs">
                                        <thead>
                                            <tr class="text-left text-green-700 dark:text-green-300">
                                                <th class="pr-4 py-1">Rule</th>
                                                <th class="pr-4 py-1">Sintaxe</th>
                                                <th class="py-1">Descrição</th>
                                            </tr>
                                        </thead>
                                        <tbody class="text-green-800 dark:text-green-200 font-mono">
                                            <tr><td class="pr-4 py-1">before</td><td class="pr-4">before[field]</td><td class="font-sans">Antes de outra data</td></tr>
                                            <tr><td class="pr-4 py-1">after</td><td class="pr-4">after[field]</td><td class="font-sans">Depois de outra data</td></tr>
                                            <tr><td class="pr-4 py-1">after_or_equal</td><td class="pr-4">after_or_equal[field]</td><td class="font-sans">Depois ou igual</td></tr>
                                            <tr><td class="pr-4 py-1">date_format</td><td class="pr-4">date_format[Y-m-d]</td><td class="font-sans">Formato específico</td></tr>
                                            <tr><td class="pr-4 py-1">is_future</td><td class="pr-4">is_future</td><td class="font-sans">Data no futuro</td></tr>
                                            <tr><td class="pr-4 py-1">is_past</td><td class="pr-4">is_past</td><td class="font-sans">Data no passado</td></tr>
                                        </tbody>
                                    </table>
                                </div>
                            </div>

                            {{-- FormatRule --}}
                            <div class="rounded-xl border border-purple-200 bg-purple-50 p-4 dark:border-purple-900/50 dark:bg-purple-900/20">
                                <h3 class="font-semibold text-purple-900 dark:text-purple-200 mb-3 text-sm">FormatRule — Regras de Formato</h3>
                                <div class="overflow-x-auto">
                                    <table class="min-w-full text-xs">
                                        <thead>
                                            <tr class="text-left text-purple-700 dark:text-purple-300">
                                                <th class="pr-4 py-1">Rule</th>
                                                <th class="pr-4 py-1">Sintaxe</th>
                                                <th class="py-1">Descrição</th>
                                            </tr>
                                        </thead>
                                        <tbody class="text-purple-800 dark:text-purple-200 font-mono">
                                            <tr><td class="pr-4 py-1">email</td><td class="pr-4">email</td><td class="font-sans">Email válido</td></tr>
                                            <tr><td class="pr-4 py-1">url</td><td class="pr-4">url</td><td class="font-sans">URL válida</td></tr>
                                            <tr><td class="pr-4 py-1">regex_match</td><td class="pr-4">regex_match[/pattern/]</td><td class="font-sans">Match de regex</td></tr>
                                            <tr><td class="pr-4 py-1">valid_json</td><td class="pr-4">valid_json</td><td class="font-sans">JSON válido</td></tr>
                                            <tr><td class="pr-4 py-1">alpha_dash</td><td class="pr-4">alpha_dash</td><td class="font-sans">Letras, números, _, -</td></tr>
                                            <tr><td class="pr-4 py-1">integer</td><td class="pr-4">integer</td><td class="font-sans">Deve ser inteiro</td></tr>
                                        </tbody>
                                    </table>
                                </div>
                            </div>

                            {{-- FileRule --}}
                            <div class="rounded-xl border border-amber-200 bg-amber-50 p-4 dark:border-amber-900/50 dark:bg-amber-900/20">
                                <h3 class="font-semibold text-amber-900 dark:text-amber-200 mb-3 text-sm">FileRule — Regras de Ficheiros</h3>
                                <pre class="text-xs bg-white dark:bg-black/20 p-2 rounded"><code>$rules = [
    'avatar' => [
        'uploaded[avatar]',
        'max_size[avatar,1024]',        // 1MB
        'is_image[avatar]',
        'mime_in[avatar,image/jpeg,image/png,image/webp]'
    ],
    'document' => [
        'uploaded[document]',
        'max_size[document,5120]',      // 5MB
        'ext_in[document,pdf,doc,docx]'
    ],
];</code></pre>
                            </div>
                        </section>

                        {{-- Modifiers --}}
                        <section id="modificadores" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.adjustments-horizontal class="size-6 text-slate-500" />
                                Modificadores Especiais
                            </h2>

                            <div class="space-y-4">
                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-bold text-zinc-900 dark:text-white mb-1 text-sm font-mono">if_exist</h3>
                                    <p class="text-xs text-zinc-600 dark:text-zinc-400 mb-2">Valida apenas se campo existir nos dados</p>
                                    <pre class="text-xs bg-zinc-950 p-2 rounded text-zinc-300"><code>'phone' => ['if_exist', 'string', 'max_length[20]'],
// Campo ausente = válido
// Campo presente = aplica regras</code></pre>
                                </div>

                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-bold text-zinc-900 dark:text-white mb-1 text-sm font-mono">permit_empty</h3>
                                    <p class="text-xs text-zinc-600 dark:text-zinc-400 mb-2">Permite valores vazios, mas valida se preenchido</p>
                                    <pre class="text-xs bg-zinc-950 p-2 rounded text-zinc-300"><code>'bio' => ['permit_empty', 'string', 'max_length[500]'],
// Campo vazio = válido
// Campo com valor = valida regras</code></pre>
                                </div>

                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-bold text-zinc-900 dark:text-white mb-1 text-sm font-mono">sometimes</h3>
                                    <p class="text-xs text-zinc-600 dark:text-zinc-400 mb-2">Aplica regras apenas se campo existir no dataset</p>
                                    <pre class="text-xs bg-zinc-950 p-2 rounded text-zinc-300"><code>'email' => ['sometimes', 'required', 'email'],
// Campo ausente = não valida
// Campo presente = email obrigatório</code></pre>
                                </div>

                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-bold text-zinc-900 dark:text-white mb-1 text-sm font-mono">checkbox</h3>
                                    <p class="text-xs text-zinc-600 dark:text-zinc-400 mb-2">Define um valor default quando o campo está ausente/vazio</p>
                                    <pre class="text-xs bg-zinc-950 p-2 rounded text-zinc-300"><code>// Default 0 se vazio
'newsletter' => ['checkbox[0]'],

// Obrigatório como 1
'terms' => ['required', 'checkbox[1]']
// Transforma em: 0 ou 1</code></pre>
                                </div>
                            </div>
                        </section>

                        {{-- Conditionable Trait --}}
                        <section id="conditionable" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.arrows-pointing-in class="size-6 text-teal-500" />
                                Validação Condicional
                            </h2>
                            <p class="text-zinc-700 dark:text-zinc-300">
                                Use <code>Rule::when()</code> para escolher rules em tempo de execução:
                            </p>

                            <div class="space-y-4">
                                <div class="rounded-xl border border-teal-200 bg-teal-50 p-4 dark:border-teal-900/50 dark:bg-teal-900/20">
                                    <h3 class="font-semibold text-teal-900 dark:text-teal-200 mb-2 text-sm">Rule::when() — Condicional</h3>
                                    <pre class="text-xs bg-white dark:bg-black/20 p-2 rounded overflow-x-auto"><code>$isUpdate = $request->has('id');

$rules = [
    'email' => Rule::when(
        $isUpdate,
        // SE update:
        ['sometimes', 'email', Rule::unique('users', 'email')->ignore($userId, 'id')],
        // SENÃO (create):
        ['required', 'email', Rule::unique('users', 'email')]
    ),
];</code></pre>
                                </div>

                                <div class="rounded-xl border border-teal-200 bg-teal-50 p-4 dark:border-teal-900/50 dark:bg-teal-900/20">
                                    <h3 class="font-semibold text-teal-900 dark:text-teal-200 mb-2 text-sm">Com Callback</h3>
                                    <pre class="text-xs bg-white dark:bg-black/20 p-2 rounded overflow-x-auto"><code>$rules = [
    'phone' => Rule::when(
        fn () => $data['contact_method'] === 'phone',
        ['required', 'regex_match[/^\\+\\d+$/]'],
        ['nullable']
    ),
    'category_id' => Rule::when(
        $data['use_default_category'] ?? false,
        ['nullable'],
        ['required', 'exists[categories,id,{active:1}]']
    ),
];</code></pre>
                                </div>
                            </div>
                        </section>

                        {{-- Custom Messages --}}
                        <section id="after-callbacks" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.clock class="size-6 text-orange-500" />
                                After (pós-validação)
                            </h2>
                            <p class="text-zinc-700 dark:text-zinc-300">
                                Use <code>after()</code> no <code>BaseRequest</code> para executar validações que dependem de contexto externo
                                (ex.: permissões, estado em base de dados, integrações). O callback recebe o <code>ValidatorInterface</code> e pode
                                adicionar erros com <code>setError()</code> / <code>addError()</code>.
                            </p>

                            <div class="rounded-xl bg-zinc-950 p-4 text-sm overflow-x-auto">
                                <pre class="language-php"><code>use Og\Modules\Common\Http\BaseRequest;
use Og\Modules\Common\Validation\Contracts\ValidatorInterface;
use Og\Modules\Common\Validation\Rules\Rule;

class AmsConciergeNoteRequest extends BaseRequest
{
    public function authorize(): bool
    {
        return true;
    }

    public function rules(): array
    {
        return [
            'subject' => [
                Rule::requiredIf(fn () => $this->request->isPost()),
                'string',
                'max_length[128]',
            ],
        ];
    }

    public function after(): \Closure
    {
        return function (ValidatorInterface $validator): void {
            if (!$this->request->route('concierge')) {
                $validator->setError('concierge', 'Concierge inválido.');
            }
        };
    }
}</code></pre>
                            </div>
                        </section>

                        <section id="mensagens" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.chat-bubble-bottom-center-text class="size-6 text-pink-500" />
                                Mensagens Customizadas
                            </h2>

                            <div class="space-y-4">
                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-bold text-zinc-900 dark:text-white mb-2 text-sm">Em BaseRequest</h3>
                                    <pre class="text-xs bg-zinc-950 p-3 rounded-lg text-zinc-300"><code>class UserRequest extends BaseRequest
{
    public function rules(): array
    {
        return [
            'email' => ['required', 'email'],
            'password' => ['required', 'min_length[8]'],
        ];
    }

    protected function messages(): array
    {
        return [
            'email' => [
                'required' => 'O email é obrigatório.',
                'email' => 'Forneça um email válido.',
            ],
            'password' => [
                'min_length' => 'Mínimo 8 caracteres.',
            ],
        ];
    }
}</code></pre>
                                </div>

                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-bold text-zinc-900 dark:text-white mb-2 text-sm">Por Campo Inline</h3>
                                    <pre class="text-xs bg-zinc-950 p-3 rounded-lg text-zinc-300"><code>$rules = [
    'email' => [
        'label' => 'Endereço de Email',
        'rules' => ['required', 'email'],
        'errors' => [
            'required' => 'Preencha o email.',
            'email' => 'Formato inválido.',
        ]
    ]
];

// Placeholders disponíveis:
// {field} - Nome do campo
// {param} - Parâmetro da rule
// {value} - Valor actual</code></pre>
                                </div>
                            </div>
                        </section>

                        {{-- Dot Notation & Wildcards --}}
                        <section id="wildcards" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.squares-2x2 class="size-6 text-indigo-500" />
                                Dot Notation & Wildcards
                            </h2>
                            <p class="text-zinc-700 dark:text-zinc-300">
                                O sistema suporta validação de estruturas aninhadas e placeholders dinâmicos:
                            </p>

                            <div class="rounded-xl bg-zinc-950 p-4 text-sm overflow-x-auto">
                                <pre class="language-php"><code>$rules = [
    // Acesso aninhado
    'user.name' => ['required', 'string'],
    'user.profile.bio' => ['string', 'max_length[500]'],
    
    // Wildcard para arrays
    'items.*.name' => ['required', 'string'],
    'items.*.price' => ['required', 'numeric', 'greater_than[0]'],
    
    // Placeholders dinâmicos - substituídos pelos valores actuais
    'min_value' => ['required', 'numeric'],
    'max_value' => ['required', 'numeric', 'greater_than[{min_value}]'],
    'value' => ['required', 'numeric', 'greater_than[{min_value}]', 'less_than[{max_value}]'],
    
    // O validator substitui {min_value} por 10 e {max_value} por 100
];</code></pre>
                            </div>
                        </section>

                        {{-- Validator API --}}
                        <section id="validator-api" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.command-line class="size-6 text-cyan-500" />
                                Validator API
                            </h2>
                            <p class="text-zinc-700 dark:text-zinc-300">
                                Métodos do validator (instância retornada por <code>app(ValidatorInterface::class)</code>):
                            </p>

                            <div class="space-y-4">
                                <div class="rounded-xl border border-cyan-200 bg-cyan-50 p-4 dark:border-cyan-900/50 dark:bg-cyan-900/20">
                                    <h3 class="font-semibold text-cyan-900 dark:text-cyan-200 mb-2 text-sm">Configuração & Execução</h3>
                                    <pre class="text-xs bg-white dark:bg-black/20 p-2 rounded"><code>$v = app(ValidatorInterface::class);

// Definir rules
$v->setRule('email', 'Email', 'required|email');
$v->setRules(['email' => 'required', 'name' => 'string']);

// Carregar dados
$v->withRequest($data);
$v->defineData($data);

// Executar
$passes = $v->validate();
$passes = $v->validate($data); // inline

// Validação rápida de valor
$ok = $v->check($email, ['required', 'email']);</code></pre>
                                </div>

                                <div class="rounded-xl border border-cyan-200 bg-cyan-50 p-4 dark:border-cyan-900/50 dark:bg-cyan-900/20">
                                    <h3 class="font-semibold text-cyan-900 dark:text-cyan-200 mb-2 text-sm">Resultados & Erros</h3>
                                    <pre class="text-xs bg-white dark:bg-black/20 p-2 rounded"><code>// Erros
$erros = $v->errors();
$has = $v->hasError('email');
$msg = $v->error('email');

// Dados validados
$all = $v->validated();
$one = $v->validated('email');
$some = $v->validated(['email', 'name']);

// Filtros
$only = $v->only(['email'])->validated();
$except = $v->except(['password'])->validated();

// Adicionar erro manual
$v->addError('field', 'Mensagem');

// Callback after (pós-validação)
$v->after(function (ValidatorInterface $validator): void {
    $validator->setError('field', 'Mensagem');
});</code></pre>
                                </div>
                            </div>
                        </section>

                        {{-- Custom Rules --}}
                        <section id="custom-rules" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.puzzle-piece class="size-6 text-rose-500" />
                                Custom Rules
                            </h2>
                            <p class="text-zinc-700 dark:text-zinc-300">
                                Crie regras personalizadas implementando as interfaces disponíveis:
                            </p>

                            <div class="space-y-4">
                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-bold text-zinc-900 dark:text-white mb-2 text-sm">RuleInterface — Básico</h3>
                                    <pre class="text-xs bg-zinc-950 p-3 rounded-lg text-zinc-300"><code>class ValidNIF implements RuleInterface
{
    public function passes($attr, $value): bool
    {
        $nif = preg_replace('/\s+/', '', $value);
        if (!preg_match('/^[0-9]{9}$/', $nif)) {
            return false;
        }
        // Validar dígito de controlo...
        return (int)$nif[8] === $checkDigit;
    }

    public function message(): string
    {
        return 'O campo {field} deve ter um NIF válido.';
    }
}

// Uso
'nif' => ['required', new ValidNIF()]</code></pre>
                                </div>

                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-bold text-zinc-900 dark:text-white mb-2 text-sm">DataAwareRule — Com Dados</h3>
                                    <p class="text-xs text-zinc-600 dark:text-zinc-400 mb-2">
                                        Quando a rule precisa enxergar outros campos do payload, implemente <code>DataAwareRule</code>.
                                        O validator injeta o dataset completo antes de chamar <code>passes()</code>.
                                    </p>
                                    <pre class="text-xs bg-zinc-950 p-3 rounded-lg text-zinc-300"><code>class UniqueInvoice implements RuleInterface, DataAwareRule
{
    private array $data = [];
    
    public function defineData(array $data): void
    {
        $this->data = $data;
    }

    public function passes($attr, $value): bool
    {
        // Acesso a outros campos
        $companyId = $this->data['company_id'];
        $year = $this->data['year'];
        
        return !Invoice::where('company_id', $companyId)
            ->where('year', $year)
            ->where('number', $value)
            ->exists();
    }
}</code></pre>
                                </div>

                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-bold text-zinc-900 dark:text-white mb-2 text-sm">ValidatorAwareRule — Acesso ao Validator</h3>
                                    <p class="text-xs text-zinc-600 dark:text-zinc-400 mb-2">
                                        Útil quando você precisa adicionar erro em outro campo, ou consultar estado do validator durante a validação.
                                        O validator injeta a instância via <code>defineValidator()</code>.
                                    </p>
                                    <pre class="text-xs bg-zinc-950 p-3 rounded-lg text-zinc-300"><code>use Og\Modules\Common\Validation\Contracts\ValidatorAwareRule;
use Og\Modules\Common\Validation\Contracts\ValidatorInterface;

class RequiresInvoiceIdWhenTypeIsInvoice implements RuleInterface, DataAwareRule, ValidatorAwareRule
{
    private array $data = [];
    private ?ValidatorInterface $validator = null;

    public function defineData(array $data): void
    {
        $this->data = $data;
    }

    public function defineValidator(ValidatorInterface $validator): void
    {
        $this->validator = $validator;
    }

    public function passes($attr, $value): bool
    {
        $type = $this->data['type'] ?? null;
        $invoiceId = $this->data['invoice_id'] ?? null;

        if ($type === 'invoice' && empty($invoiceId)) {
            $this->validator?->setError('invoice_id', 'Informe a fatura para este tipo.');
        }

        return true;
    }

    public function message(): string
    {
        return '';
    }
}</code></pre>
                                </div>

                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-bold text-zinc-900 dark:text-white mb-2 text-sm">CompilableRules (WIP) — ForEach Style</h3>
                                    <p class="text-xs text-zinc-600 dark:text-zinc-400 mb-2">
                                        A interface <code>CompilableRules</code> existe para viabilizar regras “tipo <code>forEach()</code> do Laravel”:
                                        permitir que um objeto gere, em runtime, um conjunto de rules a partir do <code>$attribute</code> e do <code>$value</code>
                                        (inclusive para campos resolvidos por wildcard como <code>items.0.price</code>, <code>items.1.price</code>).
                                    </p>
                                    <p class="text-xs text-zinc-600 dark:text-zinc-400 mb-2">
                                        Status: contrato presente, mas a integração automática no validator pode ainda não estar finalizada.
                                        Enquanto isso, use wildcards (<code>items.*.field</code>) + <code>Rule::when()</code> e/ou rules custom com <code>DataAwareRule</code>.
                                    </p>
                                    <pre class="text-xs bg-zinc-950 p-3 rounded-lg text-zinc-300"><code>use Og\Modules\Common\Validation\Contracts\CompilableRules;

class AmountRules implements CompilableRules
{
    public function compile(string $attribute, mixed $value, array $data = null): array
    {
        $data ??= [];

        $rules = ['numeric', 'greater_than[0]'];
        if (($data['currency'] ?? 'EUR') === 'EUR') {
            $rules[] = 'max_value[999999]';
        }

        return $rules;
    }
}</code></pre>
                                </div>
                            </div>

                            <div class="rounded-xl border border-sky-200 bg-sky-50 p-4 dark:border-sky-900/50 dark:bg-sky-900/20">
                                <h3 class="font-semibold text-sky-900 dark:text-sky-200 mb-2 text-sm">Interfaces Disponíveis</h3>
                                <div class="grid gap-2 md:grid-cols-2 text-xs text-sky-800 dark:text-sky-300">
                                    <div><code>RuleInterface</code> — Implementa <code>passes()</code> e <code>message()</code></div>
                                    <div><code>DataAwareRule</code> — Recebe o payload via <code>defineData()</code></div>
                                    <div><code>ValidatorAwareRule</code> — Recebe o validator via <code>defineValidator()</code></div>
                                    <div><code>CompilableRules</code> — Gera um array de rules via <code>compile()</code></div>
                                </div>
                            </div>
                        </section>

                        {{-- File Rules Detail --}}
                        <section id="file-rules" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.document-arrow-up class="size-6 text-emerald-500" />
                                Validação de Ficheiros
                            </h2>
                            <p class="text-zinc-700 dark:text-zinc-300">
                                Rules especializadas para upload de ficheiros:
                            </p>

                            <div class="overflow-x-auto">
                                <table class="min-w-full divide-y divide-zinc-200 dark:divide-zinc-700 text-sm">
                                    <thead class="bg-zinc-50 dark:bg-zinc-800">
                                        <tr>
                                            <th class="px-4 py-2 text-left text-xs font-medium text-zinc-500 uppercase">Rule</th>
                                            <th class="px-4 py-2 text-left text-xs font-medium text-zinc-500 uppercase">Sintaxe</th>
                                            <th class="px-4 py-2 text-left text-xs font-medium text-zinc-500 uppercase">Descrição</th>
                                        </tr>
                                    </thead>
                                    <tbody class="divide-y divide-zinc-200 dark:divide-zinc-700">
                                        <tr><td class="px-4 py-2 font-mono text-xs">uploaded</td><td class="px-4 py-2 text-xs">uploaded[field]</td><td class="px-4 py-2 text-xs">Ficheiro foi uploaded com sucesso</td></tr>
                                        <tr><td class="px-4 py-2 font-mono text-xs">max_size</td><td class="px-4 py-2 text-xs">max_size[field,2048]</td><td class="px-4 py-2 text-xs">Tamanho máximo em KB</td></tr>
                                        <tr><td class="px-4 py-2 font-mono text-xs">mime_in</td><td class="px-4 py-2 text-xs">mime_in[field,image/png,image/jpeg]</td><td class="px-4 py-2 text-xs">Tipos MIME permitidos</td></tr>
                                        <tr><td class="px-4 py-2 font-mono text-xs">ext_in</td><td class="px-4 py-2 text-xs">ext_in[field,pdf,doc,docx]</td><td class="px-4 py-2 text-xs">Extensões permitidas</td></tr>
                                        <tr><td class="px-4 py-2 font-mono text-xs">max_dims</td><td class="px-4 py-2 text-xs">max_dims[field,1024,768]</td><td class="px-4 py-2 text-xs">Dimensões máximas (imagens)</td></tr>
                                        <tr><td class="px-4 py-2 font-mono text-xs">is_image</td><td class="px-4 py-2 text-xs">is_image[field]</td><td class="px-4 py-2 text-xs">Valida se é imagem</td></tr>
                                    </tbody>
                                </table>
                            </div>

                            <div class="rounded-xl bg-zinc-950 p-4 text-sm overflow-x-auto">
                                <pre class="language-php"><code>$rules = [
    'avatar' => [
        'uploaded[avatar]',
        'max_size[avatar,1024]',              // 1MB
        'is_image[avatar]',
        'mime_in[avatar,image/jpeg,image/png,image/webp]',
        'max_dims[avatar,800,800]'            // Max 800x800px
    ],
    'document' => [
        'uploaded[document]',
        'max_size[document,5120]',            // 5MB
        'ext_in[document,pdf,doc,docx,xls,xlsx]'
    ],
    'attachments.*' => [                       // Array de ficheiros
        'uploaded[attachments.*]',
        'max_size[attachments.*,2048]'
    ]
];</code></pre>
                            </div>
                        </section>

                        {{-- Navigation --}}
                        <div class="flex justify-between items-center pt-8 border-t border-zinc-200 dark:border-zinc-800">
                            <a href="{{ route('docs.show', 'notifications') }}" class="inline-flex items-center gap-2 text-sm text-zinc-600 dark:text-zinc-400 hover:text-primary">
                                <flux:icon.arrow-left class="size-4" />
                                Notificações
                            </a>
                            <a href="{{ route('docs.show', 'cache') }}" class="inline-flex items-center gap-2 text-sm font-medium text-primary hover:underline">
                                Cache System
                                <flux:icon.arrow-right class="size-4" />
                            </a>
                        </div>
                    </div>

                </div>
            </div>

            {{-- Right Sidebar: Table of Contents --}}
            @include('docs.partials.toc', ['sections' => [
                'arquitectura' => 'Arquitectura',
                'helpers' => 'Helpers Globais',
                'base-request' => 'BaseRequest',
                'rule-factory' => 'Rule Factory',
                'catalogo-rules' => 'Catálogo de Rules',
                'modificadores' => 'Modificadores',
                'conditionable' => 'Validação Condicional',
                'after-callbacks' => 'After (pós-validação)',
                'mensagens' => 'Mensagens Customizadas',
                'wildcards' => 'Dot Notation',
                'validator-api' => 'Validator API',
                'custom-rules' => 'Custom Rules',
                'file-rules' => 'Ficheiros',
            ]])
        </div>
    </div>
@endsection
