@extends('layouts.docs')

@section('title', 'Web Layer System - OG Framework')

@section('body')
    <div class="relative min-h-screen bg-transparent px-4 pb-12 pt-6 sm:px-6 lg:px-8">
        {{-- Floating shapes for styling --}}
        <div class="floating-blur cyan -left-10 -top-10 hidden lg:block"></div>
        <div class="floating-blur purple right-10 top-32 hidden lg:block"></div>

        <div class="mx-auto flex w-full max-w-7xl gap-8 bg-transparent">
            {{-- Left Sidebar: Navigation --}}
            @include('docs.partials.sidebar')

            {{-- Main Content --}}
            <div class="flex-1 min-w-0">
                <div class="relative overflow-hidden rounded-3xl border border-white/50 bg-white/85 p-8 shadow-xl shadow-primary/20 backdrop-blur dark:border-zinc-800 dark:bg-zinc-900/80">

                    {{-- Page Header --}}
                    <div class="mb-8">
                        <a href="{{ route('docs.index') }}" class="inline-flex items-center gap-1 text-sm text-primary hover:underline mb-4">
                            <flux:icon.arrow-left class="size-4" />
                            Voltar para Documentação
                        </a>
                        <p class="text-xs uppercase tracking-[0.3em] text-primary dark:text-blue-200">HTTP & API</p>
                        <h1 class="font-display text-4xl font-bold text-zinc-900 dark:text-white sm:text-5xl">Web Layer System</h1>
                        <p class="mt-3 text-base text-zinc-700 dark:text-zinc-200 max-w-3xl">
                            Arquitectura moderna e extensível para processamento de requests HTTP. Integra <strong>Http System</strong>, <strong>Middleware Pipeline</strong> e <strong>Routing System</strong> com mais de <strong>75 controllers</strong> implementados.
                        </p>
                    </div>

                    <div class="space-y-10">

                        {{-- Architecture SVG Diagram --}}
                        <section id="arquitectura" class="scroll-mt-28 space-y-4">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.cube-transparent class="size-6 text-sky-500" />
                                Pipeline de Request
                            </h2>

                            {{-- SVG: Request Flow --}}
                            <div class="rounded-xl border border-zinc-200 bg-white p-6 dark:border-zinc-800 dark:bg-zinc-900/50 overflow-x-auto">
                                <svg viewBox="0 0 850 160" class="w-full max-w-4xl mx-auto" xmlns="http://www.w3.org/2000/svg">
                                    {{-- Input --}}
                                    <g transform="translate(10,10)">
                                        <rect x="0" y="0" width="90" height="130" rx="8" fill="#fef3c7" stroke="#f59e0b" stroke-width="1.5"/>
                                        <text x="45" y="22" text-anchor="middle" class="text-[9px] font-semibold" fill="#92400e">Entrada</text>
                                        <rect x="8" y="35" width="74" height="22" rx="3" fill="#ffffff" stroke="#fcd34d"/>
                                        <text x="45" y="50" text-anchor="middle" class="text-[8px]" fill="#78350f">HTTP Request</text>
                                        <rect x="8" y="65" width="74" height="22" rx="3" fill="#ffffff" stroke="#fcd34d"/>
                                        <text x="45" y="80" text-anchor="middle" class="text-[8px]" fill="#78350f">API Request</text>
                                        <rect x="8" y="95" width="74" height="22" rx="3" fill="#ffffff" stroke="#fcd34d"/>
                                        <text x="45" y="110" text-anchor="middle" class="text-[8px]" fill="#78350f">AJAX Request</text>
                                    </g>

                                    {{-- Arrow --}}
                                    <path d="M110,75 L140,75" stroke="#94a3b8" stroke-width="2" marker-end="url(#arrowWL)"/>

                                    {{-- Pipeline --}}
                                    <g transform="translate(150,10)">
                                        <rect x="0" y="0" width="500" height="130" rx="10" fill="#e0f2fe" stroke="#0284c7" stroke-width="2"/>
                                        <text x="250" y="22" text-anchor="middle" class="text-sm font-bold" fill="#0369a1">Pipeline de Processamento</text>

                                        {{-- Steps --}}
                                        <rect x="15" y="40" width="80" height="35" rx="5" fill="#ffffff" stroke="#bae6fd"/>
                                        <text x="55" y="55" text-anchor="middle" class="text-[8px] font-mono" fill="#0c4a6e">capture()</text>
                                        <text x="55" y="68" text-anchor="middle" class="text-[7px]" fill="#64748b">Request</text>

                                        <path d="M100,58 L115,58" stroke="#0284c7" stroke-width="1.5" marker-end="url(#arrowWL2)"/>

                                        <rect x="120" y="40" width="80" height="35" rx="5" fill="#ffffff" stroke="#bae6fd"/>
                                        <text x="160" y="55" text-anchor="middle" class="text-[8px] font-mono" fill="#0c4a6e">Global MW</text>
                                        <text x="160" y="68" text-anchor="middle" class="text-[7px]" fill="#64748b">Middleware</text>

                                        <path d="M205,58 L220,58" stroke="#0284c7" stroke-width="1.5" marker-end="url(#arrowWL2)"/>

                                        <rect x="225" y="40" width="80" height="35" rx="5" fill="#ffffff" stroke="#bae6fd"/>
                                        <text x="265" y="55" text-anchor="middle" class="text-[8px] font-mono" fill="#0c4a6e">Matching</text>
                                        <text x="265" y="68" text-anchor="middle" class="text-[7px]" fill="#64748b">Router</text>

                                        <path d="M310,58 L325,58" stroke="#0284c7" stroke-width="1.5" marker-end="url(#arrowWL2)"/>

                                        <rect x="330" y="40" width="75" height="35" rx="5" fill="#ffffff" stroke="#bae6fd"/>
                                        <text x="368" y="55" text-anchor="middle" class="text-[8px] font-mono" fill="#0c4a6e">Route MW</text>
                                        <text x="368" y="68" text-anchor="middle" class="text-[7px]" fill="#64748b">Middleware</text>

                                        <path d="M410,58 L425,58" stroke="#0284c7" stroke-width="1.5" marker-end="url(#arrowWL2)"/>

                                        <rect x="430" y="40" width="55" height="35" rx="5" fill="#ffffff" stroke="#bae6fd"/>
                                        <text x="458" y="55" text-anchor="middle" class="text-[8px] font-mono" fill="#0c4a6e">Dispatch</text>
                                        <text x="458" y="68" text-anchor="middle" class="text-[7px]" fill="#64748b">Controller</text>

                                        {{-- Routes --}}
                                        <text x="250" y="100" text-anchor="middle" class="text-[8px] font-semibold" fill="#0369a1">RouterCollector • MiddlewareManager • RouteDispatcher</text>
                                        <text x="250" y="118" text-anchor="middle" class="text-[7px]" fill="#64748b">1095 lines • Pipeline Pattern • DI automática</text>
                                    </g>

                                    {{-- Arrow --}}
                                    <path d="M660,75 L690,75" stroke="#94a3b8" stroke-width="2" marker-end="url(#arrowWL)"/>

                                    {{-- Output --}}
                                    <g transform="translate(700,20)">
                                        <rect x="0" y="0" width="130" height="110" rx="8" fill="#f0fdf4" stroke="#22c55e" stroke-width="1.5"/>
                                        <text x="65" y="20" text-anchor="middle" class="text-[10px] font-semibold" fill="#166534">Saída</text>
                                        <rect x="12" y="35" width="106" height="28" rx="4" fill="#dcfce7" stroke="#86efac"/>
                                        <text x="65" y="53" text-anchor="middle" class="text-[9px] font-mono" fill="#166534">Response::json()</text>
                                        <rect x="12" y="70" width="106" height="28" rx="4" fill="#fef3c7" stroke="#fcd34d"/>
                                        <text x="65" y="88" text-anchor="middle" class="text-[9px] font-mono" fill="#92400e">AjaxServer</text>
                                    </g>

                                    <defs>
                                        <marker id="arrowWL" markerWidth="10" markerHeight="7" refX="9" refY="3.5" orient="auto">
                                            <polygon points="0 0, 10 3.5, 0 7" fill="#94a3b8"/>
                                        </marker>
                                        <marker id="arrowWL2" markerWidth="8" markerHeight="6" refX="7" refY="3" orient="auto">
                                            <polygon points="0 0, 8 3, 0 6" fill="#0284c7"/>
                                        </marker>
                                    </defs>
                                </svg>
                            </div>
                        </section>

                        {{-- Controllers --}}
                        <section id="controllers" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.rectangle-group class="size-6 text-violet-500" />
                                Controllers
                            </h2>
                            <p class="text-zinc-700 dark:text-zinc-300">
                                O sistema oferece duas classes base com diferentes níveis de integração:
                            </p>

                            <div class="space-y-4">
                                {{-- BaseController --}}
                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-bold text-zinc-900 dark:text-white mb-2 text-sm">BaseController — Integração Legacy</h3>
                                    <p class="text-xs text-zinc-600 dark:text-zinc-400 mb-3">Acesso directo aos objectos globais do sistema legacy:</p>
                                    <pre class="text-xs bg-zinc-950 p-3 rounded-lg text-zinc-300 overflow-x-auto"><code>class BaseController
{
    public Aplication $application;  // Instância da aplicação
    public $user;                    // Utilizador autenticado
    public $modules;                 // Sistema de módulos legacy
    public $database;                // Instância de database

    public function __construct()
    {
        global $a, $u, $m, $db;
        $this->application = $a;
        $this->user = $u;
        $this->modules = $m;
        $this->database = $db;
    }
}

// Uso típico
class CustomerController extends BaseController
{
    public function index(): Response
    {
        $customers = $this->modules['entidades']->listaClientesAPI(limit: 250);
        return response()->json($customers);
    }
}</code></pre>
                                </div>

                                {{-- Controller with Policies --}}
                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-bold text-zinc-900 dark:text-white mb-2 text-sm">Controller — Moderno com Policies</h3>
                                    <p class="text-xs text-zinc-600 dark:text-zinc-400 mb-3">Estende <code>BaseClass</code> e adiciona suporte a authorization policies:</p>
                                    <pre class="text-xs bg-zinc-950 p-3 rounded-lg text-zinc-300 overflow-x-auto"><code>class Controller extends BaseClass
{
    protected array $policies = [];
    protected string $policy;

    public function authorize(string $method, string $modelClassName, ?array $params = null): void
    {
        // 1. Usa $this->policy se definida
        // 2. Procura no mapa $this->policies
        // 3. Inferência automática: Controller → Policy
        //    UserController → UserPolicy
        //    Og\Crm\Controllers\AppointmentController → Og\Crm\Policies\AppointmentPolicy
    }
}</code></pre>
                                </div>

                                {{-- Controller Example --}}
                                <div class="rounded-xl border border-violet-200 bg-violet-50 p-4 dark:border-violet-900/50 dark:bg-violet-900/20">
                                    <h3 class="font-semibold text-violet-900 dark:text-violet-200 mb-2 text-sm">Exemplo Real: AppointmentController</h3>
                                    <pre class="text-xs bg-white dark:bg-black/20 p-3 rounded overflow-x-auto"><code>class AppointmentController extends Controller
{
    public function index(AppointmentsIndexRequest $request): Response
    {
        // 1. Autorização via Policy
        $this->authorize('viewAny', static::class);
        
        // 2. Dados validados
        $validData = $request->validated();
        
        // 3. Delegar para Action
        $appointments = (new GetAppointments())($validData);
        
        return response()->json($appointments);
    }

    public function create(AppointmentsCreateRequest $request, CreateAppointment $action): Response
    {
        $this->authorize('create', static::class);
        
        [$appointment, $statusCode] = $action->execute($request->validated());
        
        return response()->json($appointment, $statusCode);
    }
}</code></pre>
                                    <p class="text-xs text-violet-700 dark:text-violet-300 mt-2">
                                        <strong>Padrões:</strong> Request Classes • Action Classes • Policy Authorization • Dependency Injection
                                    </p>
                                </div>
                            </div>
                        </section>

                        {{-- Request Class --}}
                        <section id="request" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.inbox-arrow-down class="size-6 text-blue-500" />
                                Request Class
                            </h2>
                            <p class="text-zinc-700 dark:text-zinc-300">
                                Estende <code>Symfony\Component\HttpFoundation\Request</code> com API rica para HTTP:
                            </p>

                            <div class="space-y-4">
                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-bold text-zinc-900 dark:text-white mb-2 text-sm">Request::capture() — Factory Method</h3>
                                    <pre class="text-xs bg-zinc-950 p-3 rounded-lg text-zinc-300 overflow-x-auto"><code>// Cria Request a partir de superglobals PHP
$request = Request::capture();

// Processa JSON automaticamente
if (str_contains($headers->get('CONTENT_TYPE'), 'application/json')) {
    $data = json_decode($request->getContent(), true);
    $request->request->replace($data);
}

// Suporta method override (_method)
// POST com _method=PUT → $request->method() retorna 'PUT'</code></pre>
                                </div>

                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-bold text-zinc-900 dark:text-white mb-2 text-sm">Input Handling com Dot Notation</h3>
                                    <pre class="text-xs bg-zinc-950 p-3 rounded-lg text-zinc-300 overflow-x-auto"><code>// Obter apenas campos específicos
$data = $request->only(['name', 'email', 'user.profile.bio']);

// Excluir campos
$data = $request->except(['password', 'password_confirmation']);

// Verificações
$request->filled('name');    // Campo preenchido?
$request->missing('token');  // Campo ausente?
$request->has(['email']);    // Campos existem?
$request->ajax();            // É AJAX?
$request->expectsJson();     // Espera JSON?

// Merge de dados
$request->merge(['active' => true]);
$request->mergeIfMissing(['role' => 'user']);</code></pre>
                                </div>

                                <div class="rounded-xl border border-blue-200 bg-blue-50 p-4 dark:border-blue-900/50 dark:bg-blue-900/20">
                                    <h3 class="font-semibold text-blue-900 dark:text-blue-200 mb-2 text-sm">Input Trait — Helpers Adicionais</h3>
                                    <pre class="text-xs bg-white dark:bg-black/20 p-2 rounded overflow-x-auto"><code>$request->header('X-API-KEY');           // Header específico
$request->bearerToken();                  // Extract Bearer token
$request->cookie('session_id');           // Cookies
$request->user();                         // Utilizador autenticado
$request->all();                          // Inputs + ficheiros
$request->remove('temp_field');           // Remover campo</code></pre>
                                </div>
                            </div>
                        </section>

                        {{-- Response Class --}}
                        <section id="response" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.arrow-up-tray class="size-6 text-green-500" />
                                Response Class
                            </h2>

                            <div class="rounded-xl bg-zinc-950 p-4 text-sm overflow-x-auto">
                                <pre class="language-php"><code>// JSON Response
return response()->json(['users' => $users], 200);

// XML Response
return response()->xml($xmlContent);

// Text Response
return response()->text('Plain text content');

// Redirect com flash data
return response()->redirect('/dashboard')->with('message', 'Welcome!');

// No Content (204)
return response()->noContent();

// Definir headers
return response()->header('X-Custom', 'value')->json($data);

// Integração Legacy
response()->sendAjax($content);      // Via AjaxServer
response()->ogJson($data, 1000);     // Formato OfficeGest</code></pre>
                            </div>
                        </section>

                        {{-- Middleware --}}
                        <section id="middleware" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.funnel class="size-6 text-amber-500" />
                                Middleware System
                            </h2>
                            <p class="text-zinc-700 dark:text-zinc-300">
                                O <code>MiddlewareManager</code> implementa o <strong>Pipeline Pattern</strong> para processamento de middlewares:
                            </p>

                            <div class="space-y-4">
                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-bold text-zinc-900 dark:text-white mb-2 text-sm">MiddlewareInterface</h3>
                                    <pre class="text-xs bg-zinc-950 p-3 rounded-lg text-zinc-300 overflow-x-auto"><code>interface MiddlewareInterface
{
    public function handle(Request $request, Closure $next, mixed $params = null): mixed;
}</code></pre>
                                </div>

                                <div class="rounded-xl border border-amber-200 bg-amber-50 p-4 dark:border-amber-900/50 dark:bg-amber-900/20">
                                    <h3 class="font-semibold text-amber-900 dark:text-amber-200 mb-2 text-sm">Criar Middleware Personalizado</h3>
                                    <pre class="text-xs bg-white dark:bg-black/20 p-2 rounded overflow-x-auto"><code>class ValidateApiKeyMiddleware implements MiddlewareInterface
{
    public function handle(Request $request, Closure $next, mixed $params = null): mixed
    {
        $apiKey = $request->header('X-API-KEY');
        
        if (!$this->isValidApiKey($apiKey)) {
            return response()->json(['error' => 'Invalid API key'], 401);
        }

        // Adicionar info ao request
        $request->merge(['api_client' => $this->getClientByKey($apiKey)]);

        return $next($request);
    }
}</code></pre>
                                </div>

                                <div class="rounded-xl border border-zinc-200 bg-white p-4 dark:border-zinc-800 dark:bg-zinc-900/50">
                                    <h3 class="font-bold text-zinc-900 dark:text-white mb-2 text-sm">Middleware com Parâmetros</h3>
                                    <pre class="text-xs bg-zinc-950 p-3 rounded-lg text-zinc-300 overflow-x-auto"><code>class ThrottleMiddleware implements MiddlewareInterface
{
    public function handle(Request $request, Closure $next, mixed $params = null): mixed
    {
        // Params: ['60', '1'] de 'throttle:60,1'
        $maxAttempts = (int) ($params[0] ?? 60);
        $decayMinutes = (int) ($params[1] ?? 1);
        
        if ($this->tooManyAttempts($key, $maxAttempts)) {
            return response()->json(['error' => 'Too many requests'], 429);
        }
        
        return $next($request);
    }
}

// Uso
Route::middleware(['throttle:100,1'])->group(function() {
    Route::get('/api/heavy', 'ApiController@heavy');
});</code></pre>
                                </div>
                            </div>
                        </section>

                        {{-- Action Classes --}}
                        <section id="actions" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.play-circle class="size-6 text-rose-500" />
                                Action Classes
                            </h2>
                            <p class="text-zinc-700 dark:text-zinc-300">
                                Padrão para isolar lógica de negócio em classes reutilizáveis e testáveis:
                            </p>

                            <div class="rounded-xl bg-zinc-950 p-4 text-sm overflow-x-auto">
                                <pre class="language-php"><code>class CreateAppointment
{
    public function execute(array $data): array
    {
        // Validação de regras de negócio
        $this->validateBusinessRules($data);
        
        // Criar agendamento
        $appointment = Appointment::create([
            'customer_id' => $data['customer_id'],
            'scheduled_at' => Carbon::parse($data['scheduled_at']),
            'duration' => $data['duration'] ?? 60,
            'notes' => $data['notes'] ?? null,
        ]);
        
        // Enviar notificação
        $appointment->customer->notify(new AppointmentCreated($appointment));
        
        return [$appointment->toArray(), Response::HTTP_CREATED];
    }
}

// Uso no Controller (injectado automaticamente via DI)
public function create(AppointmentsCreateRequest $request, CreateAppointment $action): Response
{
    [$appointment, $statusCode] = $action->execute($request->validated());
    return response()->json($appointment, $statusCode);
}</code></pre>
                            </div>

                            <div class="rounded-xl border border-rose-200 bg-rose-50 p-4 dark:border-rose-900/50 dark:bg-rose-900/20 text-sm">
                                <p class="font-semibold text-rose-900 dark:text-rose-200 mb-1">Benefícios:</p>
                                <ul class="text-rose-700 dark:text-rose-300 list-disc list-inside space-y-1 text-xs">
                                    <li><strong>Single Responsibility</strong> — Controller só orquestra, Action executa</li>
                                    <li><strong>Reutilização</strong> — Mesma action em API, CLI, Jobs</li>
                                    <li><strong>Testabilidade</strong> — Actions testáveis sem HTTP</li>
                                    <li><strong>DI Automática</strong> — RouteDispatcher resolve dependências</li>
                                </ul>
                            </div>
                        </section>

                        {{-- OpenAPI Annotations --}}
                        <section id="openapi" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.document-text class="size-6 text-emerald-500" />
                                OpenAPI/Swagger Annotations
                            </h2>
                            <p class="text-zinc-700 dark:text-zinc-300">
                                Controllers podem incluir anotações OpenAPI para documentação automática da API:
                            </p>

                            <div class="rounded-xl bg-zinc-950 p-4 text-sm overflow-x-auto">
                                <pre class="language-php"><code>/**
 * @OA\Tag(name="Customers")
 */
class CustomerController extends Controller
{
    /**
     * @OA\Get(
     *     path="/entities/customers",
     *     summary="List all customers",
     *     tags={"Customers"},
     *     @OA\Parameter(name="name", in="query", @OA\Schema(type="string")),
     *     @OA\Parameter(name="taxId", in="query", @OA\Schema(type="string")),
     *     @OA\Response(response=200, description="OK", 
     *         @OA\JsonContent(type="array", @OA\Items(ref="#/components/schemas/CustomerOutputDTO")))
     * )
     */
    public function index(CustomerIndexRequest $request): Response
    {
        // ...
    }

    /**
     * @OA\Post(
     *     path="/entities/customers",
     *     summary="Create a new customer",
     *     tags={"Customers"},
     *     @OA\RequestBody(ref="#/components/requestBodies/CreateCustomerRequest"),
     *     @OA\Response(response=201, description="Created"),
     *     @OA\Response(response=422, description="Unprocessable Entity")
     * )
     */
    public function create(CustomerCreateRequest $request): Response
    {
        // ...
    }
}</code></pre>
                            </div>
                        </section>

                        {{-- RelationshipLoader --}}
                        <section id="relationships" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.link class="size-6 text-indigo-500" />
                                RelationshipLoader
                            </h2>
                            <p class="text-zinc-700 dark:text-zinc-300">
                                Utilitário para carregar relações dinamicamente em arrays de dados:
                            </p>

                            <div class="rounded-xl bg-zinc-950 p-4 text-sm overflow-x-auto">
                                <pre class="language-php"><code>use Og\Modules\Common\Database\Relations\RelationshipLoader;

// Carregar relações para array de clientes
$customers = (new RelationshipLoader())->load(
    $customers,                // Array de dados
    ['country', 'contacts'],  // Relações a carregar
    EntityRelation::class      // Classe que define as relações
);

// Uso típico em controller
public function show(int $id): Response
{
    $customer = $this->modules['entidades']->loadCliente($id);
    
    // Carrega relação 'country' automaticamente
    $customer = (new RelationshipLoader())->load(
        $customer, 
        ['country'], 
        EntityRelation::class
    );
    
    return response()->json(EntitiesCustomerOutputDTO::make($customer));
}</code></pre>
                            </div>
                        </section>

                        {{-- UploadedFile --}}
                        <section id="uploads" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.cloud-arrow-up class="size-6 text-sky-500" />
                                UploadedFile
                            </h2>
                            <p class="text-zinc-700 dark:text-zinc-300">
                                Classe para gestão de ficheiros enviados via upload:
                            </p>

                            <div class="rounded-xl bg-zinc-950 p-4 text-sm overflow-x-auto">
                                <pre class="language-php"><code>$file = $request->file('avatar');

// Informações do ficheiro
$file->getClientOriginalName();  // 'photo.jpg'
$file->getClientMimeType();      // 'image/jpeg'
$file->getSize();                // 245632 bytes
$file->getExtension();           // 'jpg'

// Mover para destino
$file->move('/uploads/avatars', 'user_123.jpg');

// Store com nome automático
$path = $file->store('avatars');  // 'avatars/abc123.jpg'

// Validação (ver página Validação)
$rules = [
    'avatar' => ['uploaded', 'is_image', 'max_size[2048]'],
];</code></pre>
                            </div>
                        </section>

                        {{-- Routing Reference --}}
                        <section id="routing" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.map class="size-6 text-amber-500" />
                                Routing System
                            </h2>
                            <p class="text-zinc-700 dark:text-zinc-300">
                                O sistema de rotas é documentado em detalhe na sua própria página:
                            </p>
                            <a href="{{ route('docs.show', 'routing') }}" class="inline-flex items-center gap-2 px-4 py-2 rounded-lg bg-amber-100 dark:bg-amber-900/30 text-amber-800 dark:text-amber-200 font-medium text-sm hover:bg-amber-200 dark:hover:bg-amber-900/50 transition-colors">
                                <flux:icon.arrow-right class="size-4" />
                                Ver Documentação de Routing
                            </a>
                        </section>

                        {{-- RouteDispatcher --}}
                        <section id="dispatcher" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.play class="size-6 text-cyan-500" />
                                RouteDispatcher
                            </h2>
                            <p class="text-zinc-700 dark:text-zinc-300">
                                Executa handlers com <strong>Dependency Injection automática</strong>:
                            </p>

                            <div class="rounded-xl bg-zinc-950 p-4 text-sm overflow-x-auto">
                                <pre class="language-php"><code>readonly class RouteDispatcher
{
    public function dispatch(callable|string $handler, array $routeParams = []): mixed
    {
        return match (true) {
            $handler instanceof Closure => $this->dispatchClosure($handler, $routeParams),
            $this->isValidHandlerString($handler) => $this->dispatchController($handler, $routeParams),
            default => throw new RuntimeException('Invalid route handler.')
        };
    }

    // Resolução automática de parâmetros:
    // - Route params: {id}, {slug} → conversão automática de tipos (int, bool, float)
    // - Request injection: Request $request
    // - BaseRequest validation: AppointmentsCreateRequest $request
    // - Dependency injection: CreateAppointment $action
}</code></pre>
                            </div>
                        </section>

                        {{-- Facades --}}
                        <section id="facades" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.bolt class="size-6 text-purple-500" />
                                Facades
                            </h2>

                            <div class="rounded-xl bg-zinc-950 p-4 text-sm overflow-x-auto">
                                <pre class="language-php"><code>use Og\Modules\Common\Facades\Request;
use Og\Modules\Common\Facades\Response;
use Og\Modules\Common\Facades\Route;

// Request Facade
$data = Request::all();
$name = Request::get('name', 'default');
$token = Request::bearerToken();
$user = Request::user();

// Response Facade
return Response::json(['success' => true]);
return Response::redirect('/dashboard')->with('message', 'OK');

// Route Facade
Route::get('/users', 'UserController@index')->name('users.index');
Route::middleware(['auth'])->group(function() { /* ... */ });</code></pre>
                            </div>
                        </section>

                        {{-- Controllers Table --}}
                        <section id="controllers-list" class="scroll-mt-28 space-y-4 border-t border-zinc-200 dark:border-zinc-800 pt-8">
                            <h2 class="text-2xl font-semibold text-zinc-900 dark:text-white flex items-center gap-2">
                                <flux:icon.squares-2x2 class="size-6 text-indigo-500" />
                                Controllers no Sistema (75+)
                            </h2>

                            <div class="overflow-x-auto">
                                <table class="min-w-full divide-y divide-zinc-200 dark:divide-zinc-700 text-sm">
                                    <thead class="bg-zinc-50 dark:bg-zinc-800">
                                        <tr>
                                            <th class="px-4 py-2 text-left text-xs font-medium text-zinc-500 uppercase">Domínio</th>
                                            <th class="px-4 py-2 text-left text-xs font-medium text-zinc-500 uppercase">Qty</th>
                                            <th class="px-4 py-2 text-left text-xs font-medium text-zinc-500 uppercase">Exemplos</th>
                                        </tr>
                                    </thead>
                                    <tbody class="divide-y divide-zinc-200 dark:divide-zinc-700">
                                        <tr><td class="px-4 py-2 font-medium">Ams</td><td class="px-4 py-2">18</td><td class="px-4 py-2 text-xs">BoardingGateController, TransportServicesController</td></tr>
                                        <tr><td class="px-4 py-2 font-medium">Entidade</td><td class="px-4 py-2">9</td><td class="px-4 py-2 text-xs">CustomerController, EmployeeController, SupplierController</td></tr>
                                        <tr><td class="px-4 py-2 font-medium">Crm</td><td class="px-4 py-2">5</td><td class="px-4 py-2 text-xs">AppointmentController, ParameterController, AbcController</td></tr>
                                        <tr><td class="px-4 py-2 font-medium">ChatFlow</td><td class="px-4 py-2">3</td><td class="px-4 py-2 text-xs">ChatFlowController, WhatsappController</td></tr>
                                        <tr><td class="px-4 py-2 font-medium">Tickets</td><td class="px-4 py-2">3</td><td class="px-4 py-2 text-xs">TicketController, TicketMessageController</td></tr>
                                        <tr><td class="px-4 py-2 font-medium">Outros</td><td class="px-4 py-2">37+</td><td class="px-4 py-2 text-xs">AI, Auth, Developer, SAFT, Portal...</td></tr>
                                    </tbody>
                                </table>
                            </div>
                        </section>

                        {{-- Navigation --}}
                        <div class="flex justify-between items-center pt-8 border-t border-zinc-200 dark:border-zinc-800">
                            <a href="{{ route('docs.show', 'routing') }}" class="inline-flex items-center gap-2 text-sm text-zinc-600 dark:text-zinc-400 hover:text-primary">
                                <flux:icon.arrow-left class="size-4" />
                                Routing
                            </a>
                            <a href="{{ route('docs.show', 'helpers') }}" class="inline-flex items-center gap-2 text-sm font-medium text-primary hover:underline">
                                Helpers
                                <flux:icon.arrow-right class="size-4" />
                            </a>
                        </div>
                    </div>

                </div>
            </div>

            {{-- Right Sidebar: Table of Contents --}}
            @include('docs.partials.toc', ['sections' => [
                'arquitectura' => 'Pipeline de Request',
                'controllers' => 'Controllers',
                'request' => 'Request Class',
                'response' => 'Response Class',
                'middleware' => 'Middleware System',
                'actions' => 'Action Classes',
                'openapi' => 'OpenAPI Annotations',
                'relationships' => 'RelationshipLoader',
                'uploads' => 'UploadedFile',
                'routing' => 'Routing System',
                'dispatcher' => 'RouteDispatcher',
                'facades' => 'Facades',
                'controllers-list' => 'Controllers (75+)',
            ]])
        </div>
    </div>
@endsection
